<?php

namespace Drupal\lox_backup\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller for LOX Backup actions.
 */
class LoxBackupController extends ControllerBase {

  /**
   * The LOX API service.
   *
   * @var \Drupal\lox_backup\Service\LoxApi
   */
  protected $loxApi;

  /**
   * The LOX Backup Manager service.
   *
   * @var \Drupal\lox_backup\Service\LoxBackupManager
   */
  protected $backupManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->loxApi = $container->get('lox_backup.api');
    $instance->backupManager = $container->get('lox_backup.manager');
    return $instance;
  }

  /**
   * Test API connection.
   */
  public function testConnection() {
    $result = $this->loxApi->testConnection();

    if ($result['success']) {
      $this->messenger()->addStatus(
        $this->t('Connection successful! Tenant: @tenant', ['@tenant' => $result['data']['name']])
      );
    }
    else {
      $this->messenger()->addError(
        $this->t('Connection failed: @error', ['@error' => $result['error']])
      );
    }

    return new RedirectResponse(Url::fromRoute('lox_backup.settings')->toString());
  }

  /**
   * Run backup manually.
   */
  public function runBackup() {
    // Increase execution time.
    set_time_limit(3600);

    $result = $this->backupManager->runBackup();

    if ($result['success']) {
      $this->messenger()->addStatus(
        $this->t('Backup completed successfully! UUID: @uuid, Size: @size', [
          '@uuid' => $result['uuid'],
          '@size' => format_size($result['size_bytes']),
        ])
      );
    }
    else {
      $this->messenger()->addError(
        $this->t('Backup failed: @error', ['@error' => $result['error']])
      );
    }

    return new RedirectResponse(Url::fromRoute('lox_backup.settings')->toString());
  }

  /**
   * Run backup for a specific schedule.
   *
   * @param string $lox_backup_schedule
   *   The schedule ID.
   */
  public function runSchedule(string $lox_backup_schedule) {
    $schedule = $this->entityTypeManager()
      ->getStorage('lox_backup_schedule')
      ->load($lox_backup_schedule);

    if (!$schedule) {
      $this->messenger()->addError($this->t('Schedule not found.'));
      return new RedirectResponse(Url::fromRoute('entity.lox_backup_schedule.collection')->toString());
    }

    // Increase execution time.
    set_time_limit(3600);

    $result = $this->backupManager->runBackup([
      'component' => $schedule->getComponent(),
    ]);

    if ($result['success']) {
      $this->messenger()->addStatus(
        $this->t('Backup for %schedule completed! UUID: @uuid', [
          '%schedule' => $schedule->label(),
          '@uuid' => $result['uuid'],
        ])
      );
    }
    else {
      $this->messenger()->addError(
        $this->t('Backup failed: @error', ['@error' => $result['error']])
      );
    }

    return new RedirectResponse(Url::fromRoute('entity.lox_backup_schedule.collection')->toString());
  }

  /**
   * Status page.
   */
  public function status() {
    $config = $this->config('lox_backup.settings');
    $quota = $this->loxApi->getQuota();

    $build = [
      '#theme' => 'lox_backup_status',
      '#last_backup' => $config->get('last_backup'),
      '#last_status' => $config->get('last_backup_status'),
      '#last_uuid' => $config->get('last_backup_uuid'),
      '#quota' => $quota['success'] ? $quota['data'] : NULL,
    ];

    return $build;
  }

  /**
   * Request restore for a backup.
   *
   * @param string $uuid
   *   The backup UUID.
   */
  public function restore(string $uuid) {
    $result = $this->loxApi->requestRestore($uuid);

    if ($result['success']) {
      $this->messenger()->addStatus(
        $this->t('Restore requested for backup @uuid. You will be notified when it is ready for download.', [
          '@uuid' => substr($uuid, 0, 8) . '...',
        ])
      );
    }
    else {
      $this->messenger()->addError(
        $this->t('Failed to request restore: @error', ['@error' => $result['error']])
      );
    }

    return new RedirectResponse(Url::fromRoute('lox_backup.backups')->toString());
  }

  /**
   * Download a backup.
   *
   * @param string $uuid
   *   The backup UUID.
   */
  public function download(string $uuid) {
    $result = $this->loxApi->getDownloadUrl($uuid);

    if ($result['success'] && !empty($result['data']['url'])) {
      // Redirect to the download URL.
      return new RedirectResponse($result['data']['url']);
    }
    elseif ($result['success'] && isset($result['data']['status']) && $result['data']['status'] === 'pending') {
      $this->messenger()->addWarning(
        $this->t('Backup is being prepared for download. Please try again in a few minutes.')
      );
    }
    else {
      $this->messenger()->addError(
        $this->t('Failed to get download URL: @error', [
          '@error' => $result['error'] ?? 'Backup not ready for download',
        ])
      );
    }

    return new RedirectResponse(Url::fromRoute('lox_backup.backups')->toString());
  }

  /**
   * Dashboard page with tabs.
   */
  public function dashboard() {
    $config = $this->config('lox_backup.settings');
    $quota = $this->loxApi->getQuota();

    // Get recent backups.
    $backups_result = $this->loxApi->listBackups(['limit' => 5]);
    $recent_backups = $backups_result['success'] ? ($backups_result['data']['backups'] ?? $backups_result['data'] ?? []) : [];

    // Get schedules.
    $schedules = $this->entityTypeManager()
      ->getStorage('lox_backup_schedule')
      ->loadMultiple();

    $build = [
      '#theme' => 'lox_backup_dashboard',
      '#last_backup' => $config->get('last_backup'),
      '#last_status' => $config->get('last_backup_status'),
      '#quota' => $quota['success'] ? $quota['data'] : NULL,
      '#recent_backups' => $recent_backups,
      '#schedules' => $schedules,
      '#attached' => [
        'library' => ['lox_backup/admin'],
      ],
    ];

    return $build;
  }

  /**
   * AJAX endpoint to list backups.
   */
  public function ajaxListBackups() {
    $page = \Drupal::request()->query->get('page', 0);
    $limit = 20;

    $result = $this->loxApi->listBackups([
      'limit' => $limit,
      'offset' => $page * $limit,
    ]);

    return new JsonResponse($result);
  }

  /**
   * AJAX endpoint to list cloud profiles.
   */
  public function ajaxListProfiles() {
    $result = $this->loxApi->getProfiles();

    return new JsonResponse($result);
  }

  /**
   * AJAX endpoint to get profile details with versions.
   *
   * @param string $uuid
   *   Profile UUID.
   */
  public function ajaxGetProfile(string $uuid) {
    $result = $this->loxApi->getProfile($uuid);

    return new JsonResponse($result);
  }

  /**
   * AJAX endpoint to get profile versions (backups).
   *
   * @param string $uuid
   *   Profile UUID.
   */
  public function ajaxGetProfileVersions(string $uuid) {
    $page = \Drupal::request()->query->get('page', 1);
    $per_page = \Drupal::request()->query->get('per_page', 20);

    $result = $this->loxApi->getProfileVersions($uuid, (int) $page, (int) $per_page);

    return new JsonResponse($result);
  }

  /**
   * AJAX endpoint to run profile backup.
   *
   * @param string $uuid
   *   Profile UUID.
   */
  public function ajaxRunProfileBackup(string $uuid) {
    $request = \Drupal::request();
    $name = $request->query->get('name', 'Profile Backup');
    $components = $request->query->get('components', 'database');

    // Parse components.
    $component_list = array_filter(array_map('trim', explode(',', $components)));
    if (empty($component_list)) {
      $component_list = ['database'];
    }

    $result = $this->backupManager->runProfileBackup($uuid, $name, $component_list);

    return new JsonResponse($result);
  }

}
