<?php

namespace Drupal\lox_backup\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure LOX Backup settings.
 */
class LoxBackupSettingsForm extends ConfigFormBase {

  /**
   * The LOX API service.
   *
   * @var \Drupal\lox_backup\Service\LoxApi
   */
  protected $loxApi;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->loxApi = $container->get('lox_backup.api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'lox_backup_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['lox_backup.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('lox_backup.settings');

    // Status card.
    $form['status'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Backup Status'),
      '#attributes' => ['class' => ['lox-status-card']],
    ];

    $last_backup = $config->get('last_backup');
    $last_status = $config->get('last_backup_status');

    $form['status']['info'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['lox-status-info']],
    ];

    $form['status']['info']['last_backup'] = [
      '#type' => 'item',
      '#title' => $this->t('Last Backup'),
      '#markup' => $last_backup ? $this->t('@time ago', ['@time' => \Drupal::service('date.formatter')->formatTimeDiffSince(strtotime($last_backup))]) : $this->t('Never'),
    ];

    $form['status']['info']['last_status'] = [
      '#type' => 'item',
      '#title' => $this->t('Status'),
      '#markup' => $last_status ? ucfirst($last_status) : $this->t('N/A'),
    ];

    // Get quota.
    $quota = $this->loxApi->getQuota();
    $form['status']['info']['storage'] = [
      '#type' => 'item',
      '#title' => $this->t('Storage Used'),
      '#markup' => $quota['success']
        ? format_size($quota['data']['used_bytes']) . ' / ' . ($quota['data']['unlimited'] ? $this->t('Unlimited') : format_size($quota['data']['quota_bytes']))
        : $this->t('Not connected'),
    ];

    // Actions.
    $form['status']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['lox-actions']],
    ];

    $form['status']['actions']['test'] = [
      '#type' => 'link',
      '#title' => $this->t('Test Connection'),
      '#url' => Url::fromRoute('lox_backup.test_connection'),
      '#attributes' => ['class' => ['button']],
    ];

    $form['status']['actions']['run'] = [
      '#type' => 'link',
      '#title' => $this->t('Run Backup Now'),
      '#url' => Url::fromRoute('lox_backup.run_backup'),
      '#attributes' => ['class' => ['button', 'button--primary']],
    ];

    // API Settings.
    $form['api'] = [
      '#type' => 'details',
      '#title' => $this->t('API Configuration'),
      '#open' => TRUE,
    ];

    $form['api']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $config->get('api_key'),
      '#description' => $this->t('Your LOX API key. Get it from <a href="https://backlox.com/settings/api" target="_blank">backlox.com</a>'),
      '#required' => TRUE,
    ];

    $form['api']['api_url'] = [
      '#type' => 'url',
      '#title' => $this->t('API URL'),
      '#default_value' => $config->get('api_url') ?: 'https://backlox.com/api',
      '#description' => $this->t('For self-hosted LOX instances. Leave default for cloud service.'),
    ];

    // Backup Settings.
    $form['backup'] = [
      '#type' => 'details',
      '#title' => $this->t('Backup Configuration'),
      '#open' => TRUE,
    ];

    $form['backup']['backup_database'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Backup Database'),
      '#default_value' => $config->get('backup_database'),
      '#description' => $this->t('Include the database in backups.'),
    ];

    $form['backup']['backup_files'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Backup Public Files'),
      '#default_value' => $config->get('backup_files'),
      '#description' => $this->t('Include public files (images, documents, etc.).'),
    ];

    $form['backup']['backup_private'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Backup Private Files'),
      '#default_value' => $config->get('backup_private'),
      '#description' => $this->t('Include private files if configured.'),
    ];

    $form['backup']['backup_config'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Backup Configuration'),
      '#default_value' => $config->get('backup_config'),
      '#description' => $this->t('Include site configuration export.'),
    ];

    $form['backup']['schedule'] = [
      '#type' => 'select',
      '#title' => $this->t('Backup Schedule'),
      '#default_value' => $config->get('schedule') ?: 'daily',
      '#options' => [
        'hourly' => $this->t('Hourly'),
        'daily' => $this->t('Daily'),
        'weekly' => $this->t('Weekly'),
        'monthly' => $this->t('Monthly'),
        'disabled' => $this->t('Disabled'),
      ],
      '#description' => $this->t('How often to run automated backups. Requires cron to be configured.'),
    ];

    $form['backup']['retention_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Retention Days'),
      '#default_value' => $config->get('retention_days') ?: 30,
      '#min' => 1,
      '#max' => 3650,
      '#description' => $this->t('Number of days to keep backups in cold storage.'),
    ];

    $form['backup']['immutable_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Immutable Period (days)'),
      '#default_value' => $config->get('immutable_days'),
      '#min' => 0,
      '#max' => 3650,
      '#description' => $this->t('Days during which backups cannot be deleted (anti-ransomware protection). Leave empty to use retention period, or 0 to allow immediate deletion.'),
    ];

    $form['backup']['tags'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Backup Tags'),
      '#default_value' => $config->get('tags') ?: 'drupal',
      '#description' => $this->t('Comma-separated tags for organizing backups.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('lox_backup.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('api_url', $form_state->getValue('api_url'))
      ->set('backup_database', $form_state->getValue('backup_database'))
      ->set('backup_files', $form_state->getValue('backup_files'))
      ->set('backup_private', $form_state->getValue('backup_private'))
      ->set('backup_config', $form_state->getValue('backup_config'))
      ->set('schedule', $form_state->getValue('schedule'))
      ->set('retention_days', $form_state->getValue('retention_days'))
      ->set('immutable_days', $form_state->getValue('immutable_days'))
      ->set('tags', $form_state->getValue('tags'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
