<?php

namespace Drupal\lox_backup\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\lox_backup\Service\LoxApi;
use Drupal\lox_backup\Service\LoxBackupManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for managing cloud backup profiles.
 */
class LoxCloudProfilesForm extends FormBase {

  /**
   * The LOX API client.
   *
   * @var \Drupal\lox_backup\Service\LoxApi
   */
  protected $api;

  /**
   * The backup manager.
   *
   * @var \Drupal\lox_backup\Service\LoxBackupManager
   */
  protected $backupManager;

  /**
   * Constructs a LoxCloudProfilesForm object.
   */
  public function __construct(LoxApi $api, LoxBackupManager $backup_manager) {
    $this->api = $api;
    $this->backupManager = $backup_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('lox_backup.api'),
      $container->get('lox_backup.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'lox_cloud_profiles_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'lox_backup/admin';

    // Header.
    $form['header'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['lox-header']],
    ];

    $form['header']['title'] = [
      '#markup' => '<h1>' . $this->t('Cloud Profiles') . '</h1>',
    ];

    $form['header']['description'] = [
      '#markup' => '<p>' . $this->t('Manage backup profiles synced with LOX server. Each profile groups related backup versions.') . '</p>',
    ];

    // Fetch profiles from server.
    $result = $this->api->getProfiles();
    $profiles = [];

    if ($result['success'] && isset($result['data'])) {
      $data = $result['data'];
      $profiles = $data['profiles'] ?? $data;
    }

    if (empty($profiles)) {
      $form['empty'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['lox-empty-state']],
      ];
      $form['empty']['icon'] = [
        '#markup' => '<div class="lox-empty-icon">☁️</div>',
      ];
      $form['empty']['message'] = [
        '#markup' => '<h3>' . $this->t('No cloud profiles yet') . '</h3>',
      ];
      $form['empty']['help'] = [
        '#markup' => '<p>' . $this->t('Profiles are automatically created when you run backups. Each profile groups related backup versions.') . '</p>',
      ];
    }
    else {
      // Profiles table.
      $form['profiles'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Profile'),
          $this->t('Versions'),
          $this->t('Total Size'),
          $this->t('Last Backup'),
          $this->t('Actions'),
        ],
        '#empty' => $this->t('No profiles found.'),
        '#attributes' => ['class' => ['lox-profiles-table']],
      ];

      foreach ($profiles as $index => $profile) {
        $uuid = $profile['uuid'] ?? '';
        $name = $profile['name'] ?? 'Unknown';
        $backup_count = $profile['backup_count'] ?? 0;
        $total_size = $profile['total_size_bytes'] ?? 0;
        $last_backup = $profile['last_backup_at'] ?? NULL;

        $form['profiles'][$index]['name'] = [
          '#type' => 'container',
          '#attributes' => ['class' => ['lox-profile-name']],
          'name' => [
            '#markup' => '<strong>' . htmlspecialchars($name) . '</strong>',
          ],
          'uuid' => [
            '#markup' => '<br><small class="lox-uuid">' . substr($uuid, 0, 8) . '...</small>',
          ],
        ];

        $form['profiles'][$index]['versions'] = [
          '#markup' => '<span class="lox-badge">' . $backup_count . '</span>',
        ];

        $form['profiles'][$index]['size'] = [
          '#markup' => $this->formatBytes($total_size),
        ];

        $form['profiles'][$index]['last_backup'] = [
          '#markup' => $last_backup ? date('Y-m-d H:i', strtotime($last_backup)) : '<em>' . $this->t('Never') . '</em>',
        ];

        $form['profiles'][$index]['actions'] = [
          '#type' => 'container',
          '#attributes' => ['class' => ['lox-actions']],
        ];

        $form['profiles'][$index]['actions']['run'] = [
          '#type' => 'submit',
          '#value' => $this->t('Run Backup'),
          '#name' => 'run_' . $index,
          '#attributes' => ['class' => ['button', 'button--primary', 'button--small']],
          '#submit' => ['::runProfileBackup'],
          '#profile_uuid' => $uuid,
          '#profile_name' => $name,
        ];

        $form['profiles'][$index]['actions']['view'] = [
          '#type' => 'submit',
          '#value' => $this->t('View Versions'),
          '#name' => 'view_' . $index,
          '#attributes' => ['class' => ['button', 'button--small']],
          '#submit' => ['::viewProfileVersions'],
          '#profile_uuid' => $uuid,
        ];
      }
    }

    // Refresh button.
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['refresh'] = [
      '#type' => 'submit',
      '#value' => $this->t('Refresh'),
      '#submit' => ['::refreshProfiles'],
      '#attributes' => ['class' => ['button']],
    ];

    // Components selection for backup.
    $form['backup_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Backup Components'),
      '#open' => FALSE,
    ];

    $form['backup_options']['components'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Components to include in backup'),
      '#options' => [
        'database' => $this->t('Database'),
        'files' => $this->t('Public Files'),
        'private' => $this->t('Private Files'),
        'config' => $this->t('Configuration'),
      ],
      '#default_value' => ['database'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Default submit - refresh.
    $this->refreshProfiles($form, $form_state);
  }

  /**
   * Refresh profiles from server.
   */
  public function refreshProfiles(array &$form, FormStateInterface $form_state) {
    $this->messenger()->addStatus($this->t('Profiles refreshed.'));
  }

  /**
   * Run backup for a profile.
   */
  public function runProfileBackup(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $profile_uuid = $triggering_element['#profile_uuid'] ?? '';
    $profile_name = $triggering_element['#profile_name'] ?? 'Profile';

    if (empty($profile_uuid)) {
      $this->messenger()->addError($this->t('Invalid profile.'));
      return;
    }

    // Get selected components.
    $components = array_filter($form_state->getValue('components') ?: ['database']);
    if (empty($components)) {
      $components = ['database'];
    }

    // Run the backup.
    $result = $this->backupManager->runProfileBackup($profile_uuid, $profile_name, array_keys($components));

    if ($result['success']) {
      $this->messenger()->addStatus($this->t('Profile backup completed! UUID: @uuid', [
        '@uuid' => $result['uuid'],
      ]));
    }
    else {
      $this->messenger()->addError($this->t('Backup failed: @error', [
        '@error' => $result['error'],
      ]));
    }
  }

  /**
   * View profile versions.
   */
  public function viewProfileVersions(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $profile_uuid = $triggering_element['#profile_uuid'] ?? '';

    if (empty($profile_uuid)) {
      $this->messenger()->addError($this->t('Invalid profile.'));
      return;
    }

    // Redirect to backups list filtered by profile.
    $form_state->setRedirect('lox_backup.backups', [], [
      'query' => ['profile' => $profile_uuid],
    ]);
  }

  /**
   * Format bytes to human-readable size.
   *
   * @param int $bytes
   *   Size in bytes.
   *
   * @return string
   *   Formatted size.
   */
  protected function formatBytes(int $bytes): string {
    if ($bytes === 0) {
      return '--';
    }

    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = 0;

    while ($bytes >= 1024 && $i < count($units) - 1) {
      $bytes /= 1024;
      $i++;
    }

    return round($bytes, 2) . ' ' . $units[$i];
  }

}
