<?php

namespace Drupal\lox_backup\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to confirm backup restoration with text confirmation.
 */
class LoxRestoreConfirmForm extends ConfirmFormBase {

  /**
   * The backup UUID to restore.
   *
   * @var string
   */
  protected $uuid;

  /**
   * The backup details.
   *
   * @var array
   */
  protected $backupDetails;

  /**
   * The LOX API service.
   *
   * @var \Drupal\lox_backup\Service\LoxApi
   */
  protected $loxApi;

  /**
   * The confirmation text required.
   */
  const CONFIRMATION_TEXT = 'RESTORE';

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->loxApi = $container->get('lox_backup.api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'lox_restore_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to restore this backup?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('lox_backup.backups');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This will overwrite your current site data. This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Restore Backup');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $uuid = NULL) {
    $this->uuid = $uuid;

    // Get backup details from API.
    $result = $this->loxApi->getBackup($uuid);
    if ($result['success']) {
      $this->backupDetails = $result['data'];
    }

    $form = parent::buildForm($form, $form_state);

    // Hide the default description and add our custom warning.
    $form['description']['#access'] = FALSE;

    // Warning section.
    $form['warning'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['lox-restore-warning', 'messages', 'messages--warning'],
      ],
      '#weight' => -100,
    ];

    $form['warning']['icon'] = [
      '#markup' => '<strong>' . $this->t('Warning: This action is irreversible!') . '</strong>',
    ];

    $form['warning']['text'] = [
      '#markup' => '<p>' . $this->t('This will overwrite your current site data. This action cannot be undone.') . '</p>',
    ];

    // Backup details.
    if ($this->backupDetails) {
      $form['backup_details'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['lox-backup-details']],
        '#weight' => -90,
      ];

      $form['backup_details']['heading'] = [
        '#markup' => '<h4>' . $this->t('Backup to restore:') . '</h4>',
      ];

      $form['backup_details']['info'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['lox-backup-info-box']],
      ];

      $date = isset($this->backupDetails['created_at'])
        ? \Drupal::service('date.formatter')->format(strtotime($this->backupDetails['created_at']), 'medium')
        : $this->t('Unknown');

      $size = isset($this->backupDetails['size'])
        ? format_size($this->backupDetails['size'])
        : $this->t('Unknown');

      $form['backup_details']['info']['uuid'] = [
        '#markup' => '<div class="backup-uuid"><strong>' . $this->t('UUID:') . '</strong> ' . substr($uuid, 0, 12) . '...</div>',
      ];

      $form['backup_details']['info']['date'] = [
        '#markup' => '<div class="backup-date"><strong>' . $this->t('Created:') . '</strong> ' . $date . '</div>',
      ];

      $form['backup_details']['info']['size'] = [
        '#markup' => '<div class="backup-size"><strong>' . $this->t('Size:') . '</strong> ' . $size . '</div>',
      ];

      if (isset($this->backupDetails['component'])) {
        $form['backup_details']['info']['component'] = [
          '#markup' => '<div class="backup-component"><strong>' . $this->t('Component:') . '</strong> ' . ucfirst($this->backupDetails['component']) . '</div>',
        ];
      }
    }

    // Confirmation text input.
    $form['confirmation_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('To confirm, type %text below:', [
        '%text' => self::CONFIRMATION_TEXT,
      ]),
      '#required' => TRUE,
      '#attributes' => [
        'autocomplete' => 'off',
        'placeholder' => $this->t('Type @text to confirm', ['@text' => self::CONFIRMATION_TEXT]),
        'class' => ['lox-confirmation-input'],
      ],
      '#weight' => -80,
    ];

    // Store UUID in form.
    $form['uuid'] = [
      '#type' => 'hidden',
      '#value' => $uuid,
    ];

    // Add custom styles.
    $form['#attached']['library'][] = 'lox_backup/admin';

    // Make the submit button look dangerous.
    $form['actions']['submit']['#attributes']['class'][] = 'button--danger';

    // Add inline CSS for warning styling.
    $form['#attached']['html_head'][] = [
      [
        '#type' => 'html_tag',
        '#tag' => 'style',
        '#value' => '
          .lox-restore-warning {
            background-color: #fef3f2;
            border: 1px solid #fecaca;
            border-radius: 6px;
            padding: 16px;
            margin-bottom: 20px;
          }
          .lox-backup-details {
            margin-bottom: 20px;
          }
          .lox-backup-info-box {
            background: #f3f4f6;
            border-radius: 6px;
            padding: 12px 16px;
            margin-top: 8px;
          }
          .lox-backup-info-box > div {
            margin-bottom: 4px;
          }
          .lox-confirmation-input {
            width: 100%;
            max-width: 400px;
            padding: 12px;
            font-size: 16px;
            border: 2px solid #d1d5db;
            border-radius: 6px;
          }
          .lox-confirmation-input:focus {
            border-color: #dc2626;
            outline: none;
          }
          .button--danger {
            background-color: #dc2626 !important;
            border-color: #b91c1c !important;
            color: white !important;
          }
          .button--danger:hover {
            background-color: #b91c1c !important;
          }
        ',
      ],
      'lox_restore_styles',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $confirmation = strtoupper(trim($form_state->getValue('confirmation_text')));

    if ($confirmation !== self::CONFIRMATION_TEXT) {
      $form_state->setErrorByName('confirmation_text', $this->t('Please type @text to confirm the restore.', [
        '@text' => self::CONFIRMATION_TEXT,
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $uuid = $form_state->getValue('uuid');

    $result = $this->loxApi->requestRestore($uuid);

    if ($result['success']) {
      $this->messenger()->addStatus(
        $this->t('Restore requested for backup @uuid. You will be notified when it is ready for download.', [
          '@uuid' => substr($uuid, 0, 8) . '...',
        ])
      );
    }
    else {
      $this->messenger()->addError(
        $this->t('Failed to request restore: @error', ['@error' => $result['error']])
      );
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
