<?php

namespace Drupal\lox_backup\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * LOX API client service.
 */
class LoxApi {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * API key.
   *
   * @var string
   */
  protected $apiKey;

  /**
   * Base URL.
   *
   * @var string
   */
  protected $baseUrl;

  /**
   * Constructs a LoxApi object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ClientInterface $http_client) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;

    $config = $this->configFactory->get('lox_backup.settings');
    $this->apiKey = $config->get('api_key');
    $this->baseUrl = rtrim($config->get('api_url') ?: 'https://backlox.com/api', '/');
  }

  /**
   * Test API connection.
   *
   * @return array
   *   Result array with 'success' and 'data' or 'error' keys.
   */
  public function testConnection(): array {
    return $this->request('GET', '/v1/tenants/me');
  }

  /**
   * Get tenant quota.
   *
   * @return array
   *   Result array with quota information.
   */
  public function getQuota(): array {
    return $this->request('GET', '/v1/tenants/usage');
  }

  /**
   * Upload backup file.
   *
   * @param string $file_path
   *   Path to backup file.
   * @param array $options
   *   Backup options including:
   *   - name: Backup name
   *   - tags: Comma-separated tags
   *   - retention_days: Retention period (NULL = use frequency-based default)
   *   - immutable_days: Immutability period (NULL = same as retention)
   *   - frequency: Backup frequency (hourly, daily, weekly, monthly, yearly)
   *   - description: Backup description
   *   - source: Source identifier (drupal)
   *   - component: Component being backed up
   *   - source_identifier: Site identifier
   *   - profile_uuid: Associate with a remote profile
   *   - extra_metadata: Site metadata (auto-generated if not provided)
   *
   * @return array
   *   Result array with backup information.
   */
  public function uploadBackup(string $file_path, array $options = []): array {
    if (!file_exists($file_path)) {
      return ['success' => FALSE, 'error' => 'Backup file not found'];
    }

    $defaults = [
      'name' => basename($file_path),
      'tags' => 'drupal',
      'retention_days' => NULL,  // NULL = use frequency-based default from account settings
      'immutable_days' => NULL,  // NULL = use frequency-based default from account settings
      'frequency' => NULL,       // hourly, daily, weekly, monthly, yearly - determines retention policy
      'description' => 'Drupal backup from ' . \Drupal::request()->getHost(),
      'source' => 'drupal',
      'component' => NULL,
      'source_identifier' => $this->getSiteIdentifier(),
      'profile_uuid' => NULL,
      'extra_metadata' => NULL,
    ];

    $options = array_merge($defaults, $options);

    // Auto-generate site metadata if not provided
    if ($options['extra_metadata'] === NULL) {
      $options['extra_metadata'] = json_encode($this->getSiteMetadata());
    }
    elseif (is_array($options['extra_metadata'])) {
      $options['extra_metadata'] = json_encode($options['extra_metadata']);
    }

    // Build multipart data.
    $multipart = [
      [
        'name' => 'file',
        'contents' => fopen($file_path, 'r'),
        'filename' => basename($file_path),
      ],
      [
        'name' => 'name',
        'contents' => $options['name'],
      ],
      [
        'name' => 'tags',
        'contents' => $options['tags'],
      ],
    ];

    // Add retention_days if set (NULL = use frequency-based default).
    if ($options['retention_days'] !== NULL) {
      $multipart[] = [
        'name' => 'retention_days',
        'contents' => (string) $options['retention_days'],
      ];
    }

    // Add immutable_days if set.
    if ($options['immutable_days'] !== NULL) {
      $multipart[] = [
        'name' => 'immutable_days',
        'contents' => (string) $options['immutable_days'],
      ];
    }

    // Add frequency if set (determines retention policy).
    if (!empty($options['frequency'])) {
      $multipart[] = [
        'name' => 'frequency',
        'contents' => $options['frequency'],
      ];
    }

    $multipart[] = [
      'name' => 'description',
      'contents' => $options['description'],
    ];
    $multipart[] = [
      'name' => 'source',
      'contents' => $options['source'],
    ];

    // Add component if set.
    if (!empty($options['component'])) {
      $multipart[] = [
        'name' => 'component',
        'contents' => $options['component'],
      ];
    }

    // Add source_identifier if set.
    if (!empty($options['source_identifier'])) {
      $multipart[] = [
        'name' => 'source_identifier',
        'contents' => $options['source_identifier'],
      ];
    }

    // Add profile_uuid if set (for versioned profiles).
    if (!empty($options['profile_uuid'])) {
      $multipart[] = [
        'name' => 'profile_uuid',
        'contents' => $options['profile_uuid'],
      ];
    }

    // Add extra_metadata (site info for backup context).
    if (!empty($options['extra_metadata'])) {
      $multipart[] = [
        'name' => 'extra_metadata',
        'contents' => $options['extra_metadata'],
      ];
    }

    try {
      $response = $this->httpClient->request('POST', $this->baseUrl . '/v1/backups', [
        'headers' => [
          'X-API-Key' => $this->apiKey,
          'User-Agent' => 'LOX-Drupal/' . \Drupal::VERSION,
        ],
        'multipart' => $multipart,
        'timeout' => 300,
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);
      return ['success' => TRUE, 'data' => $data];
    }
    catch (GuzzleException $e) {
      return ['success' => FALSE, 'error' => $e->getMessage()];
    }
  }

  /**
   * Get backup status.
   *
   * @param string $uuid
   *   Backup UUID.
   *
   * @return array
   *   Result array with backup status.
   */
  public function getBackupStatus(string $uuid): array {
    return $this->request('GET', '/v1/backups/' . $uuid);
  }

  /**
   * Get latest backups by component.
   *
   * @param string $source
   *   Source identifier.
   * @param string|null $source_identifier
   *   Site identifier.
   *
   * @return array
   *   Result array with latest backups per component.
   */
  public function getLatestBackups(string $source = 'drupal', ?string $source_identifier = NULL): array {
    $params = ['source' => $source];
    if ($source_identifier === NULL) {
      $source_identifier = $this->getSiteIdentifier();
    }
    if ($source_identifier) {
      $params['source_identifier'] = $source_identifier;
    }
    return $this->request('GET', '/v1/backups/latest', $params);
  }

  /**
   * Get backup profiles from server.
   *
   * @param string $source
   *   Source filter (default: drupal).
   * @param string|null $source_identifier
   *   Site identifier (default: current site).
   *
   * @return array
   *   Result array with profiles list.
   */
  public function getProfiles(string $source = 'drupal', ?string $source_identifier = NULL): array {
    $params = ['source' => $source];
    if ($source_identifier === NULL) {
      $source_identifier = $this->getSiteIdentifier();
    }
    if ($source_identifier) {
      $params['source_identifier'] = $source_identifier;
    }
    // Note: trailing slash required for this endpoint
    return $this->request('GET', '/v1/backup-profiles/', $params);
  }

  /**
   * Get a specific backup profile with recent backups.
   *
   * @param string $uuid
   *   Profile UUID.
   * @param int $limit
   *   Number of recent backups to include.
   *
   * @return array
   *   Result array with profile details and backups.
   */
  public function getProfile(string $uuid, int $limit = 10): array {
    return $this->request('GET', '/v1/backup-profiles/' . $uuid, ['limit' => $limit]);
  }

  /**
   * Get all versions (backups) for a profile.
   *
   * @param string $uuid
   *   Profile UUID.
   * @param int $page
   *   Page number.
   * @param int $per_page
   *   Items per page.
   *
   * @return array
   *   Result array with paginated backups list.
   */
  public function getProfileVersions(string $uuid, int $page = 1, int $per_page = 20): array {
    return $this->request('GET', '/v1/backup-profiles/' . $uuid . '/versions', [
      'page' => $page,
      'per_page' => $per_page,
    ]);
  }

  /**
   * Create a custom backup profile on the server.
   *
   * @param string $name
   *   Profile name.
   * @param array $options
   *   Additional profile options.
   *
   * @return array
   *   Result array with created profile.
   */
  public function createProfile(string $name, array $options = []): array {
    $data = array_merge([
      'name' => $name,
      'source' => 'drupal',
      'source_identifier' => $this->getSiteIdentifier(),
      'is_custom' => TRUE,
    ], $options);
    return $this->request('POST', '/v1/backup-profiles', $data);
  }

  /**
   * Update a backup profile.
   *
   * @param string $uuid
   *   Profile UUID.
   * @param array $data
   *   Update data.
   *
   * @return array
   *   Result array with updated profile.
   */
  public function updateProfile(string $uuid, array $data): array {
    return $this->request('PATCH', '/v1/backup-profiles/' . $uuid, $data);
  }

  /**
   * Get Drupal site metadata for backup context.
   *
   * @return array
   *   Site metadata array.
   */
  public function getSiteMetadata(): array {
    $site_config = \Drupal::config('system.site');
    $modules = \Drupal::moduleHandler()->getModuleList();

    return [
      'site_url' => \Drupal::request()->getSchemeAndHttpHost(),
      'site_name' => $site_config->get('name'),
      'drupal_version' => \Drupal::VERSION,
      'php_version' => phpversion(),
      'module_count' => count($modules),
      'theme' => \Drupal::theme()->getActiveTheme()->getName(),
      'locale' => \Drupal::languageManager()->getCurrentLanguage()->getId(),
      'source' => 'drupal',
      'source_identifier' => $this->getSiteIdentifier(),
    ];
  }

  /**
   * Generate a unique identifier for this Drupal installation.
   *
   * Uses a hash of the site URL to create a consistent, privacy-friendly identifier.
   *
   * @return string
   *   Site identifier.
   */
  public function getSiteIdentifier(): string {
    // Check if we have a stored identifier.
    $config = $this->configFactory->get('lox_backup.settings');
    $storedId = $config->get('site_identifier');
    if ($storedId) {
      return $storedId;
    }

    // Generate identifier based on site information.
    $host = \Drupal::request()->getHost();
    $siteSettings = \Drupal\Core\Site\Settings::getInstance();
    $hashSalt = $siteSettings->getHashSalt();

    // For multisite installations, the host will be different.
    $identifier = $host;

    // Add a short hash for uniqueness.
    $hash = substr(md5($host . $hashSalt), 0, 8);
    $identifier = $identifier . '-' . $hash;

    // Store for consistency.
    $editableConfig = $this->configFactory->getEditable('lox_backup.settings');
    $editableConfig->set('site_identifier', $identifier);
    $editableConfig->save();

    return $identifier;
  }

  /**
   * List backups.
   *
   * @param array $params
   *   Query parameters (limit, offset, status, source).
   *
   * @return array
   *   Result array with backups list.
   */
  public function listBackups(array $params = []): array {
    $defaults = [
      'limit' => 50,
      'offset' => 0,
      'source' => 'drupal',
    ];
    return $this->request('GET', '/v1/backups', array_merge($defaults, $params));
  }

  /**
   * Request restore for a backup.
   *
   * @param string $uuid
   *   Backup UUID.
   * @param string $priority
   *   Restore priority (normal, high).
   *
   * @return array
   *   Result array with restore request status.
   */
  public function requestRestore(string $uuid, string $priority = 'normal'): array {
    return $this->request('POST', '/v1/backups/' . $uuid . '/restore', [
      'priority' => $priority,
    ]);
  }

  /**
   * Get download URL for a backup.
   *
   * @param string $uuid
   *   Backup UUID.
   *
   * @return array
   *   Result array with download URL.
   */
  public function getDownloadUrl(string $uuid): array {
    return $this->request('GET', '/v1/backups/' . $uuid . '/download');
  }

  /**
   * Cancel a pending backup.
   *
   * @param string $uuid
   *   Backup UUID.
   *
   * @return array
   *   Result array.
   */
  public function cancelBackup(string $uuid): array {
    return $this->request('POST', '/v1/backups/' . $uuid . '/cancel');
  }

  // NOTE: deleteBackup is intentionally NOT implemented.
  // Backups expire automatically based on retention policy.
  // Manual deletion is disabled as a security measure.

  /**
   * Wait for backup completion.
   *
   * @param string $uuid
   *   Backup UUID.
   * @param int $timeout
   *   Timeout in seconds.
   *
   * @return array
   *   Result array with completed backup data.
   */
  public function waitForCompletion(string $uuid, int $timeout = 3600): array {
    $start_time = time();
    $poll_interval = 5;

    while (TRUE) {
      $backup = $this->getBackupStatus($uuid);

      if (!$backup['success']) {
        return $backup;
      }

      if ($backup['data']['status'] === 'completed') {
        return ['success' => TRUE, 'data' => $backup['data']];
      }

      if ($backup['data']['status'] === 'failed') {
        return [
          'success' => FALSE,
          'error' => 'Backup failed: ' . ($backup['data']['status_message'] ?? 'Unknown error'),
        ];
      }

      if ($backup['data']['status'] === 'quarantine') {
        return [
          'success' => FALSE,
          'error' => 'Backup quarantined: Potential malware detected',
        ];
      }

      if ((time() - $start_time) >= $timeout) {
        return ['success' => FALSE, 'error' => 'Timeout waiting for backup completion'];
      }

      sleep($poll_interval);
    }
  }

  /**
   * Make API request.
   *
   * @param string $method
   *   HTTP method.
   * @param string $endpoint
   *   API endpoint.
   * @param array $params
   *   Request parameters.
   *
   * @return array
   *   Result array with 'success' and 'data' or 'error' keys.
   */
  protected function request(string $method, string $endpoint, array $params = []): array {
    if (empty($this->apiKey)) {
      return ['success' => FALSE, 'error' => 'API key not configured'];
    }

    $url = $this->baseUrl . $endpoint;

    $options = [
      'headers' => [
        'X-API-Key' => $this->apiKey,
        'Content-Type' => 'application/json',
        'User-Agent' => 'LOX-Drupal/' . \Drupal::VERSION,
      ],
      'timeout' => 30,
    ];

    if ($method === 'GET' && !empty($params)) {
      $options['query'] = $params;
    }
    elseif ($method !== 'GET' && !empty($params)) {
      $options['json'] = $params;
    }

    try {
      $response = $this->httpClient->request($method, $url, $options);
      $data = json_decode($response->getBody()->getContents(), TRUE);
      return ['success' => TRUE, 'data' => $data];
    }
    catch (GuzzleException $e) {
      $code = $e->getCode();
      $message = match ($code) {
        401 => 'Authentication failed. Check your API key.',
        403 => 'Access denied',
        404 => 'Resource not found',
        413 => 'Storage quota exceeded',
        default => $e->getMessage(),
      };
      return ['success' => FALSE, 'error' => $message];
    }
  }

}
