# -*- coding: utf-8 -*-

from odoo import http
from odoo.http import request
import json
import logging
import hmac
import hashlib
import tempfile
import os
import tarfile
import subprocess
import secrets

_logger = logging.getLogger(__name__)


class LoxBackupController(http.Controller):
    """Controller for LOX Backup webhooks and API endpoints"""

    def _verify_signature(self, api_key, signature, body):
        """Verify webhook signature"""
        if not signature:
            return False
        expected = hmac.new(
            api_key.encode('utf-8'),
            body,
            hashlib.sha256
        ).hexdigest()
        return hmac.compare_digest(signature, expected)

    def _get_config(self):
        """Get active LOX backup configuration"""
        return request.env['lox.backup.config'].sudo().search([
            ('active', '=', True),
            ('api_key', '!=', False),
        ], limit=1)

    @http.route('/lox/webhook/backup-status', type='json', auth='public', methods=['POST'], csrf=False)
    def webhook_backup_status(self, **kwargs):
        """Webhook endpoint for backup status updates from LOX"""
        try:
            data = request.get_json_data()
            backup_uuid = data.get('backup_id') or data.get('uuid')
            status = data.get('status')

            if not backup_uuid or not status:
                return {'error': 'Missing backup_id or status'}

            # Find the backup log
            log = request.env['lox.backup.log'].sudo().search([
                ('backup_uuid', '=', backup_uuid),
            ], limit=1)

            if not log:
                _logger.warning(f'Webhook: Backup log not found for UUID {backup_uuid}')
                return {'error': 'Backup not found'}

            # Map status
            status_mapping = {
                'PENDING': 'pending',
                'UPLOADING': 'in_progress',
                'VALIDATING': 'validating',
                'COMPLETED': 'completed',
                'FAILED': 'failed',
                'CANCELLED': 'cancelled',
            }

            updates = {
                'status': status_mapping.get(status, 'pending'),
            }

            if data.get('size'):
                updates['size_bytes'] = data['size']
            if data.get('checksum'):
                updates['checksum'] = data['checksum']
            if data.get('completed_at'):
                updates['completed_at'] = data['completed_at']
            if data.get('error'):
                updates['error_message'] = data['error']

            log.write(updates)
            _logger.info(f'Webhook: Updated backup {backup_uuid} status to {status}')

            return {'success': True}

        except Exception as e:
            _logger.exception('Webhook error')
            return {'error': str(e)}

    @http.route('/lox/webhook/restore-ready', type='json', auth='public', methods=['POST'], csrf=False)
    def webhook_restore_ready(self, **kwargs):
        """Webhook endpoint for restore ready notifications"""
        try:
            data = request.get_json_data()
            backup_uuid = data.get('backup_id') or data.get('uuid')
            download_url = data.get('download_url') or data.get('url')
            expires_at = data.get('expires_at')

            if not backup_uuid:
                return {'error': 'Missing backup_id'}

            # Find the backup log
            log = request.env['lox.backup.log'].sudo().search([
                ('backup_uuid', '=', backup_uuid),
            ], limit=1)

            if not log:
                _logger.warning(f'Webhook: Backup log not found for UUID {backup_uuid}')
                return {'error': 'Backup not found'}

            log.write({
                'restore_status': 'ready',
                'restore_url': download_url,
                'restore_expires_at': expires_at,
            })

            _logger.info(f'Webhook: Restore ready for backup {backup_uuid}')
            return {'success': True}

        except Exception as e:
            _logger.exception('Webhook error')
            return {'error': str(e)}

    @http.route('/lox/api/status', type='json', auth='public', methods=['GET', 'POST'], csrf=False)
    def api_status(self, **kwargs):
        """API endpoint to check module status"""
        config = self._get_config()
        return {
            'status': 'active' if config else 'not_configured',
            'source_identifier': config.source_identifier if config else None,
            'module_version': '18.0.1.0.0',
        }

    @http.route('/lox/api/trigger-backup', type='json', auth='public', methods=['POST'], csrf=False)
    def api_trigger_backup(self, **kwargs):
        """API endpoint to trigger a backup remotely"""
        try:
            data = request.get_json_data() or {}
            api_key = request.httprequest.headers.get('X-API-Key')

            config = self._get_config()
            if not config:
                return {'error': 'LOX Backup not configured'}

            # Verify API key
            if not api_key or api_key != config.api_key:
                return {'error': 'Invalid API key'}

            component = data.get('component', 'full')
            if component not in ('full', 'database', 'filestore'):
                component = 'full'

            # Create backup via wizard
            wizard = request.env['lox.backup.wizard'].sudo().create({
                'config_id': config.id,
                'component': component,
            })

            # Run backup
            wizard.action_run_backup()

            return {
                'success': True,
                'message': 'Backup triggered',
            }

        except Exception as e:
            _logger.exception('API trigger backup error')
            return {'error': str(e)}

    @http.route('/lox/api/backups', type='json', auth='public', methods=['GET', 'POST'], csrf=False)
    def api_list_backups(self, **kwargs):
        """API endpoint to list backups"""
        try:
            api_key = request.httprequest.headers.get('X-API-Key')

            config = self._get_config()
            if not config:
                return {'error': 'LOX Backup not configured'}

            # Verify API key
            if not api_key or api_key != config.api_key:
                return {'error': 'Invalid API key'}

            data = request.get_json_data() or {}
            limit = data.get('limit', 50)
            offset = data.get('offset', 0)

            logs = request.env['lox.backup.log'].sudo().search([
                ('config_id', '=', config.id),
            ], limit=limit, offset=offset, order='create_date desc')

            backups = []
            for log in logs:
                backups.append({
                    'uuid': log.backup_uuid,
                    'component': log.component,
                    'status': log.status,
                    'size': log.size_bytes,
                    'created_at': log.create_date.isoformat() if log.create_date else None,
                    'completed_at': log.completed_at.isoformat() if log.completed_at else None,
                })

            return {
                'backups': backups,
                'total': request.env['lox.backup.log'].sudo().search_count([
                    ('config_id', '=', config.id),
                ]),
            }

        except Exception as e:
            _logger.exception('API list backups error')
            return {'error': str(e)}

    # ========== PULL API Endpoints ==========

    def _verify_pull_token(self):
        """Verify PULL token from request header"""
        token = request.httprequest.headers.get('X-LOX-Pull-Token')
        if not token:
            return {'error': 'missing_token', 'message': 'Missing X-LOX-Pull-Token header'}

        config = self._get_config()
        if not config:
            return {'error': 'not_configured', 'message': 'LOX Backup not configured'}

        stored_token = config.pull_token
        token_expiry = config.pull_token_expiry

        if not stored_token:
            return {'error': 'not_configured', 'message': 'PULL token not configured'}

        import time
        if token_expiry and time.time() > token_expiry:
            return {'error': 'token_expired', 'message': 'PULL token has expired'}

        if not hmac.compare_digest(stored_token, token):
            return {'error': 'invalid_token', 'message': 'Invalid PULL token'}

        return None

    @http.route('/lox/pull/backup', type='json', auth='public', methods=['POST'], csrf=False)
    def pull_backup(self, **kwargs):
        """PULL endpoint: LOX initiates backup"""
        error = self._verify_pull_token()
        if error:
            return error

        try:
            data = request.get_json_data() or {}
            backup_type = data.get('type', 'full')
            frequency = data.get('frequency', 'daily')

            result = self._run_backup_for_pull(backup_type, data, frequency)

            if 'error' in result:
                return result

            # Create download token
            download_token = self._create_download_token(result['archive_path'])

            return {
                'success': True,
                'backup_name': result['name'],
                'size_bytes': os.path.getsize(result['archive_path']),
                'checksum': self._hash_file(result['archive_path']),
                'download_url': f'/lox/pull/download/{download_token}',
                'expires_in': 3600,
            }

        except Exception as e:
            _logger.exception('PULL backup error')
            return {'error': 'backup_failed', 'message': str(e)}

    def _run_backup_for_pull(self, backup_type, options, frequency):
        """Run backup for PULL mode (without LOX upload)"""
        import time
        from datetime import datetime

        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        db_name = request.env.cr.dbname
        backup_dir = tempfile.mkdtemp(prefix='lox_pull_')

        backup_name = f"odoo-{db_name}"

        try:
            if backup_type == 'full':
                backup_name += f"-full-{timestamp}"
                self._collect_full_backup(backup_dir)
            elif backup_type == 'component':
                component = options.get('component')
                if not component:
                    self._cleanup_temp(backup_dir)
                    return {'error': 'missing_component', 'message': 'Component required'}
                backup_name += f"-{component}-{timestamp}"
                self._collect_component_backup(backup_dir, component)
            elif backup_type == 'custom':
                elements = options.get('elements', [])
                if not elements:
                    self._cleanup_temp(backup_dir)
                    return {'error': 'missing_elements', 'message': 'Elements required'}
                slug = '-'.join([e.lower().replace(' ', '') for e in elements])
                backup_name += f"-custom-{slug}-{timestamp}"
                self._collect_custom_backup(backup_dir, elements)
            else:
                self._cleanup_temp(backup_dir)
                return {'error': 'invalid_type', 'message': 'Invalid backup type'}

            # Check if we have files
            if not os.listdir(backup_dir):
                self._cleanup_temp(backup_dir)
                return {'error': 'backup_empty', 'message': 'No files to backup'}

            # Create archive
            archive_path = os.path.join(tempfile.gettempdir(), f'{backup_name}.tar.gz')
            self._create_archive(backup_dir, archive_path)
            self._cleanup_temp(backup_dir)

            return {
                'name': backup_name,
                'archive_path': archive_path,
                'type': backup_type,
            }

        except Exception as e:
            self._cleanup_temp(backup_dir)
            raise

    def _collect_full_backup(self, backup_dir):
        """Collect full backup"""
        config = self._get_config()

        # Database backup
        db_path = os.path.join(backup_dir, 'database.sql')
        self._backup_database(db_path)

        # Filestore backup
        data_dir = os.environ.get('ODOO_DATA_DIR', '/var/lib/odoo')
        filestore_path = os.path.join(data_dir, 'filestore', request.env.cr.dbname)
        if os.path.exists(filestore_path):
            dest_path = os.path.join(backup_dir, 'filestore')
            self._recursive_copy(filestore_path, dest_path)

    def _collect_component_backup(self, backup_dir, component):
        """Collect component backup"""
        if component == 'database':
            db_path = os.path.join(backup_dir, 'database.sql')
            self._backup_database(db_path)
        elif component == 'filestore':
            data_dir = os.environ.get('ODOO_DATA_DIR', '/var/lib/odoo')
            filestore_path = os.path.join(data_dir, 'filestore', request.env.cr.dbname)
            if os.path.exists(filestore_path):
                dest_path = os.path.join(backup_dir, 'filestore')
                self._recursive_copy(filestore_path, dest_path)
        else:
            raise ValueError(f'Unknown component: {component}')

    def _collect_custom_backup(self, backup_dir, elements):
        """Collect custom backup"""
        if 'database' in elements:
            db_path = os.path.join(backup_dir, 'database.sql')
            self._backup_database(db_path)

        if 'filestore' in elements:
            data_dir = os.environ.get('ODOO_DATA_DIR', '/var/lib/odoo')
            filestore_path = os.path.join(data_dir, 'filestore', request.env.cr.dbname)
            if os.path.exists(filestore_path):
                dest_path = os.path.join(backup_dir, 'filestore')
                self._recursive_copy(filestore_path, dest_path)

    def _backup_database(self, output_path):
        """Backup PostgreSQL database"""
        db_name = request.env.cr.dbname

        # Use pg_dump
        cmd = ['pg_dump', '--no-owner', '--no-acl', '-f', output_path, db_name]

        env = os.environ.copy()
        # Add Odoo db credentials if available
        config = request.env['ir.config_parameter'].sudo()
        db_host = config.get_param('db_host', 'localhost')
        db_port = config.get_param('db_port', '5432')
        db_user = config.get_param('db_user', 'odoo')

        if db_host:
            env['PGHOST'] = db_host
        if db_port:
            env['PGPORT'] = str(db_port)
        if db_user:
            env['PGUSER'] = db_user

        try:
            subprocess.run(cmd, env=env, check=True, capture_output=True)
        except subprocess.CalledProcessError as e:
            _logger.error(f'pg_dump failed: {e.stderr.decode()}')
            raise RuntimeError(f'Database backup failed: {e.stderr.decode()}')

    def _recursive_copy(self, src, dst):
        """Recursively copy directory"""
        import shutil
        shutil.copytree(src, dst)

    def _create_archive(self, src_dir, archive_path):
        """Create tar.gz archive"""
        with tarfile.open(archive_path, 'w:gz') as tar:
            for item in os.listdir(src_dir):
                tar.add(os.path.join(src_dir, item), arcname=item)

    def _cleanup_temp(self, path):
        """Cleanup temporary directory"""
        import shutil
        if os.path.exists(path):
            shutil.rmtree(path, ignore_errors=True)

    def _hash_file(self, path):
        """Calculate SHA256 hash of file"""
        sha256 = hashlib.sha256()
        with open(path, 'rb') as f:
            for chunk in iter(lambda: f.read(8192), b''):
                sha256.update(chunk)
        return sha256.hexdigest()

    def _create_download_token(self, archive_path):
        """Create temporary download token"""
        token = secrets.token_hex(16)
        config = self._get_config()
        if config:
            downloads = json.loads(config.pull_downloads or '{}')
            downloads[token] = {
                'path': archive_path,
                'expiry': int(__import__('time').time()) + 3600,
            }
            config.sudo().write({'pull_downloads': json.dumps(downloads)})
        return token

    @http.route('/lox/pull/download/<string:token>', type='http', auth='public', methods=['GET'], csrf=False)
    def pull_download(self, token, **kwargs):
        """PULL endpoint: Download backup file"""
        config = self._get_config()
        if not config:
            return request.make_response(json.dumps({'error': 'not_configured'}), [('Content-Type', 'application/json')], 404)

        downloads = json.loads(config.pull_downloads or '{}')

        if token not in downloads:
            return request.make_response(json.dumps({'error': 'invalid_token'}), [('Content-Type', 'application/json')], 404)

        download = downloads[token]

        import time
        if time.time() > download['expiry'] or not os.path.exists(download['path']):
            del downloads[token]
            config.sudo().write({'pull_downloads': json.dumps(downloads)})
            return request.make_response(json.dumps({'error': 'expired'}), [('Content-Type', 'application/json')], 410)

        # Remove token (one-time download)
        del downloads[token]
        config.sudo().write({'pull_downloads': json.dumps(downloads)})

        # Return file
        with open(download['path'], 'rb') as f:
            content = f.read()

        checksum = self._hash_file(download['path'])
        os.unlink(download['path'])

        return request.make_response(
            content,
            [
                ('Content-Type', 'application/gzip'),
                ('Content-Disposition', f'attachment; filename="{os.path.basename(download["path"])}"'),
                ('X-LOX-Checksum', checksum),
            ]
        )

    @http.route('/lox/pull/status', type='json', auth='public', methods=['GET', 'POST'], csrf=False)
    def pull_status(self, **kwargs):
        """PULL endpoint: Get site status"""
        error = self._verify_pull_token()
        if error:
            return error

        config = self._get_config()

        # Get database size
        db_size = 0
        try:
            request.env.cr.execute("""
                SELECT pg_database_size(current_database())
            """)
            db_size = request.env.cr.fetchone()[0]
        except:
            pass

        return {
            'site_url': request.httprequest.host_url.rstrip('/'),
            'database': request.env.cr.dbname,
            'odoo_version': request.env['ir.module.module'].sudo().search([('name', '=', 'base')], limit=1).latest_version,
            'module_version': '18.0.1.3.0',
            'pull_enabled': True,
            'components': {
                'database': {'enabled': True, 'size_bytes': db_size},
                'filestore': {'enabled': True},
            },
            'last_backup': config.last_backup if config else None,
            'last_status': config.last_status if config else None,
        }

    @http.route('/lox/pull/token/refresh', type='json', auth='public', methods=['POST'], csrf=False)
    def pull_refresh_token(self, **kwargs):
        """PULL endpoint: Refresh PULL token"""
        api_key = request.httprequest.headers.get('X-API-Key')

        config = self._get_config()
        if not config:
            return {'error': 'not_configured'}

        if not api_key or api_key != config.api_key:
            return {'error': 'invalid_api_key'}

        import time
        token = secrets.token_hex(32)
        expiry = int(time.time()) + (365 * 86400)

        config.sudo().write({
            'pull_token': token,
            'pull_token_expiry': expiry,
        })

        return {
            'success': True,
            'token': token,
            'expires_at': __import__('datetime').datetime.fromtimestamp(expiry).isoformat(),
        }
