# -*- coding: utf-8 -*-

from odoo import models, api
import requests
import json
import logging
import tempfile
import os
import subprocess

_logger = logging.getLogger(__name__)


class LoxApi(models.AbstractModel):
    """LOX API Client - Abstract model for API interactions"""
    _name = 'lox.api'
    _description = 'LOX API Client'

    @api.model
    def create_client(self, config, db_name=None):
        """Create a new API client instance wrapper"""
        if db_name is None:
            db_name = self.env.cr.dbname
        return LoxApiClient(config.api_url, config.api_key, config.source_identifier, db_name)

    @api.model
    def create_client_from_params(self, api_url, api_key, source_identifier=None, db_name=None):
        """Create a new API client instance from parameters"""
        if db_name is None:
            db_name = self.env.cr.dbname
        return LoxApiClient(api_url, api_key, source_identifier, db_name)


class LoxApiClient:
    """API Client for LOX Backup Service"""

    def __init__(self, api_url, api_key, source_identifier=None, db_name=None):
        self.api_url = api_url.rstrip('/')
        self.api_key = api_key
        self.source_identifier = source_identifier
        self.db_name = db_name
        self.timeout = 30
        self._site_identifier = None

    def get_site_identifier(self):
        """Generate a unique identifier for this Odoo installation"""
        if self._site_identifier:
            return self._site_identifier

        import hashlib

        # Use database name as the base identifier
        db_name = self.db_name or 'odoo'

        # Create a consistent hash for uniqueness
        hash_input = f"{db_name}-{self.api_url}"
        hash_suffix = hashlib.md5(hash_input.encode()).hexdigest()[:8]

        self._site_identifier = f"odoo-{db_name}-{hash_suffix}"
        return self._site_identifier

    def _get_headers(self):
        """Get request headers"""
        return {
            'X-API-Key': self.api_key,
            'Content-Type': 'application/json',
            'Accept': 'application/json',
        }

    def _request(self, method, endpoint, data=None, params=None, timeout=None):
        """Make API request"""
        url = f'{self.api_url}/v1{endpoint}'
        headers = self._get_headers()
        timeout = timeout or self.timeout

        _logger.debug(f'LOX API {method} {url}')

        try:
            response = requests.request(
                method=method,
                url=url,
                headers=headers,
                json=data,
                params=params,
                timeout=timeout,
            )

            if response.status_code >= 400:
                _logger.error(f'LOX API error: {response.status_code} - {response.text}')
                return self._handle_error_response(response)

            try:
                return response.json()
            except:
                return {'success': True, 'data': response.text}

        except requests.exceptions.Timeout:
            _logger.error('LOX API timeout')
            return {'success': False, 'error_code': 'lox_conn_timeout', 'error': 'Request timeout'}
        except requests.exceptions.ConnectionError as e:
            _logger.error(f'LOX API connection error: {e}')
            return {'success': False, 'error_code': 'lox_conn_failed', 'error': f'Connection error: {str(e)}'}
        except Exception as e:
            _logger.exception('LOX API unexpected error')
            return {'success': False, 'error_code': 'lox_upload_failed', 'error': str(e)}

    def _handle_error_response(self, response):
        """Handle API error response with standard error codes"""
        status_code = response.status_code
        try:
            error_data = response.json()
            message = error_data.get('detail', response.text)
        except:
            message = response.text

        if status_code == 401:
            return {'success': False, 'error_code': 'lox_auth_invalid_key', 'error': message or 'Authentication failed'}
        elif status_code == 403:
            return {'success': False, 'error_code': 'lox_auth_forbidden', 'error': message or 'Access denied'}
        elif status_code == 404:
            return {'success': False, 'error_code': 'lox_backup_not_found', 'error': message or 'Resource not found'}
        elif status_code == 413:
            if 'quota' in message.lower():
                return {'success': False, 'error_code': 'lox_upload_quota_exceeded', 'error': message}
            return {'success': False, 'error_code': 'lox_upload_file_too_large', 'error': message or 'File too large'}
        elif status_code == 429:
            return {'success': False, 'error_code': 'lox_rate_exceeded', 'error': message or 'Rate limit exceeded'}
        else:
            return {'success': False, 'error_code': 'lox_upload_failed', 'error': message}

    def test_connection(self):
        """Test API connection by fetching tenant info"""
        result = self._request('GET', '/tenants/me')
        if result.get('success') and result.get('data'):
            return {'success': True, 'tenant': result.get('data')}
        elif result.get('id') or result.get('slug'):
            # Direct response from API (not wrapped)
            return {'success': True, 'tenant': result}
        return result

    def create_backup(self, backup_data):
        """Create a new backup"""
        return self._request('POST', '/backups', data=backup_data)

    def get_backup_status(self, backup_uuid):
        """Get backup status"""
        return self._request('GET', f'/backups/{backup_uuid}')

    def get_backup(self, backup_uuid):
        """Get backup details (alias for get_backup_status)"""
        return self.get_backup_status(backup_uuid)

    def list_backups(self, source_identifier=None, status=None, limit=50, offset=0):
        """List backups"""
        params = {
            'limit': limit,
            'offset': offset,
        }
        if source_identifier:
            params['source_identifier'] = source_identifier
        if status:
            params['status'] = status

        return self._request('GET', '/backups', params=params)

    def request_restore(self, backup_uuid, priority='normal'):
        """Request backup restore"""
        return self._request('POST', f'/backups/{backup_uuid}/restore', data={
            'priority': priority,
        })

    def get_download_url(self, backup_uuid):
        """Get download URL for backup"""
        return self._request('GET', f'/backups/{backup_uuid}/download')

    def cancel_backup(self, backup_uuid):
        """Cancel a pending backup"""
        return self._request('POST', f'/backups/{backup_uuid}/cancel')

    # ============================================
    # Remote Profiles API
    # ============================================

    def get_profiles(self, source='odoo', source_identifier=None):
        """Get backup profiles from server.

        Args:
            source: Filter by source (default: odoo)
            source_identifier: Filter by site identifier (default: current site)

        Returns:
            API response with profiles list
        """
        params = {'source': source}
        if source_identifier is None:
            source_identifier = self.get_site_identifier()
        if source_identifier:
            params['source_identifier'] = source_identifier

        # Note: trailing slash required for this endpoint
        return self._request('GET', '/backup-profiles/', params=params)

    def get_profile(self, profile_uuid, limit=10):
        """Get a specific backup profile with recent backups.

        Args:
            profile_uuid: Profile UUID
            limit: Number of recent backups to include

        Returns:
            API response with profile details and backups
        """
        return self._request('GET', f'/backup-profiles/{profile_uuid}', params={'limit': limit})

    def get_profile_versions(self, profile_uuid, page=1, per_page=20):
        """Get all versions (backups) for a profile.

        Args:
            profile_uuid: Profile UUID
            page: Page number
            per_page: Items per page

        Returns:
            API response with paginated backups list
        """
        return self._request('GET', f'/backup-profiles/{profile_uuid}/versions', params={
            'page': page,
            'per_page': per_page,
        })

    def create_profile(self, name, options=None):
        """Create a custom backup profile on the server.

        Args:
            name: Profile name
            options: Additional profile options (description, metadata, etc.)

        Returns:
            API response with created profile
        """
        options = options or {}
        data = {
            'name': name,
            'source': 'odoo',
            'source_identifier': self.get_site_identifier(),
            'is_custom': True,
        }
        data.update(options)
        return self._request('POST', '/backup-profiles', data=data)

    def update_profile(self, profile_uuid, data):
        """Update a backup profile.

        Args:
            profile_uuid: Profile UUID
            data: Update data

        Returns:
            API response with updated profile
        """
        return self._request('PATCH', f'/backup-profiles/{profile_uuid}', data=data)

    def get_site_metadata(self, env=None):
        """Get Odoo site metadata for backup context.

        Args:
            env: Odoo environment (optional, for additional metadata)

        Returns:
            Dict with site metadata
        """
        import odoo
        from odoo.release import version_info

        metadata = {
            'site_name': self.db_name,
            'odoo_version': '.'.join(map(str, version_info[:3])),
            'odoo_edition': 'enterprise' if hasattr(odoo, 'addons_path') else 'community',
            'python_version': '.'.join(map(str, __import__('sys').version_info[:3])),
            'source': 'odoo',
            'source_identifier': self.get_site_identifier(),
        }

        if env:
            try:
                # Get additional metadata from Odoo
                IrModule = env['ir.module.module'].sudo()
                installed_count = IrModule.search_count([('state', '=', 'installed')])
                metadata['module_count'] = installed_count

                # Get company info
                company = env.company
                if company:
                    metadata['company_name'] = company.name
                    metadata['currency'] = company.currency_id.name if company.currency_id else None

                # Get language
                metadata['locale'] = env.lang or 'en_US'

            except Exception as e:
                _logger.warning(f'Could not get additional metadata: {e}')

        return metadata

    # NOTE: delete_backup is now available for backups that are no longer immutable
    # Backups can only be deleted after their immutability period has expired.
    # This is a security measure - backups cannot be deleted while immutable
    # to protect against ransomware attacks and credential compromise.

    def delete_backup(self, backup_uuid):
        """Delete a backup that is no longer immutable.

        Args:
            backup_uuid: The UUID of the backup to delete

        Returns:
            API response dict with success status or error

        Note:
            The API will reject deletion requests for backups that are still
            within their immutability period. Only backups past their
            immutable_until date can be deleted.
        """
        return self._request('DELETE', f'/backups/{backup_uuid}')

    def upload_backup(self, file_path, options=None, env=None):
        """Upload backup file using standard multipart POST to /v1/backups.

        Args:
            file_path: Path to the backup file
            options: Dict with optional keys:
                - name: Backup name (defaults to filename)
                - description: Backup description
                - tags: Comma-separated tags or list
                - retention_days: Retention period (default 30)
                - immutable_days: Immutability period in days (0 = use server default)
                - component: Component being backed up (database, filestore, full)
                - metadata: Additional metadata dict
                - profile_uuid: Associate backup with a remote profile
                - extra_metadata: Site metadata (auto-generated if not provided)
            env: Odoo environment for generating site metadata

        Returns:
            API response dict with backup info or error
        """
        options = options or {}
        url = f'{self.api_url}/v1/backups'
        headers = {
            'X-API-Key': self.api_key,
            'User-Agent': 'LOX-Odoo/1.0.0',
        }

        # Prepare form fields
        form_data = {
            'name': options.get('name', os.path.basename(file_path)),
            'source': 'odoo',
            'source_identifier': self.get_site_identifier(),
        }

        # Add retention_days only if explicitly set (otherwise use frequency-based default)
        if options.get('retention_days') is not None:
            form_data['retention_days'] = str(options['retention_days'])

        # Add immutable_days if explicitly set (0 = use server default)
        if options.get('immutable_days') is not None:
            form_data['immutable_days'] = str(options['immutable_days'])

        # Add frequency if set (determines retention policy)
        if options.get('frequency'):
            form_data['frequency'] = options['frequency']

        if options.get('description'):
            form_data['description'] = options['description']

        if options.get('tags'):
            tags = options['tags']
            if isinstance(tags, list):
                tags = ','.join(tags)
            form_data['tags'] = tags

        if options.get('component'):
            form_data['component'] = options['component']

        if options.get('metadata'):
            form_data['metadata'] = json.dumps(options['metadata'])

        # Add profile_uuid if provided (for versioned profiles)
        if options.get('profile_uuid'):
            form_data['profile_uuid'] = options['profile_uuid']

        # Add extra_metadata (site info for backup context)
        if options.get('extra_metadata'):
            extra = options['extra_metadata']
            if isinstance(extra, dict):
                extra = json.dumps(extra)
            form_data['extra_metadata'] = extra
        else:
            # Auto-generate site metadata
            form_data['extra_metadata'] = json.dumps(self.get_site_metadata(env))

        try:
            with open(file_path, 'rb') as f:
                files = {'file': (os.path.basename(file_path), f, 'application/gzip')}
                response = requests.post(
                    url,
                    headers=headers,
                    data=form_data,
                    files=files,
                    timeout=3600,  # 1 hour timeout for large files
                )

            if response.status_code >= 400:
                _logger.error(f'Upload failed: {response.status_code} - {response.text}')
                return self._handle_error_response(response)

            return response.json()

        except requests.exceptions.Timeout:
            _logger.error('Upload timeout')
            return {'success': False, 'error_code': 'lox_conn_timeout', 'error': 'Upload timeout'}
        except FileNotFoundError:
            return {'success': False, 'error_code': 'lox_upload_file_not_found', 'error': f'File not found: {file_path}'}
        except Exception as e:
            _logger.exception('Upload failed')
            return {'success': False, 'error_code': 'lox_upload_failed', 'error': str(e)}


class OdooBackupCreator:
    """Helper class to create Odoo backups"""

    def __init__(self, env):
        self.env = env
        self.db_name = env.cr.dbname

    def create_database_backup(self):
        """Create database backup"""
        import odoo.tools.config as config

        temp_dir = tempfile.mkdtemp()
        backup_file = os.path.join(temp_dir, f'{self.db_name}_db.sql')

        try:
            # Get database connection info from Odoo config (works in any installation)
            db_host = config.get('db_host') or os.environ.get('HOST') or 'localhost'
            db_port = str(config.get('db_port') or os.environ.get('PORT') or 5432)
            db_user = config.get('db_user') or os.environ.get('USER') or 'odoo'
            db_password = config.get('db_password') or os.environ.get('PASSWORD') or ''

            # Use pg_dump to create database backup
            cmd = [
                'pg_dump',
                '--no-owner',
                '--no-acl',
                '-f', backup_file,
                self.db_name,
            ]

            # Add host/port only if not localhost socket
            if db_host and db_host not in ('localhost', '/var/run/postgresql'):
                cmd.extend(['-h', db_host])
            if db_port and db_port != '5432':
                cmd.extend(['-p', db_port])
            if db_user:
                cmd.extend(['-U', db_user])

            # Set password via environment
            env = os.environ.copy()
            if db_password:
                env['PGPASSWORD'] = db_password

            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                env=env,
            )

            if result.returncode != 0:
                raise Exception(f'pg_dump failed: {result.stderr}')

            # Compress the backup
            compressed_file = backup_file + '.gz'
            import gzip
            import shutil

            with open(backup_file, 'rb') as f_in:
                with gzip.open(compressed_file, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)

            os.remove(backup_file)
            return compressed_file

        except Exception as e:
            _logger.exception('Database backup failed')
            raise

    def create_filestore_backup(self):
        """Create filestore backup"""
        import tarfile

        # Get filestore path
        data_dir = self.env['ir.config_parameter'].sudo().get_param(
            'ir_attachment.location',
            default='/var/lib/odoo/filestore'
        )

        filestore_path = os.path.join(data_dir, self.db_name)

        if not os.path.exists(filestore_path):
            # Try alternative path
            filestore_path = os.path.join('/var/lib/odoo/filestore', self.db_name)

        if not os.path.exists(filestore_path):
            _logger.warning(f'Filestore not found at {filestore_path}')
            return None

        temp_dir = tempfile.mkdtemp()
        backup_file = os.path.join(temp_dir, f'{self.db_name}_filestore.tar.gz')

        try:
            with tarfile.open(backup_file, 'w:gz') as tar:
                tar.add(filestore_path, arcname='filestore')

            return backup_file

        except Exception as e:
            _logger.exception('Filestore backup failed')
            raise

    def create_full_backup(self):
        """Create full backup (database + filestore)"""
        import tarfile

        temp_dir = tempfile.mkdtemp()
        backup_file = os.path.join(temp_dir, f'{self.db_name}_full.tar.gz')

        try:
            db_backup = self.create_database_backup()
            filestore_backup = self.create_filestore_backup()

            with tarfile.open(backup_file, 'w:gz') as tar:
                if db_backup:
                    tar.add(db_backup, arcname=os.path.basename(db_backup))
                if filestore_backup:
                    tar.add(filestore_backup, arcname=os.path.basename(filestore_backup))

            # Cleanup temp files
            if db_backup:
                os.remove(db_backup)
            if filestore_backup:
                os.remove(filestore_backup)

            return backup_file

        except Exception as e:
            _logger.exception('Full backup failed')
            raise
