# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import UserError, ValidationError
import logging
import hashlib

_logger = logging.getLogger(__name__)


class LoxBackupConfig(models.Model):
    _name = 'lox.backup.config'
    _description = 'LOX Backup Configuration'
    _inherit = ['mail.thread', 'mail.activity.mixin']
    _rec_name = 'display_name'

    name = fields.Char(
        string='Name',
        compute='_compute_name',
        store=True,
        readonly=False,
    )
    display_name = fields.Char(
        string='Display Name',
        compute='_compute_display_name',
        store=True,
    )
    company_id = fields.Many2one(
        'res.company',
        string='Company',
        required=True,
        default=lambda self: self.env.company,
        tracking=True,
    )
    active = fields.Boolean(
        string='Active',
        default=True,
        tracking=True,
    )
    api_url = fields.Char(
        string='API URL',
        default='https://backlox.com/api',
        help='LOX API base URL (e.g., https://backlox.com/api)',
        tracking=True,
    )
    api_key = fields.Char(
        string='API Key',
        help='Your LOX API key',
        groups='base.group_system',
    )
    source_identifier = fields.Char(
        string='Source Identifier',
        compute='_compute_source_identifier',
        store=True,
        help='Auto-generated unique identifier for this Odoo instance',
    )
    last_connection_test = fields.Datetime(
        string='Last Connection Test',
        readonly=True,
    )
    connection_status = fields.Selection([
        ('unknown', 'Unknown'),
        ('connected', 'Connected'),
        ('error', 'Error'),
    ], string='Connection Status', default='unknown', readonly=True)

    # Backup settings
    backup_database = fields.Boolean(
        string='Backup Database',
        default=True,
        help='Include database in backups',
    )
    backup_filestore = fields.Boolean(
        string='Backup Filestore',
        default=True,
        help='Include filestore (attachments) in backups',
    )
    backup_modules = fields.Boolean(
        string='Backup Modules Info',
        default=True,
        help='Include list of installed modules and their versions',
    )

    # Retention settings
    retention_days = fields.Integer(
        string='Retention Days',
        default=30,
        help='Number of days to retain backups in LOX storage',
    )
    immutable_days = fields.Integer(
        string='Immutable Period (days)',
        default=0,
        help='Number of days backups are protected from deletion (0 = use server default, leave empty for no immutability override)',
    )

    # Tags
    default_tags = fields.Char(
        string='Default Tags',
        default='odoo,automated',
        help='Comma-separated tags to apply to all backups',
    )

    # PULL API settings
    pull_enabled = fields.Boolean(
        string='Enable PULL API',
        default=False,
        help='Allow LOX to initiate backups remotely (PULL model)',
    )
    pull_token = fields.Char(
        string='PULL Token',
        groups='base.group_system',
        help='Token for PULL API authentication (auto-generated)',
    )
    pull_token_expiry = fields.Integer(
        string='PULL Token Expiry',
        help='PULL token expiry timestamp',
    )
    pull_downloads = fields.Text(
        string='Pending Downloads',
        default='{}',
        help='JSON storage for pending download tokens',
    )
    last_backup = fields.Datetime(
        string='Last PULL Backup',
        readonly=True,
    )
    last_status = fields.Char(
        string='Last PULL Status',
        readonly=True,
    )

    # Statistics
    total_backups = fields.Integer(
        string='Total Backups',
        compute='_compute_backup_stats',
        store=False,
    )
    last_backup_date = fields.Datetime(
        string='Last Backup',
        compute='_compute_backup_stats',
        store=False,
    )

    # Schedules
    schedule_ids = fields.One2many(
        'lox.backup.schedule',
        'config_id',
        string='Backup Schedules',
    )

    # Logs
    log_ids = fields.One2many(
        'lox.backup.log',
        'config_id',
        string='Backup Logs',
    )

    # Profiles
    profile_ids = fields.One2many(
        'lox.backup.profile',
        'config_id',
        string='Backup Profiles',
    )

    _sql_constraints = [
        ('company_unique', 'UNIQUE(company_id)', 'Only one LOX configuration per company is allowed!'),
    ]

    @api.depends('company_id')
    def _compute_name(self):
        for record in self:
            if record.company_id:
                record.name = f'LOX Backup - {record.company_id.name}'
            else:
                record.name = 'LOX Backup Configuration'

    @api.depends('company_id', 'connection_status')
    def _compute_display_name(self):
        for record in self:
            status_icon = ''
            if record.connection_status == 'connected':
                status_icon = ' ✓'
            elif record.connection_status == 'error':
                status_icon = ' ✗'
            if record.company_id:
                record.display_name = f'{record.company_id.name}{status_icon}'
            else:
                record.display_name = f'LOX Backup{status_icon}'

    @api.depends('company_id')
    def _compute_source_identifier(self):
        """Generate a unique source identifier for this Odoo instance."""
        db_name = self.env.cr.dbname
        base_url = self.env['ir.config_parameter'].sudo().get_param('web.base.url', '')
        for record in self:
            # Create unique identifier from db name + base URL + company
            unique_string = f"{db_name}:{base_url}:{record.company_id.id if record.company_id else 0}"
            record.source_identifier = hashlib.md5(unique_string.encode()).hexdigest()[:16]

    @api.model
    def get_config_for_company(self, company=None):
        """Get or create LOX config for the given company (or current company)."""
        if company is None:
            company = self.env.company
        config = self.search([('company_id', '=', company.id)], limit=1)
        return config

    @api.model
    def ensure_config_for_company(self, company=None):
        """Get or create LOX config for the given company (or current company)."""
        if company is None:
            company = self.env.company
        config = self.get_config_for_company(company)
        if not config:
            config = self.create({
                'company_id': company.id,
            })
        return config

    @api.depends('log_ids')
    def _compute_backup_stats(self):
        for record in self:
            logs = self.env['lox.backup.log'].search([
                ('config_id', '=', record.id),
                ('status', '=', 'completed'),
            ])
            record.total_backups = len(logs)
            if logs:
                record.last_backup_date = max(logs.mapped('create_date'))
            else:
                record.last_backup_date = False

    def action_test_connection(self):
        """Test API connection"""
        self.ensure_one()
        api = self.env['lox.api'].create_client(self)

        try:
            result = api.test_connection()
            if result.get('success'):
                self.write({
                    'connection_status': 'connected',
                    'last_connection_test': fields.Datetime.now(),
                })
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Success'),
                        'message': _('Connection to LOX API successful!'),
                        'type': 'success',
                        'sticky': False,
                    }
                }
            else:
                raise UserError(_('Connection failed: %s') % result.get('error', 'Unknown error'))
        except Exception as e:
            self.write({
                'connection_status': 'error',
                'last_connection_test': fields.Datetime.now(),
            })
            raise UserError(_('Connection failed: %s') % str(e))

    def action_run_backup(self):
        """Run a manual backup"""
        self.ensure_one()

        if not self.api_key:
            raise UserError(_('Please configure your API key first.'))

        return {
            'name': _('Run Backup'),
            'type': 'ir.actions.act_window',
            'res_model': 'lox.backup.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_config_id': self.id,
            }
        }

    def action_view_backups(self):
        """View all backups for this configuration"""
        self.ensure_one()
        return {
            'name': _('Backups'),
            'type': 'ir.actions.act_window',
            'res_model': 'lox.backup.log',
            'view_mode': 'tree,form',
            'domain': [('config_id', '=', self.id)],
            'context': {'default_config_id': self.id},
        }

    def action_view_schedules(self):
        """View all schedules for this configuration"""
        self.ensure_one()
        return {
            'name': _('Schedules'),
            'type': 'ir.actions.act_window',
            'res_model': 'lox.backup.schedule',
            'view_mode': 'tree,form',
            'domain': [('config_id', '=', self.id)],
            'context': {'default_config_id': self.id},
        }
