# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import UserError
import logging

_logger = logging.getLogger(__name__)


class LoxBackupLog(models.Model):
    _name = 'lox.backup.log'
    _description = 'LOX Backup Log'
    _order = 'create_date desc'
    _rec_name = 'display_name'

    config_id = fields.Many2one(
        'lox.backup.config',
        string='Configuration',
        required=True,
        ondelete='cascade',
    )
    company_id = fields.Many2one(
        related='config_id.company_id',
        store=True,
        readonly=True,
    )
    schedule_id = fields.Many2one(
        'lox.backup.schedule',
        string='Schedule',
        ondelete='set null',
    )
    profile_id = fields.Many2one(
        'lox.backup.profile',
        string='Profile',
        ondelete='set null',
    )
    backup_uuid = fields.Char(
        string='Backup UUID',
        readonly=True,
        index=True,
    )
    display_name = fields.Char(
        string='Name',
        compute='_compute_display_name',
        store=True,
    )

    component = fields.Selection([
        ('full', 'Full Backup'),
        ('database', 'Database Only'),
        ('filestore', 'Filestore Only'),
        ('modules', 'Modules Info Only'),
    ], string='Component', default='full')

    status = fields.Selection([
        ('pending', 'Pending'),
        ('in_progress', 'In Progress'),
        ('validating', 'Validating'),
        ('completed', 'Completed'),
        ('failed', 'Failed'),
        ('cancelled', 'Cancelled'),
    ], string='Status', default='pending', required=True, index=True)

    size_bytes = fields.Float(
        string='Size (bytes)',
        readonly=True,
    )
    size_display = fields.Char(
        string='Size',
        compute='_compute_size_display',
    )

    started_at = fields.Datetime(
        string='Started At',
        readonly=True,
    )
    completed_at = fields.Datetime(
        string='Completed At',
        readonly=True,
    )
    duration = fields.Float(
        string='Duration (s)',
        compute='_compute_duration',
        store=True,
    )

    error_message = fields.Text(
        string='Error Message',
        readonly=True,
    )

    # Storage info
    storage_path = fields.Char(
        string='Storage Path',
        readonly=True,
    )
    checksum = fields.Char(
        string='Checksum',
        readonly=True,
    )

    # Restore tracking
    restore_requested = fields.Boolean(
        string='Restore Requested',
        default=False,
    )
    restore_status = fields.Selection([
        ('none', 'Not Requested'),
        ('pending', 'Pending'),
        ('in_progress', 'In Progress'),
        ('ready', 'Ready for Download'),
        ('completed', 'Restored'),
        ('failed', 'Failed'),
    ], string='Restore Status', default='none')
    restore_url = fields.Char(
        string='Restore URL',
        readonly=True,
    )
    restore_expires_at = fields.Datetime(
        string='Restore Expires At',
        readonly=True,
    )

    # Immutability tracking
    immutable_until = fields.Datetime(
        string='Immutable Until',
        readonly=True,
        help='Backup cannot be deleted before this date. After this date, manual deletion is allowed.',
    )
    is_immutable = fields.Boolean(
        string='Is Immutable',
        compute='_compute_is_immutable',
        store=False,
        help='Whether the backup is currently protected by immutability policy.',
    )
    can_delete = fields.Boolean(
        string='Can Delete',
        compute='_compute_can_delete',
        store=False,
        help='Whether the backup can be manually deleted.',
    )

    @api.depends('immutable_until')
    def _compute_is_immutable(self):
        """Check if backup is currently immutable."""
        from datetime import datetime
        now = fields.Datetime.now()
        for record in self:
            if record.immutable_until:
                record.is_immutable = now < record.immutable_until
            else:
                # If no immutable_until is set, consider it immutable (safe default)
                record.is_immutable = True

    @api.depends('status', 'is_immutable')
    def _compute_can_delete(self):
        """Check if backup can be deleted."""
        for record in self:
            # Can only delete completed backups that are no longer immutable
            record.can_delete = (
                record.status == 'completed' and
                record.backup_uuid and
                not record.is_immutable
            )

    @api.depends('backup_uuid', 'create_date')
    def _compute_display_name(self):
        for record in self:
            date_str = record.create_date.strftime('%Y-%m-%d %H:%M') if record.create_date else 'New'
            uuid_short = record.backup_uuid[:8] if record.backup_uuid else 'pending'
            record.display_name = f'{date_str} [{uuid_short}]'

    @api.depends('size_bytes')
    def _compute_size_display(self):
        for record in self:
            if not record.size_bytes:
                record.size_display = '-'
            elif record.size_bytes < 1024:
                record.size_display = f'{record.size_bytes:.0f} B'
            elif record.size_bytes < 1024 * 1024:
                record.size_display = f'{record.size_bytes / 1024:.1f} KB'
            elif record.size_bytes < 1024 * 1024 * 1024:
                record.size_display = f'{record.size_bytes / (1024 * 1024):.1f} MB'
            else:
                record.size_display = f'{record.size_bytes / (1024 * 1024 * 1024):.2f} GB'

    @api.depends('started_at', 'completed_at')
    def _compute_duration(self):
        for record in self:
            if record.started_at and record.completed_at:
                delta = record.completed_at - record.started_at
                record.duration = delta.total_seconds()
            else:
                record.duration = 0

    def action_refresh_status(self):
        """Refresh backup status from LOX API"""
        self.ensure_one()
        if not self.backup_uuid:
            raise UserError(_('No backup UUID available.'))

        api = self.env['lox.api'].create_client(self.config_id)

        try:
            result = api.get_backup_status(self.backup_uuid)

            status_mapping = {
                'PENDING': 'pending',
                'UPLOADING': 'in_progress',
                'VALIDATING': 'validating',
                'COMPLETED': 'completed',
                'FAILED': 'failed',
                'CANCELLED': 'cancelled',
            }

            updates = {}
            if result.get('status'):
                updates['status'] = status_mapping.get(result['status'], 'pending')
            if result.get('size'):
                updates['size_bytes'] = result['size']
            if result.get('checksum'):
                updates['checksum'] = result['checksum']
            if result.get('completed_at'):
                updates['completed_at'] = result['completed_at']
            if result.get('error'):
                updates['error_message'] = result['error']

            # Parse immutable_until from API response
            if result.get('immutable_until'):
                try:
                    from dateutil import parser
                    immutable_until = parser.parse(result['immutable_until'])
                    updates['immutable_until'] = immutable_until
                except (ValueError, TypeError):
                    _logger.warning(f'Could not parse immutable_until: {result.get("immutable_until")}')

            if updates:
                self.write(updates)

            # Build notification message
            message = _('Backup status: %s') % self.status
            if self.immutable_until:
                if self.is_immutable:
                    message += _('\nImmutable until: %s') % fields.Datetime.to_string(self.immutable_until)
                else:
                    message += _('\nImmutability expired - can be deleted')

            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': _('Status Updated'),
                    'message': message,
                    'type': 'info',
                    'sticky': False,
                }
            }
        except Exception as e:
            raise UserError(_('Failed to refresh status: %s') % str(e))

    def action_request_restore(self):
        """Request restore/download of this backup"""
        self.ensure_one()
        if not self.backup_uuid:
            raise UserError(_('No backup UUID available.'))

        if self.status != 'completed':
            raise UserError(_('Only completed backups can be restored.'))

        api = self.env['lox.api'].create_client(self.config_id)

        try:
            result = api.request_restore(self.backup_uuid)

            self.write({
                'restore_requested': True,
                'restore_status': 'pending',
            })

            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': _('Restore Requested'),
                    'message': _('Restore request submitted. Check back soon for download link.'),
                    'type': 'success',
                    'sticky': False,
                }
            }
        except Exception as e:
            raise UserError(_('Failed to request restore: %s') % str(e))

    def action_get_download_url(self):
        """Get download URL for restored backup"""
        self.ensure_one()
        if not self.backup_uuid:
            raise UserError(_('No backup UUID available.'))

        api = self.env['lox.api'].create_client(self.config_id)

        try:
            result = api.get_download_url(self.backup_uuid)

            if result.get('url'):
                self.write({
                    'restore_url': result['url'],
                    'restore_status': 'ready',
                    'restore_expires_at': result.get('expires_at'),
                })

                return {
                    'type': 'ir.actions.act_url',
                    'url': result['url'],
                    'target': 'new',
                }
            elif result.get('status') == 'pending':
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Not Ready'),
                        'message': _('Backup is still being prepared for download. Please try again later.'),
                        'type': 'warning',
                        'sticky': False,
                    }
                }
            else:
                raise UserError(_('Download not available: %s') % result.get('error', 'Unknown error'))
        except UserError:
            raise
        except Exception as e:
            raise UserError(_('Failed to get download URL: %s') % str(e))

    def action_cancel_backup(self):
        """Cancel a pending or in-progress backup"""
        self.ensure_one()
        if self.status not in ('pending', 'in_progress', 'validating'):
            raise UserError(_('Only pending or in-progress backups can be cancelled.'))

        # TODO: Call API to cancel backup
        self.write({
            'status': 'cancelled',
        })

        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': _('Backup Cancelled'),
                'message': _('Backup has been cancelled.'),
                'type': 'info',
                'sticky': False,
            }
        }

    def action_delete_backup(self):
        """Delete a backup that is no longer immutable"""
        self.ensure_one()
        if not self.backup_uuid:
            raise UserError(_('No backup UUID available.'))

        if self.status != 'completed':
            raise UserError(_('Only completed backups can be deleted.'))

        if self.is_immutable:
            if self.immutable_until:
                raise UserError(_(
                    'This backup is protected by immutability policy until %s. '
                    'It cannot be deleted before that date.'
                ) % fields.Datetime.to_string(self.immutable_until))
            else:
                raise UserError(_(
                    'This backup is protected by immutability policy. '
                    'Please refresh the status to check when it can be deleted.'
                ))

        api = self.env['lox.api'].create_client(self.config_id)

        try:
            result = api.delete_backup(self.backup_uuid)

            if result.get('success') is False:
                error = result.get('error', 'Unknown error')
                # Check if it's an immutability error from the API
                if 'immutable' in error.lower():
                    raise UserError(_(
                        'Cannot delete backup: %s\n\n'
                        'Please refresh the backup status to update immutability information.'
                    ) % error)
                raise UserError(_('Failed to delete backup: %s') % error)

            # Mark as deleted locally (or remove the record)
            self.write({
                'status': 'cancelled',
                'error_message': _('Deleted by user on %s') % fields.Datetime.now(),
            })

            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': _('Backup Deleted'),
                    'message': _('Backup has been successfully deleted from LOX storage.'),
                    'type': 'success',
                    'sticky': False,
                }
            }
        except UserError:
            raise
        except Exception as e:
            raise UserError(_('Failed to delete backup: %s') % str(e))

    @api.model
    def cron_refresh_pending_backups(self):
        """Cron job to refresh status of pending backups"""
        pending_logs = self.search([
            ('status', 'in', ('pending', 'in_progress', 'validating')),
            ('backup_uuid', '!=', False),
        ])

        for log in pending_logs:
            try:
                log.action_refresh_status()
            except Exception as e:
                _logger.error(f'Failed to refresh backup {log.backup_uuid}: {e}')
