# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import UserError
import logging
import os
import tempfile

_logger = logging.getLogger(__name__)


class LoxBackupProfile(models.Model):
    """Backup profiles for quick custom backups"""
    _name = 'lox.backup.profile'
    _description = 'LOX Backup Profile'
    _order = 'sequence, name'

    name = fields.Char(
        string='Profile Name',
        required=True,
    )
    config_id = fields.Many2one(
        'lox.backup.config',
        string='Configuration',
        required=True,
        ondelete='cascade',
    )
    company_id = fields.Many2one(
        related='config_id.company_id',
        store=True,
        readonly=True,
    )
    active = fields.Boolean(
        string='Active',
        default=True,
    )
    sequence = fields.Integer(
        string='Sequence',
        default=10,
    )

    # Remote profile sync
    remote_uuid = fields.Char(
        string='Remote Profile UUID',
        readonly=True,
        help='UUID of the profile on LOX server (for versioning)',
    )
    is_synced = fields.Boolean(
        string='Synced with Server',
        compute='_compute_is_synced',
        store=True,
    )

    # What to backup
    include_database = fields.Boolean(
        string='Include Database',
        default=True,
        help='Include full database dump',
    )
    include_filestore = fields.Boolean(
        string='Include Filestore',
        default=True,
        help='Include all attachments and files',
    )
    include_modules = fields.Boolean(
        string='Include Modules Info',
        default=True,
        help='Include list of installed modules',
    )

    # Optional: specific models only (for partial DB backup in future)
    # For now, database is always full

    # Custom settings
    custom_tags = fields.Char(
        string='Custom Tags',
        help='Additional comma-separated tags for this profile',
    )
    # Retention settings
    use_default_retention = fields.Boolean(
        string='Use Server Default Retention',
        default=True,
        help='If checked, uses the retention policy configured on the server',
    )
    retention_days = fields.Integer(
        string='Retention Days',
        default=30,
        help='Number of days to keep the backup',
    )
    use_default_immutable = fields.Boolean(
        string='Use Server Default Immutability',
        default=True,
        help='If checked, uses the immutability policy configured on the server',
    )
    immutable_days = fields.Integer(
        string='Immutable Period (days)',
        default=7,
        help='Number of days the backup cannot be deleted (anti-ransomware)',
    )
    description = fields.Text(
        string='Description',
        help='Description of what this profile backs up',
    )

    # Stats
    last_run = fields.Datetime(
        string='Last Run',
        readonly=True,
    )
    run_count = fields.Integer(
        string='Run Count',
        default=0,
        readonly=True,
    )
    backup_count = fields.Integer(
        string='Version Count',
        readonly=True,
        help='Number of backup versions on server',
    )
    total_size = fields.Integer(
        string='Total Size (bytes)',
        readonly=True,
    )
    total_size_display = fields.Char(
        string='Total Size',
        compute='_compute_total_size_display',
    )

    @api.depends('remote_uuid')
    def _compute_is_synced(self):
        for record in self:
            record.is_synced = bool(record.remote_uuid)

    @api.depends('total_size')
    def _compute_total_size_display(self):
        for record in self:
            size = record.total_size
            if not size:
                record.total_size_display = '--'
            elif size < 1024:
                record.total_size_display = f'{size} B'
            elif size < 1024 * 1024:
                record.total_size_display = f'{size / 1024:.1f} KB'
            elif size < 1024 * 1024 * 1024:
                record.total_size_display = f'{size / (1024 * 1024):.1f} MB'
            else:
                record.total_size_display = f'{size / (1024 * 1024 * 1024):.2f} GB'

    def action_run_backup(self):
        """Run backup with this profile - executes directly without wizard"""
        self.ensure_one()

        if not self.config_id.api_key:
            raise UserError(_('Please configure API key first.'))

        # Validate at least one component
        if not any([self.include_database, self.include_filestore, self.include_modules]):
            raise UserError(_('Please select at least one component to backup in the profile.'))

        from .lox_api import OdooBackupCreator

        config = self.config_id
        api = self.env['lox.api'].create_client(config)
        backup_creator = OdooBackupCreator(self.env)
        backup_file = None

        # Determine component type for log
        if self.include_database and self.include_filestore:
            component = 'full'
        elif self.include_database:
            component = 'database'
        elif self.include_filestore:
            component = 'filestore'
        else:
            component = 'modules'

        # Create log entry
        log = self.env['lox.backup.log'].create({
            'config_id': config.id,
            'profile_id': self.id,
            'component': component,
            'status': 'in_progress',
        })

        try:
            # Create backup file based on profile settings
            if self.include_database and self.include_filestore:
                backup_file = backup_creator.create_full_backup()
            elif self.include_database:
                backup_file = backup_creator.create_database_backup()
            elif self.include_filestore:
                backup_file = backup_creator.create_filestore_backup()
            else:
                # Modules only - create a simple JSON file
                modules_info = self._get_modules_info()
                import json
                temp_dir = tempfile.mkdtemp()
                backup_file = os.path.join(temp_dir, f'{self.env.cr.dbname}_modules.json')
                with open(backup_file, 'w') as f:
                    json.dump(modules_info, f, indent=2)

            if not backup_file:
                raise UserError(_('Failed to create backup file.'))

            # Build tags
            tags = [config.default_tags] if config.default_tags else []
            if self.custom_tags:
                tags.append(self.custom_tags)
            tags.append('odoo')
            tags.append(f'profile:{self.name}')

            # Build metadata
            metadata = {
                'profile_id': self.id,
                'profile_name': self.name,
                'include_database': self.include_database,
                'include_filestore': self.include_filestore,
                'include_modules': self.include_modules,
            }

            if self.include_modules:
                metadata['modules'] = self._get_modules_info()

            # Upload options
            upload_options = {
                'name': f'{self.name} - {self.env.cr.dbname}',
                'tags': ','.join(tags),
                'retention_days': None if self.use_default_retention else self.retention_days,
                'immutable_days': None if self.use_default_immutable else self.immutable_days,
                'component': component,
                'metadata': metadata,
                'profile_uuid': self.remote_uuid,
            }

            # Upload to LOX
            result = api.upload_backup(backup_file, upload_options, self.env)

            if result.get('uuid') or result.get('id'):
                backup_uuid = result.get('uuid') or result.get('id')
                log.write({
                    'backup_uuid': backup_uuid,
                    'status': 'completed',
                    'size_bytes': os.path.getsize(backup_file) if backup_file else 0,
                })
                self._update_stats()

                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Backup Completed'),
                        'message': _('Backup uploaded successfully: %s') % backup_uuid,
                        'type': 'success',
                        'sticky': False,
                    }
                }
            else:
                error_msg = result.get('error', 'Unknown error')
                log.write({
                    'status': 'failed',
                    'error_message': error_msg,
                })
                raise UserError(_('Backup failed: %s') % error_msg)

        except Exception as e:
            _logger.exception('Backup failed for profile %s', self.name)
            log.write({
                'status': 'failed',
                'error_message': str(e),
            })
            raise UserError(_('Backup failed: %s') % str(e))

        finally:
            # Cleanup temp file
            if backup_file and os.path.exists(backup_file):
                try:
                    os.remove(backup_file)
                    # Also try to remove parent temp dir
                    parent_dir = os.path.dirname(backup_file)
                    if parent_dir and parent_dir.startswith(tempfile.gettempdir()):
                        os.rmdir(parent_dir)
                except:
                    pass

    def _get_modules_info(self):
        """Get list of installed modules"""
        modules = self.env['ir.module.module'].sudo().search([
            ('state', '=', 'installed')
        ])
        return [{
            'name': m.name,
            'version': m.installed_version or '',
            'author': m.author or '',
        } for m in modules]

    def _update_stats(self):
        """Update profile statistics after backup"""
        self.ensure_one()
        self.write({
            'last_run': fields.Datetime.now(),
            'run_count': self.run_count + 1,
        })

    def _run_scheduled_backup(self, schedule=None):
        """Run backup for scheduled execution (called by Schedule cron)

        Similar to action_run_backup but:
        - Doesn't return notification (for cron use)
        - Accepts schedule reference for logging
        - Raises exception on failure (for schedule to handle)
        """
        self.ensure_one()

        if not self.config_id.api_key:
            raise ValueError('API key not configured')

        if not any([self.include_database, self.include_filestore, self.include_modules]):
            raise ValueError('No backup components selected in profile')

        from .lox_api import OdooBackupCreator

        config = self.config_id
        api = self.env['lox.api'].create_client(config)
        backup_creator = OdooBackupCreator(self.env)
        backup_file = None

        # Determine component type
        if self.include_database and self.include_filestore:
            component = 'full'
        elif self.include_database:
            component = 'database'
        elif self.include_filestore:
            component = 'filestore'
        else:
            component = 'modules'

        # Create log entry
        log_vals = {
            'config_id': config.id,
            'profile_id': self.id,
            'component': component,
            'status': 'in_progress',
        }
        if schedule:
            log_vals['schedule_id'] = schedule.id
        log = self.env['lox.backup.log'].create(log_vals)

        try:
            # Create backup file
            if self.include_database and self.include_filestore:
                backup_file = backup_creator.create_full_backup()
            elif self.include_database:
                backup_file = backup_creator.create_database_backup()
            elif self.include_filestore:
                backup_file = backup_creator.create_filestore_backup()
            else:
                # Modules only
                modules_info = self._get_modules_info()
                import json
                temp_dir = tempfile.mkdtemp()
                backup_file = os.path.join(temp_dir, f'{self.env.cr.dbname}_modules.json')
                with open(backup_file, 'w') as f:
                    json.dump(modules_info, f, indent=2)

            if not backup_file:
                raise ValueError('Failed to create backup file')

            # Build tags
            tags = [config.default_tags] if config.default_tags else []
            if self.custom_tags:
                tags.append(self.custom_tags)
            tags.extend(['odoo', f'profile:{self.name}'])
            if schedule:
                tags.append('scheduled')

            # Build metadata
            metadata = {
                'profile_id': self.id,
                'profile_name': self.name,
                'include_database': self.include_database,
                'include_filestore': self.include_filestore,
                'include_modules': self.include_modules,
                'scheduled': bool(schedule),
            }
            if schedule:
                metadata['schedule_id'] = schedule.id
                metadata['schedule_name'] = schedule.name

            if self.include_modules:
                metadata['modules'] = self._get_modules_info()

            # Map schedule interval to frequency
            frequency = None
            if schedule:
                freq_map = {'hours': 'hourly', 'days': 'daily', 'weeks': 'weekly', 'months': 'monthly'}
                frequency = freq_map.get(schedule.interval_type)

            # Upload options
            upload_options = {
                'name': f'{self.name} - {self.env.cr.dbname}',
                'tags': ','.join(tags),
                'retention_days': None if self.use_default_retention else self.retention_days,
                'immutable_days': None if self.use_default_immutable else self.immutable_days,
                'component': component,
                'metadata': metadata,
                'profile_uuid': self.remote_uuid,
            }
            if frequency:
                upload_options['frequency'] = frequency

            # Upload to LOX
            result = api.upload_backup(backup_file, upload_options, self.env)

            if result.get('uuid') or result.get('id'):
                backup_uuid = result.get('uuid') or result.get('id')
                log.write({
                    'backup_uuid': backup_uuid,
                    'status': 'completed',
                    'size_bytes': os.path.getsize(backup_file) if backup_file else 0,
                })
                self._update_stats()
                return backup_uuid
            else:
                error_msg = result.get('error', 'Unknown error')
                log.write({
                    'status': 'failed',
                    'error_message': error_msg,
                })
                raise ValueError(f'Backup upload failed: {error_msg}')

        except Exception as e:
            log.write({
                'status': 'failed',
                'error_message': str(e),
            })
            raise

        finally:
            if backup_file and os.path.exists(backup_file):
                try:
                    os.remove(backup_file)
                    parent_dir = os.path.dirname(backup_file)
                    if parent_dir and parent_dir.startswith(tempfile.gettempdir()):
                        os.rmdir(parent_dir)
                except:
                    pass

    def action_sync_with_server(self):
        """Sync profile with LOX server - create or update remote profile"""
        self.ensure_one()

        if not self.config_id.api_key:
            raise UserError(_('Please configure API key first.'))

        api = self.env['lox.api'].create_client(self.config_id)

        if self.remote_uuid:
            # Update existing profile
            result = api.update_profile(self.remote_uuid, {
                'name': self.name,
                'description': self.description or '',
                'is_active': self.active,
            })
            if result.get('success') or result.get('uuid'):
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Profile Synced'),
                        'message': _('Profile updated on server.'),
                        'type': 'success',
                    }
                }
        else:
            # Create new profile
            result = api.create_profile(self.name, {
                'description': self.description or '',
            })
            if result.get('uuid'):
                self.write({'remote_uuid': result['uuid']})
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Profile Created'),
                        'message': _('Profile created on server: %s') % result['uuid'][:8],
                        'type': 'success',
                    }
                }

        raise UserError(_('Failed to sync profile: %s') % result.get('error', 'Unknown error'))

    def action_fetch_stats(self):
        """Fetch profile statistics from server"""
        self.ensure_one()

        if not self.remote_uuid:
            raise UserError(_('Profile is not synced with server.'))

        if not self.config_id.api_key:
            raise UserError(_('Please configure API key first.'))

        api = self.env['lox.api'].create_client(self.config_id)
        result = api.get_profile(self.remote_uuid)

        if result.get('uuid') or result.get('success'):
            data = result.get('data', result)
            self.write({
                'backup_count': data.get('backup_count', 0),
                'total_size': data.get('total_size_bytes', 0),
            })
            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': _('Stats Updated'),
                    'message': _('%d versions, %s total') % (
                        data.get('backup_count', 0),
                        self.total_size_display
                    ),
                    'type': 'success',
                }
            }

        raise UserError(_('Failed to fetch stats: %s') % result.get('error', 'Unknown error'))

    @api.model
    def action_sync_all_from_server(self):
        """Sync all profiles from server - import remote profiles as local"""
        config = self.env['lox.backup.config'].search([], limit=1)
        if not config or not config.api_key:
            raise UserError(_('Please configure API key first.'))

        api = self.env['lox.api'].create_client(config)
        result = api.get_profiles()

        if not (result.get('success') or result.get('profiles')):
            raise UserError(_('Failed to fetch profiles: %s') % result.get('error', 'Unknown error'))

        profiles = result.get('profiles', result.get('data', {}).get('profiles', []))
        created = 0
        updated = 0

        for profile_data in profiles:
            remote_uuid = profile_data.get('uuid')
            if not remote_uuid:
                continue

            # Check if profile already exists locally
            existing = self.search([('remote_uuid', '=', remote_uuid)], limit=1)
            if existing:
                existing.write({
                    'name': profile_data.get('name', existing.name),
                    'backup_count': profile_data.get('backup_count', 0),
                    'total_size': profile_data.get('total_size_bytes', 0),
                })
                updated += 1
            else:
                # Create new local profile
                self.create({
                    'name': profile_data.get('name', 'Imported Profile'),
                    'config_id': config.id,
                    'remote_uuid': remote_uuid,
                    'backup_count': profile_data.get('backup_count', 0),
                    'total_size': profile_data.get('total_size_bytes', 0),
                    'include_database': True,
                    'include_filestore': True,
                })
                created += 1

        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': _('Profiles Synced'),
                'message': _('Created: %d, Updated: %d') % (created, updated),
                'type': 'success',
            }
        }
