# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import UserError, ValidationError
import logging

_logger = logging.getLogger(__name__)


class LoxBackupSchedule(models.Model):
    """Schedule for automated backups - references a Profile for what to backup"""
    _name = 'lox.backup.schedule'
    _description = 'LOX Backup Schedule'
    _inherit = ['mail.thread']
    _rec_name = 'name'
    _order = 'sequence, id'

    name = fields.Char(
        string='Name',
        required=True,
        tracking=True,
    )
    config_id = fields.Many2one(
        'lox.backup.config',
        string='Configuration',
        required=True,
        ondelete='cascade',
    )
    company_id = fields.Many2one(
        related='config_id.company_id',
        store=True,
        readonly=True,
    )
    active = fields.Boolean(
        string='Active',
        default=True,
        tracking=True,
    )
    sequence = fields.Integer(
        string='Sequence',
        default=10,
    )

    # Profile to execute (defines WHAT to backup)
    profile_id = fields.Many2one(
        'lox.backup.profile',
        string='Backup Profile',
        required=True,
        ondelete='restrict',
        domain="[('config_id', '=', config_id)]",
        help='Profile that defines what to backup (components, retention, tags)',
    )

    # Schedule settings (defines WHEN to backup)
    interval_number = fields.Integer(
        string='Every',
        default=1,
        required=True,
    )
    interval_type = fields.Selection([
        ('hours', 'Hours'),
        ('days', 'Days'),
        ('weeks', 'Weeks'),
        ('months', 'Months'),
    ], string='Interval', default='days', required=True)

    # Cron job reference
    cron_id = fields.Many2one(
        'ir.cron',
        string='Scheduled Action',
        readonly=True,
        ondelete='set null',
    )

    # Stats
    last_run = fields.Datetime(
        string='Last Run',
        readonly=True,
    )
    next_run = fields.Datetime(
        string='Next Run',
        related='cron_id.nextcall',
        readonly=True,
    )
    run_count = fields.Integer(
        string='Run Count',
        default=0,
        readonly=True,
    )
    last_status = fields.Selection([
        ('success', 'Success'),
        ('failed', 'Failed'),
        ('running', 'Running'),
    ], string='Last Status', readonly=True)

    @api.model_create_multi
    def create(self, vals_list):
        records = super().create(vals_list)
        for record in records:
            if record.active:
                record._create_cron()
        return records

    def write(self, vals):
        result = super().write(vals)
        for record in self:
            if 'active' in vals:
                if vals['active']:
                    record._create_cron()
                else:
                    record._delete_cron()
            elif record.active and record.cron_id:
                record._update_cron()
        return result

    def unlink(self):
        for record in self:
            record._delete_cron()
        return super().unlink()

    def _create_cron(self):
        """Create ir.cron for this schedule"""
        self.ensure_one()
        if self.cron_id:
            return

        cron_vals = {
            'name': f'LOX Backup: {self.name}',
            'model_id': self.env['ir.model']._get('lox.backup.schedule').id,
            'state': 'code',
            'code': f'model.browse({self.id})._run_backup()',
            'interval_number': self.interval_number,
            'interval_type': self.interval_type,
            'active': True,
            'user_id': self.env.ref('base.user_root').id,
        }

        cron = self.env['ir.cron'].sudo().create(cron_vals)
        self.write({'cron_id': cron.id})
        _logger.info(f'Created cron job for schedule {self.name}')

    def _update_cron(self):
        """Update existing cron job"""
        self.ensure_one()
        if not self.cron_id:
            self._create_cron()
            return

        self.cron_id.sudo().write({
            'name': f'LOX Backup: {self.name}',
            'interval_number': self.interval_number,
            'interval_type': self.interval_type,
            'active': self.active,
        })

    def _delete_cron(self):
        """Delete associated cron job"""
        self.ensure_one()
        if self.cron_id:
            self.cron_id.sudo().unlink()
            _logger.info(f'Deleted cron job for schedule {self.name}')

    def _run_backup(self):
        """Execute scheduled backup using the associated profile"""
        self.ensure_one()
        _logger.info(f'Running scheduled backup: {self.name} with profile {self.profile_id.name}')

        self.write({
            'last_run': fields.Datetime.now(),
            'last_status': 'running',
            'run_count': self.run_count + 1,
        })

        try:
            # Delegate to profile's backup method
            self.profile_id._run_scheduled_backup(schedule=self)
            self.write({'last_status': 'success'})
            return True

        except Exception as e:
            _logger.exception(f'Scheduled backup failed: {self.name}')
            self.write({'last_status': 'failed'})
            return False

    def action_run_now(self):
        """Manually trigger this schedule"""
        self.ensure_one()
        self._run_backup()
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': _('Backup Started'),
                'message': _('Running backup with profile: %s') % self.profile_id.name,
                'type': 'info',
                'sticky': False,
            }
        }
