# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import UserError
import logging

_logger = logging.getLogger(__name__)


class ResConfigSettings(models.TransientModel):
    """
    LOX Backup settings in Odoo's General Settings.

    This is a proxy to lox.backup.config for the current company.
    All fields are related to the company's LOX config record.
    """
    _inherit = 'res.config.settings'

    # Reference to the actual config record
    lox_config_id = fields.Many2one(
        'lox.backup.config',
        string='LOX Configuration',
        compute='_compute_lox_config_id',
    )

    # Proxy fields - all related to lox.backup.config
    lox_api_url = fields.Char(
        string='LOX API URL',
        related='lox_config_id.api_url',
        readonly=False,
    )
    lox_api_key = fields.Char(
        string='LOX API Key',
        related='lox_config_id.api_key',
        readonly=False,
    )
    lox_source_identifier = fields.Char(
        string='Source Identifier',
        related='lox_config_id.source_identifier',
        readonly=True,
    )

    # Backup Components
    lox_backup_database = fields.Boolean(
        string='Backup Database',
        related='lox_config_id.backup_database',
        readonly=False,
    )
    lox_backup_filestore = fields.Boolean(
        string='Backup Filestore',
        related='lox_config_id.backup_filestore',
        readonly=False,
    )
    lox_backup_modules = fields.Boolean(
        string='Backup Modules Info',
        related='lox_config_id.backup_modules',
        readonly=False,
    )

    # Retention Settings
    lox_retention_days = fields.Integer(
        string='Retention Days',
        related='lox_config_id.retention_days',
        readonly=False,
    )
    lox_immutable_days = fields.Integer(
        string='Immutable Period (days)',
        related='lox_config_id.immutable_days',
        readonly=False,
    )

    # Tags
    lox_default_tags = fields.Char(
        string='Default Tags',
        related='lox_config_id.default_tags',
        readonly=False,
    )

    # Connection Status (from config)
    lox_connection_status = fields.Selection(
        related='lox_config_id.connection_status',
        readonly=True,
    )
    lox_last_connection_test = fields.Datetime(
        related='lox_config_id.last_connection_test',
        readonly=True,
    )

    # Stats
    lox_last_backup = fields.Datetime(
        string='Last Backup',
        related='lox_config_id.last_backup_date',
        readonly=True,
    )
    lox_total_backups = fields.Integer(
        string='Total Backups',
        related='lox_config_id.total_backups',
        readonly=True,
    )

    # Related records counts
    lox_schedule_count = fields.Integer(
        string='Schedules',
        compute='_compute_lox_counts',
    )
    lox_profile_count = fields.Integer(
        string='Profiles',
        compute='_compute_lox_counts',
    )

    @api.depends('company_id')
    def _compute_lox_config_id(self):
        """Get or create LOX config for the current company."""
        LoxConfig = self.env['lox.backup.config']
        for record in self:
            record.lox_config_id = LoxConfig.ensure_config_for_company()

    @api.depends('lox_config_id')
    def _compute_lox_counts(self):
        for record in self:
            if record.lox_config_id:
                record.lox_schedule_count = len(record.lox_config_id.schedule_ids)
                record.lox_profile_count = len(record.lox_config_id.profile_ids)
            else:
                record.lox_schedule_count = 0
                record.lox_profile_count = 0

    def action_lox_test_connection(self):
        """Test API connection."""
        self.ensure_one()
        if not self.lox_config_id:
            raise UserError(_('No LOX configuration found for this company.'))
        return self.lox_config_id.action_test_connection()

    def action_lox_run_backup(self):
        """Open backup wizard."""
        self.ensure_one()
        if not self.lox_config_id:
            raise UserError(_('No LOX configuration found for this company.'))
        return self.lox_config_id.action_run_backup()

    def action_lox_view_backups(self):
        """View all backup logs for current company."""
        self.ensure_one()
        if not self.lox_config_id:
            raise UserError(_('No LOX configuration found for this company.'))
        return self.lox_config_id.action_view_backups()

    def action_lox_view_schedules(self):
        """View all backup schedules for current company."""
        self.ensure_one()
        if not self.lox_config_id:
            raise UserError(_('No LOX configuration found for this company.'))
        return self.lox_config_id.action_view_schedules()

    def action_lox_view_profiles(self):
        """View all backup profiles for current company."""
        self.ensure_one()
        if not self.lox_config_id:
            raise UserError(_('No LOX configuration found for this company.'))
        return {
            'name': _('Backup Profiles'),
            'type': 'ir.actions.act_window',
            'res_model': 'lox.backup.profile',
            'view_mode': 'tree,form',
            'domain': [('config_id', '=', self.lox_config_id.id)],
            'context': {'default_config_id': self.lox_config_id.id},
        }

    def action_lox_open_config(self):
        """Open the full LOX configuration form."""
        self.ensure_one()
        if not self.lox_config_id:
            raise UserError(_('No LOX configuration found for this company.'))
        return {
            'name': _('LOX Backup Configuration'),
            'type': 'ir.actions.act_window',
            'res_model': 'lox.backup.config',
            'res_id': self.lox_config_id.id,
            'view_mode': 'form',
            'target': 'current',
        }
