# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import UserError
import logging
import tempfile
import os
import json

_logger = logging.getLogger(__name__)


class LoxBackupWizard(models.TransientModel):
    _name = 'lox.backup.wizard'
    _description = 'LOX Backup Wizard'

    config_id = fields.Many2one(
        'lox.backup.config',
        string='Configuration',
        required=True,
        default=lambda self: self._default_config_id(),
    )
    profile_id = fields.Many2one(
        'lox.backup.profile',
        string='Use Profile',
        domain="[('config_id', '=', config_id), ('active', '=', True)]",
        help='Select a saved profile or customize below',
    )

    # Backup type
    backup_mode = fields.Selection([
        ('full', 'Full Backup'),
        ('custom', 'Custom Selection'),
    ], string='Backup Mode', default='full', required=True)

    # What to include
    backup_database = fields.Boolean(
        string='Database',
        default=True,
        help='Full PostgreSQL database dump',
    )
    backup_filestore = fields.Boolean(
        string='Filestore',
        default=True,
        help='All attachments and uploaded files',
    )
    backup_modules = fields.Boolean(
        string='Modules Info',
        default=True,
        help='List of installed modules with versions',
    )

    # Custom settings
    custom_tags = fields.Char(
        string='Additional Tags',
        help='Comma-separated tags to add to this backup',
    )
    retention_days = fields.Integer(
        string='Retention Days',
        default=30,
        help='Days to keep this backup (0 = use default)',
    )
    immutable_days = fields.Integer(
        string='Immutable Period (days)',
        default=0,
        help='Days backup is protected from deletion (0 = use server default)',
    )
    backup_name = fields.Char(
        string='Backup Name',
        help='Optional custom name for this backup',
    )
    description = fields.Text(
        string='Description',
        help='Optional description for this backup',
    )

    # Save as profile
    save_as_profile = fields.Boolean(
        string='Save as Profile',
        default=False,
        help='Save current settings as a reusable profile',
    )
    profile_name = fields.Char(
        string='Profile Name',
    )

    @api.model
    def _default_config_id(self):
        config = self.env['lox.backup.config'].search([
            ('active', '=', True),
            ('api_key', '!=', False),
        ], limit=1)
        return config.id if config else False

    @api.onchange('profile_id')
    def _onchange_profile_id(self):
        """Load profile settings"""
        if self.profile_id:
            self.backup_mode = 'custom'
            self.backup_database = self.profile_id.include_database
            self.backup_filestore = self.profile_id.include_filestore
            self.backup_modules = self.profile_id.include_modules
            self.custom_tags = self.profile_id.custom_tags
            if self.profile_id.retention_days:
                self.retention_days = self.profile_id.retention_days

    @api.onchange('backup_mode')
    def _onchange_backup_mode(self):
        """Set defaults based on mode"""
        if self.backup_mode == 'full':
            self.backup_database = True
            self.backup_filestore = True
            self.backup_modules = True

    @api.onchange('config_id')
    def _onchange_config_id(self):
        """Load config defaults"""
        if self.config_id:
            self.retention_days = self.config_id.retention_days
            self.immutable_days = self.config_id.immutable_days

    def _get_modules_info(self):
        """Get installed modules information"""
        modules = self.env['ir.module.module'].search([
            ('state', '=', 'installed'),
        ])
        return [{
            'name': m.name,
            'version': m.installed_version or m.latest_version,
            'author': m.author,
            'summary': m.summary,
            'category': m.category_id.name if m.category_id else '',
        } for m in modules]

    def _build_tags(self):
        """Build tags list for the backup"""
        tags = ['odoo']

        # Add component tags
        if self.backup_database:
            tags.append('database')
        if self.backup_filestore:
            tags.append('filestore')
        if self.backup_modules:
            tags.append('modules')

        # Add mode tag
        if self.backup_mode == 'full':
            tags.append('full')
        else:
            tags.append('custom')

        # Add manual tag
        tags.append('manual')

        # Add config default tags
        if self.config_id.default_tags:
            for tag in self.config_id.default_tags.split(','):
                tag = tag.strip()
                if tag and tag not in tags:
                    tags.append(tag)

        # Add custom tags
        if self.custom_tags:
            for tag in self.custom_tags.split(','):
                tag = tag.strip()
                if tag and tag not in tags:
                    tags.append(tag)

        # Add profile tag if using profile
        if self.profile_id:
            tags.append(f'profile:{self.profile_id.name}')

        return tags

    def _save_profile(self):
        """Save current settings as a profile"""
        if self.save_as_profile and self.profile_name:
            self.env['lox.backup.profile'].create({
                'name': self.profile_name,
                'config_id': self.config_id.id,
                'include_database': self.backup_database,
                'include_filestore': self.backup_filestore,
                'include_modules': self.backup_modules,
                'custom_tags': self.custom_tags,
                'retention_days': self.retention_days if self.retention_days != self.config_id.retention_days else 0,
                'description': self.description,
            })

    def action_run_backup(self):
        """Execute manual backup"""
        self.ensure_one()

        config = self.config_id
        if not config.api_key:
            raise UserError(_('Please configure API key first.'))

        # Validate at least one component
        if not any([self.backup_database, self.backup_filestore, self.backup_modules]):
            raise UserError(_('Please select at least one component to backup.'))

        api = self.env['lox.api'].create_client(config)

        # Build components list
        components = []
        if self.backup_database:
            components.append('database')
        if self.backup_filestore:
            components.append('filestore')

        # Build tags
        tags = self._build_tags()

        # Get modules info if requested
        modules_info = None
        if self.backup_modules:
            modules_info = self._get_modules_info()

        # Determine component type for log
        if self.backup_database and self.backup_filestore:
            log_component = 'full'
        elif self.backup_database:
            log_component = 'database'
        elif self.backup_filestore:
            log_component = 'filestore'
        else:
            log_component = 'full'  # modules only

        try:
            # Create backup log entry
            log = self.env['lox.backup.log'].create({
                'config_id': config.id,
                'component': log_component,
                'status': 'pending',
            })

            # Build metadata
            metadata = {
                'manual_backup': True,
                'odoo_database': self.env.cr.dbname,
                'user': self.env.user.name,
                'backup_mode': self.backup_mode,
                'include_database': self.backup_database,
                'include_filestore': self.backup_filestore,
                'include_modules': self.backup_modules,
            }

            if modules_info:
                metadata['modules'] = modules_info
                metadata['modules_count'] = len(modules_info)

            if self.profile_id:
                metadata['profile_id'] = self.profile_id.id
                metadata['profile_name'] = self.profile_id.name

            if self.backup_name:
                metadata['custom_name'] = self.backup_name

            if self.description:
                metadata['description'] = self.description

            # Create backup via API
            backup_data = {
                'source_identifier': config.source_identifier,
                'components': components if components else ['database'],
                'tags': ','.join(tags),
                'retention_days': self.retention_days or config.retention_days,
                'immutable_days': self.immutable_days if self.immutable_days else None,
                'metadata': metadata,
            }

            if self.backup_name:
                backup_data['name'] = self.backup_name

            result = api.create_backup(backup_data)

            if result.get('id'):
                log.write({
                    'backup_uuid': result['id'],
                    'status': 'in_progress',
                    'started_at': fields.Datetime.now(),
                })

                # Save profile if requested
                self._save_profile()

                # Update profile stats if using one
                if self.profile_id:
                    self.profile_id._update_stats()

                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Backup Started'),
                        'message': _('Backup job created with UUID: %s') % result['id'],
                        'type': 'success',
                        'sticky': False,
                        'next': {
                            'type': 'ir.actions.act_window',
                            'res_model': 'lox.backup.log',
                            'res_id': log.id,
                            'view_mode': 'form',
                            'target': 'current',
                        }
                    }
                }
            else:
                log.write({
                    'status': 'failed',
                    'error_message': result.get('error', 'Unknown error'),
                })
                raise UserError(_('Backup failed: %s') % result.get('error', 'Unknown error'))

        except UserError:
            raise
        except Exception as e:
            _logger.exception('Manual backup failed')
            if 'log' in locals():
                log.write({
                    'status': 'failed',
                    'error_message': str(e),
                })
            raise UserError(_('Backup failed: %s') % str(e))

    def action_run_backup_with_upload(self):
        """Execute backup with local file creation and upload"""
        self.ensure_one()

        config = self.config_id
        if not config.api_key:
            raise UserError(_('Please configure API key first.'))

        # Validate at least one component
        if not any([self.backup_database, self.backup_filestore, self.backup_modules]):
            raise UserError(_('Please select at least one component to backup.'))

        from ..models.lox_api import OdooBackupCreator

        api = self.env['lox.api'].create_client(config)
        backup_creator = OdooBackupCreator(self.env)

        # Build tags
        tags = self._build_tags()

        # Get modules info if requested
        modules_info = None
        if self.backup_modules:
            modules_info = self._get_modules_info()

        # Determine component type for log
        if self.backup_database and self.backup_filestore:
            log_component = 'full'
        elif self.backup_database:
            log_component = 'database'
        elif self.backup_filestore:
            log_component = 'filestore'
        else:
            log_component = 'full'

        try:
            # Create backup log entry
            log = self.env['lox.backup.log'].create({
                'config_id': config.id,
                'component': log_component,
                'status': 'pending',
                'started_at': fields.Datetime.now(),
            })

            # Create local backup based on selections
            backup_file = None
            if self.backup_database and self.backup_filestore:
                backup_file = backup_creator.create_full_backup()
            elif self.backup_database:
                backup_file = backup_creator.create_database_backup()
            elif self.backup_filestore:
                backup_file = backup_creator.create_filestore_backup()
            else:
                # Modules only - create a JSON file
                backup_file = self._create_modules_only_backup(modules_info)

            if not backup_file:
                raise UserError(_('Failed to create local backup.'))

            # If we have modules info and backup_file is a tarball, inject it
            if modules_info and self.backup_database:
                self._inject_modules_info(backup_file, modules_info)

            # Get file size
            file_size = os.path.getsize(backup_file)
            log.write({'size_bytes': file_size})

            # Build metadata
            metadata = {
                'manual_backup': True,
                'with_upload': True,
                'odoo_database': self.env.cr.dbname,
                'user': self.env.user.name,
                'backup_mode': self.backup_mode,
                'include_database': self.backup_database,
                'include_filestore': self.backup_filestore,
                'include_modules': self.backup_modules,
            }

            if modules_info:
                metadata['modules_count'] = len(modules_info)

            if self.profile_id:
                metadata['profile_id'] = self.profile_id.id
                metadata['profile_name'] = self.profile_id.name

            # Build components list
            components = []
            if self.backup_database:
                components.append('database')
            if self.backup_filestore:
                components.append('filestore')
            if not components:
                components = ['modules']

            # Build upload options for single-step upload
            upload_options = {
                'tags': tags,
                'retention_days': None,  # Use frequency-based default
                'immutable_days': self.immutable_days if self.immutable_days else None,
                'frequency': 'daily',  # Manual backups use daily retention policy
                'component': ','.join(components),
                'metadata': metadata,
                'description': f'Odoo backup from {self.env.cr.dbname}',
            }

            if self.backup_name:
                upload_options['name'] = self.backup_name

            # Upload backup (single-step: creates record and uploads file)
            log.write({'status': 'in_progress'})
            upload_result = api.upload_backup(backup_file, upload_options)

            # Check for upload errors
            if upload_result.get('error') or upload_result.get('success') is False:
                raise UserError(_('Upload failed: %s') % upload_result.get('error', 'Unknown'))

            # Get backup UUID from response
            backup_uuid = upload_result.get('id') or upload_result.get('uuid')
            if backup_uuid:
                log.write({'backup_uuid': backup_uuid})

            # Cleanup temp file
            try:
                os.remove(backup_file)
                temp_dir = os.path.dirname(backup_file)
                if not os.listdir(temp_dir):
                    os.rmdir(temp_dir)
            except:
                pass

            # Upload succeeded (errors already handled above)
            log.write({'status': 'validating'})

            # Save profile if requested
            self._save_profile()

            # Update profile stats if using one
            if self.profile_id:
                self.profile_id._update_stats()

            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': _('Backup Uploaded'),
                    'message': _('Backup uploaded successfully. UUID: %s') % backup_uuid,
                    'type': 'success',
                    'sticky': False,
                    'next': {
                        'type': 'ir.actions.act_window',
                        'res_model': 'lox.backup.log',
                        'res_id': log.id,
                        'view_mode': 'form',
                        'target': 'current',
                    }
                }
            }

        except UserError:
            raise
        except Exception as e:
            _logger.exception('Backup with upload failed')
            if 'log' in locals():
                log.write({
                    'status': 'failed',
                    'error_message': str(e),
                })
            raise UserError(_('Backup failed: %s') % str(e))

    def _create_modules_only_backup(self, modules_info):
        """Create a backup containing only modules information"""
        import tarfile
        import gzip

        temp_dir = tempfile.mkdtemp()
        modules_file = os.path.join(temp_dir, 'modules.json')
        backup_file = os.path.join(temp_dir, f'{self.env.cr.dbname}_modules.tar.gz')

        # Write modules info
        with open(modules_file, 'w') as f:
            json.dump({
                'database': self.env.cr.dbname,
                'timestamp': fields.Datetime.now().isoformat(),
                'modules': modules_info,
            }, f, indent=2)

        # Create tarball
        with tarfile.open(backup_file, 'w:gz') as tar:
            tar.add(modules_file, arcname='modules.json')

        os.remove(modules_file)
        return backup_file

    def _inject_modules_info(self, backup_file, modules_info):
        """Inject modules.json into existing backup archive"""
        # For now, we'll include it in metadata
        # In a full implementation, we would repack the tarball
        pass
