# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import UserError, ValidationError
import logging

_logger = logging.getLogger(__name__)


class LoxRestoreWizard(models.TransientModel):
    _name = 'lox.restore.wizard'
    _description = 'LOX Backup Restore Wizard'

    CONFIRMATION_TEXT = 'RESTORE'

    config_id = fields.Many2one(
        'lox.backup.config',
        string='Configuration',
        required=True,
        default=lambda self: self._default_config_id(),
    )
    backup_uuid = fields.Char(
        string='Backup UUID',
        required=True,
        readonly=True,
    )
    backup_name = fields.Char(
        string='Backup Name',
        readonly=True,
    )
    backup_date = fields.Datetime(
        string='Backup Date',
        readonly=True,
    )
    backup_size = fields.Char(
        string='Backup Size',
        readonly=True,
    )
    backup_component = fields.Char(
        string='Component',
        readonly=True,
    )
    backup_status = fields.Char(
        string='Status',
        readonly=True,
    )

    # Confirmation field
    confirmation_text = fields.Char(
        string='Confirmation',
        required=True,
        help='Type RESTORE to confirm the restoration',
    )

    # Display fields
    warning_message = fields.Html(
        string='Warning',
        compute='_compute_warning_message',
    )

    @api.model
    def _default_config_id(self):
        config = self.env['lox.backup.config'].search([
            ('active', '=', True),
            ('api_key', '!=', False),
        ], limit=1)
        return config.id if config else False

    @api.depends('backup_uuid')
    def _compute_warning_message(self):
        for record in self:
            record.warning_message = '''
                <div style="background-color: #fef2f2; border: 1px solid #fecaca; border-radius: 8px; padding: 16px; margin-bottom: 16px;">
                    <p style="font-weight: bold; color: #991b1b; margin: 0 0 8px 0;">
                        ⚠️ Warning: This action is irreversible!
                    </p>
                    <p style="color: #991b1b; margin: 0;">
                        This will overwrite your current Odoo data. All current database records,
                        filestore attachments, and configurations will be replaced with the backup data.
                        This action cannot be undone.
                    </p>
                </div>
            '''

    @api.constrains('confirmation_text')
    def _check_confirmation_text(self):
        for record in self:
            if record.confirmation_text and record.confirmation_text.upper().strip() != self.CONFIRMATION_TEXT:
                raise ValidationError(
                    _('Please type %s to confirm the restore.') % self.CONFIRMATION_TEXT
                )

    @api.model
    def create_for_backup(self, backup_uuid, backup_info=None):
        """Create a wizard pre-populated with backup info"""
        values = {
            'backup_uuid': backup_uuid,
        }

        if backup_info:
            values['backup_name'] = backup_info.get('name', backup_uuid[:8] + '...')
            values['backup_date'] = backup_info.get('created_at')
            values['backup_component'] = backup_info.get('component', 'full')
            values['backup_status'] = backup_info.get('status', 'unknown')

            # Format size
            size_bytes = backup_info.get('size', backup_info.get('size_bytes', 0))
            if size_bytes:
                if size_bytes > 1024 * 1024 * 1024:
                    values['backup_size'] = f"{size_bytes / (1024*1024*1024):.2f} GB"
                elif size_bytes > 1024 * 1024:
                    values['backup_size'] = f"{size_bytes / (1024*1024):.2f} MB"
                elif size_bytes > 1024:
                    values['backup_size'] = f"{size_bytes / 1024:.2f} KB"
                else:
                    values['backup_size'] = f"{size_bytes} bytes"

        return self.create(values)

    def action_confirm_restore(self):
        """Request restore after confirmation"""
        self.ensure_one()

        # Validate confirmation text
        if self.confirmation_text.upper().strip() != self.CONFIRMATION_TEXT:
            raise UserError(
                _('Please type %s to confirm the restore.') % self.CONFIRMATION_TEXT
            )

        config = self.config_id
        if not config.api_key:
            raise UserError(_('Please configure API key first.'))

        api = self.env['lox.api'].create_client(config)

        try:
            # Request restore from LOX API
            result = api.request_restore(self.backup_uuid)

            if result.get('success') or result.get('task_id'):
                # Log the restore request
                self.env['lox.backup.log'].create({
                    'config_id': config.id,
                    'backup_uuid': self.backup_uuid,
                    'component': 'restore',
                    'status': 'pending',
                    'started_at': fields.Datetime.now(),
                    'notes': _('Restore requested for backup %s') % self.backup_uuid,
                })

                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Restore Requested'),
                        'message': _(
                            'Restore has been requested for backup %s. '
                            'You will be notified when the backup is ready for download and restoration.'
                        ) % (self.backup_uuid[:8] + '...'),
                        'type': 'success',
                        'sticky': True,
                        'next': {'type': 'ir.actions.act_window_close'},
                    }
                }
            else:
                raise UserError(
                    _('Failed to request restore: %s') % result.get('error', 'Unknown error')
                )

        except UserError:
            raise
        except Exception as e:
            _logger.exception('Restore request failed')
            raise UserError(_('Restore request failed: %s') % str(e))

    def action_cancel(self):
        """Cancel the wizard"""
        return {'type': 'ir.actions.act_window_close'}


class LoxBackupLogRestore(models.Model):
    """Extend backup log to add restore action"""
    _inherit = 'lox.backup.log'

    def action_request_restore(self):
        """Open restore confirmation wizard"""
        self.ensure_one()

        if not self.backup_uuid:
            raise UserError(_('No backup UUID available for this log entry.'))

        if self.status not in ['completed', 'validated']:
            raise UserError(_('Can only restore completed backups.'))

        # Get backup info from API
        api = self.env['lox.api'].create_client(self.config_id)
        backup_info = api.get_backup(self.backup_uuid)

        wizard = self.env['lox.restore.wizard'].create_for_backup(
            self.backup_uuid,
            backup_info.get('data', {}) if backup_info.get('success') else None
        )

        return {
            'name': _('Confirm Restore'),
            'type': 'ir.actions.act_window',
            'res_model': 'lox.restore.wizard',
            'res_id': wizard.id,
            'view_mode': 'form',
            'target': 'new',
            'context': {'dialog_size': 'medium'},
        }
