<?php
/**
 * LOX Backup Manager for PrestaShop
 *
 * @author    LOX Backup <support@backlox.com>
 * @copyright 2024 LOX Backup
 * @license   MIT
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

class LoxBackupManager
{
    /**
     * @var string Backup directory
     */
    private $backupDir;

    /**
     * @var LoxApi API client
     */
    private $api;

    /**
     * @var string Source identifier
     */
    protected $source = 'prestashop';

    /**
     * @var array Component definitions
     */
    protected $components = array(
        'catalog' => array(
            'name' => 'Catalog',
            'description' => 'Products, categories, manufacturers, suppliers, attributes, and features',
            'tables' => array(
                'product',
                'product_lang',
                'product_shop',
                'product_attribute',
                'product_attribute_combination',
                'product_attribute_image',
                'product_attribute_shop',
                'category',
                'category_lang',
                'category_shop',
                'category_product',
                'manufacturer',
                'manufacturer_lang',
                'manufacturer_shop',
                'supplier',
                'supplier_lang',
                'supplier_shop',
                'feature',
                'feature_lang',
                'feature_product',
                'feature_shop',
                'feature_value',
                'feature_value_lang',
                'attribute',
                'attribute_lang',
                'attribute_group',
                'attribute_group_lang',
                'attribute_group_shop',
                'attribute_shop',
                'stock_available',
                'specific_price',
                'specific_price_rule',
                'product_download',
                'tag',
                'product_tag',
            ),
        ),
        'transactional' => array(
            'name' => 'Orders & Customers',
            'description' => 'Orders, customers, carts, and payment data',
            'tables' => array(
                'orders',
                'order_detail',
                'order_detail_tax',
                'order_history',
                'order_invoice',
                'order_invoice_payment',
                'order_invoice_tax',
                'order_payment',
                'order_return',
                'order_return_detail',
                'order_slip',
                'order_slip_detail',
                'order_message',
                'customer',
                'customer_group',
                'customer_message',
                'customer_thread',
                'address',
                'cart',
                'cart_product',
                'cart_rule',
                'cart_rule_lang',
                'cart_rule_shop',
                'cart_rule_combination',
                'cart_rule_product_rule',
                'cart_rule_product_rule_group',
                'cart_rule_product_rule_value',
            ),
        ),
        'files' => array(
            'name' => 'Media Files',
            'description' => 'Product images, category images, and downloads',
            'paths' => array(
                'img/p',
                'img/c',
                'img/m',
                'img/su',
                'img/st',
                'upload',
                'download',
            ),
        ),
        'config' => array(
            'name' => 'Configuration',
            'description' => 'Store settings, modules, and themes',
            'tables' => array(
                'configuration',
                'configuration_lang',
                'shop',
                'shop_group',
                'shop_url',
                'lang',
                'lang_shop',
                'currency',
                'currency_shop',
                'country',
                'country_lang',
                'country_shop',
                'state',
                'zone',
                'zone_shop',
                'tax',
                'tax_lang',
                'tax_rule',
                'tax_rules_group',
                'tax_rules_group_shop',
                'carrier',
                'carrier_lang',
                'carrier_shop',
                'carrier_zone',
                'delivery',
                'range_price',
                'range_weight',
                'module',
                'module_shop',
                'hook',
                'hook_module',
                'hook_module_exceptions',
                'meta',
                'meta_lang',
                'cms',
                'cms_lang',
                'cms_shop',
                'cms_category',
                'cms_category_lang',
                'cms_category_shop',
            ),
            'paths' => array(
                'modules',
                'themes',
            ),
            'files' => array(
                'config/settings.inc.php',
            ),
        ),
    );

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->backupDir = _PS_CACHE_DIR_ . 'lox_backups/';
        $this->api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        if (!is_dir($this->backupDir)) {
            mkdir($this->backupDir, 0755, true);
            file_put_contents($this->backupDir . 'index.php', '<?php // Silence is golden');
            file_put_contents($this->backupDir . '.htaccess', 'deny from all');
        }
    }

    /**
     * Run full backup
     *
     * @return array
     */
    public function runBackup()
    {
        $timestamp = date('Ymd_His');
        $shopName = Tools::str2url(Configuration::get('PS_SHOP_NAME'));
        $backupName = "prestashop-{$shopName}-{$timestamp}";

        $this->log('Starting backup: ' . $backupName);

        try {
            // Create temp directory
            $tempDir = $this->backupDir . 'temp_' . $timestamp . '/';
            mkdir($tempDir, 0755, true);

            $filesToBackup = array();

            // Backup database
            if (Configuration::get('LOX_BACKUP_DATABASE')) {
                $dbFile = $this->backupDatabase($tempDir);
                if ($dbFile) {
                    $filesToBackup[] = $dbFile;
                    $this->log('Database backup created');
                }
            }

            // Backup files (upload, download)
            if (Configuration::get('LOX_BACKUP_FILES')) {
                $this->backupDirectory(_PS_UPLOAD_DIR_, $tempDir, 'upload');
                $this->backupDirectory(_PS_DOWNLOAD_DIR_, $tempDir, 'download');
                $this->log('Files backup created');
            }

            // Backup images
            if (Configuration::get('LOX_BACKUP_IMAGES')) {
                $this->backupDirectory(_PS_IMG_DIR_, $tempDir, 'img');
                $this->log('Images backup created');
            }

            // Backup modules
            if (Configuration::get('LOX_BACKUP_MODULES')) {
                $this->backupDirectory(_PS_MODULE_DIR_, $tempDir, 'modules');
                $this->log('Modules backup created');
            }

            // Backup themes
            if (Configuration::get('LOX_BACKUP_THEMES')) {
                $this->backupDirectory(_PS_THEME_DIR_, $tempDir, 'themes');
                $this->log('Themes backup created');
            }

            // Export store data to JSON
            $this->exportStoreData($tempDir);

            // Create archive
            $archivePath = $this->backupDir . $backupName . '.tar.gz';
            $this->createArchive($tempDir, $archivePath);
            $this->log('Archive created: ' . $this->formatBytes(filesize($archivePath)));

            // Upload to LOX
            $this->log('Uploading to LOX...');
            $immutableDays = Configuration::get('LOX_BACKUP_IMMUTABLE_DAYS');
            $schedule = Configuration::get('LOX_BACKUP_SCHEDULE') ?: 'daily';
            $result = $this->api->uploadBackup($archivePath, array(
                'name' => $backupName,
                'tags' => Configuration::get('LOX_BACKUP_TAGS') ?: 'prestashop,automated',
                'retention_days' => null,  // Use frequency-based default
                'immutable_days' => $immutableDays === '' ? null : (int) $immutableDays,
                'frequency' => $schedule !== 'disabled' ? $schedule : 'daily',
                'description' => sprintf(
                    'PrestaShop backup from %s (%s)',
                    Tools::getShopDomain(),
                    Configuration::get('PS_SHOP_NAME')
                ),
            ));

            if (!$result['success']) {
                $this->cleanup($tempDir, $archivePath);
                $this->updateStatus('failed', $result['error']);
                return $result;
            }

            $this->log('Upload complete, waiting for processing...');

            // Wait for completion
            $backup = $this->api->waitForCompletion($result['data']['uuid'], 3600);

            // Cleanup
            $this->cleanup($tempDir, $archivePath);

            if (!$backup['success']) {
                $this->updateStatus('failed', $backup['error']);
                return $backup;
            }

            $this->log(sprintf('Backup completed. UUID: %s', $backup['data']['uuid']));
            $this->updateStatus('completed');

            return array(
                'success' => true,
                'uuid' => $backup['data']['uuid'],
                'size_bytes' => $backup['data']['size_bytes'],
            );

        } catch (Exception $e) {
            $this->log('Backup failed: ' . $e->getMessage(), 'error');
            $this->updateStatus('failed', $e->getMessage());
            return array('success' => false, 'error' => $e->getMessage());
        }
    }

    /**
     * Backup database
     *
     * @param string $tempDir
     * @return string|null
     */
    private function backupDatabase($tempDir)
    {
        $sqlFile = $tempDir . 'database.sql';
        $handle = fopen($sqlFile, 'w');

        if (!$handle) {
            return null;
        }

        // Write header
        fwrite($handle, "-- LOX PrestaShop Backup\n");
        fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
        fwrite($handle, "-- Shop: " . Tools::getShopDomain() . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\n");
        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

        // Get tables
        $tables = Db::getInstance()->executeS('SHOW TABLES');
        $prefix = _DB_PREFIX_;

        foreach ($tables as $table) {
            $tableName = current($table);

            // Only backup PrestaShop tables
            if (strpos($tableName, $prefix) !== 0) {
                continue;
            }

            // Table structure
            $create = Db::getInstance()->executeS("SHOW CREATE TABLE `{$tableName}`");
            fwrite($handle, "DROP TABLE IF EXISTS `{$tableName}`;\n");
            fwrite($handle, $create[0]['Create Table'] . ";\n\n");

            // Table data
            $rows = Db::getInstance()->executeS("SELECT * FROM `{$tableName}`");

            if (!empty($rows)) {
                $columns = array_keys($rows[0]);
                $columnList = '`' . implode('`, `', $columns) . '`';

                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function ($v) {
                            return $v === null ? 'NULL' : "'" . pSQL($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$tableName}` ({$columnList}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
                fwrite($handle, "\n");
            }
        }

        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);

        return $sqlFile;
    }

    /**
     * Backup a directory
     *
     * @param string $source Source directory
     * @param string $tempDir Temp directory
     * @param string $name Directory name in archive
     */
    private function backupDirectory($source, $tempDir, $name)
    {
        if (!is_dir($source)) {
            return;
        }

        $destDir = $tempDir . $name . '/';
        $this->recursiveCopy($source, $destDir);
    }

    /**
     * Recursively copy directory
     *
     * @param string $source
     * @param string $dest
     */
    private function recursiveCopy($source, $dest)
    {
        if (!is_dir($dest)) {
            mkdir($dest, 0755, true);
        }

        $dir = opendir($source);
        while (($file = readdir($dir)) !== false) {
            if ($file === '.' || $file === '..') {
                continue;
            }

            $srcPath = $source . '/' . $file;
            $destPath = $dest . '/' . $file;

            if (is_dir($srcPath)) {
                $this->recursiveCopy($srcPath, $destPath);
            } else {
                copy($srcPath, $destPath);
            }
        }
        closedir($dir);
    }

    /**
     * Export store data to JSON
     *
     * @param string $tempDir
     */
    private function exportStoreData($tempDir)
    {
        $dataDir = $tempDir . 'store_data/';
        mkdir($dataDir, 0755, true);

        // Export configuration
        $config = Configuration::getMultiple(array(
            'PS_SHOP_NAME',
            'PS_SHOP_EMAIL',
            'PS_SHOP_ADDR1',
            'PS_SHOP_ADDR2',
            'PS_SHOP_CODE',
            'PS_SHOP_CITY',
            'PS_SHOP_COUNTRY',
            'PS_SHOP_STATE',
            'PS_SHOP_PHONE',
            'PS_CURRENCY_DEFAULT',
            'PS_LANG_DEFAULT',
            'PS_WEIGHT_UNIT',
            'PS_DIMENSION_UNIT',
            'PS_STOCK_MANAGEMENT',
            'PS_ORDER_OUT_OF_STOCK',
            'PS_DISPLAY_QTIES',
            'PS_TAX',
            'PS_TAX_DISPLAY',
            'PS_SHIPPING_FREE_PRICE',
            'PS_SHIPPING_FREE_WEIGHT',
        ));
        file_put_contents($dataDir . 'configuration.json', json_encode($config, JSON_PRETTY_PRINT));

        // Export categories
        $categories = Category::getCategories(Context::getContext()->language->id, false, false);
        file_put_contents($dataDir . 'categories.json', json_encode($categories, JSON_PRETTY_PRINT));

        // Export carriers
        $carriers = Carrier::getCarriers(Context::getContext()->language->id, false, false, false, null, Carrier::ALL_CARRIERS);
        file_put_contents($dataDir . 'carriers.json', json_encode($carriers, JSON_PRETTY_PRINT));

        // Export zones
        $zones = Zone::getZones(true);
        file_put_contents($dataDir . 'zones.json', json_encode($zones, JSON_PRETTY_PRINT));

        // Export tax rules
        $taxRules = TaxRulesGroup::getTaxRulesGroups(true);
        file_put_contents($dataDir . 'tax_rules.json', json_encode($taxRules, JSON_PRETTY_PRINT));

        // Export order states
        $orderStates = OrderState::getOrderStates(Context::getContext()->language->id);
        file_put_contents($dataDir . 'order_states.json', json_encode($orderStates, JSON_PRETTY_PRINT));

        // Export payment modules
        $paymentModules = PaymentModule::getInstalledPaymentModules();
        file_put_contents($dataDir . 'payment_modules.json', json_encode($paymentModules, JSON_PRETTY_PRINT));

        $this->log('Store data exported');
    }

    /**
     * Create tar.gz archive
     *
     * @param string $sourceDir
     * @param string $archivePath
     */
    private function createArchive($sourceDir, $archivePath)
    {
        $tarPath = str_replace('.tar.gz', '.tar', $archivePath);

        // Create tar
        $phar = new PharData($tarPath);
        $phar->buildFromDirectory($sourceDir);

        // Compress to gzip
        $phar->compress(Phar::GZ);

        // Remove uncompressed tar
        if (file_exists($tarPath)) {
            unlink($tarPath);
        }
    }

    /**
     * Cleanup temporary files
     *
     * @param string $tempDir
     * @param string $archive
     */
    private function cleanup($tempDir, $archive = null)
    {
        if (is_dir($tempDir)) {
            $this->recursiveDelete($tempDir);
        }

        if ($archive && file_exists($archive)) {
            unlink($archive);
        }
    }

    /**
     * Recursively delete directory
     *
     * @param string $dir
     */
    private function recursiveDelete($dir)
    {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object !== '.' && $object !== '..') {
                    if (is_dir($dir . '/' . $object)) {
                        $this->recursiveDelete($dir . '/' . $object);
                    } else {
                        unlink($dir . '/' . $object);
                    }
                }
            }
            rmdir($dir);
        }
    }

    /**
     * Update backup status
     *
     * @param string $status
     * @param string|null $message
     */
    private function updateStatus($status, $message = null)
    {
        Configuration::updateValue('LOX_BACKUP_LAST_RUN', date('Y-m-d H:i:s'));
        Configuration::updateValue('LOX_BACKUP_LAST_STATUS', $status);

        if ($message) {
            Configuration::updateValue('LOX_BACKUP_LAST_MESSAGE', $message);
        }
    }

    /**
     * Log message
     *
     * @param string $message
     * @param string $level
     */
    private function log($message, $level = 'info')
    {
        if (_PS_MODE_DEV_) {
            PrestaShopLogger::addLog(
                '[LOX Backup] ' . $message,
                $level === 'error' ? 3 : 1,
                null,
                'LoxBackup'
            );
        }
    }

    /**
     * Format bytes to human readable
     *
     * @param int $bytes
     * @return string
     */
    private function formatBytes($bytes)
    {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);

        return round($bytes, 2) . ' ' . $units[$pow];
    }

    /**
     * Get component definitions
     *
     * @return array
     */
    public function getComponents()
    {
        return $this->components;
    }

    /**
     * Get source identifier
     *
     * @return string
     */
    public function getSource()
    {
        return $this->source;
    }

    /**
     * Run component-specific backup
     *
     * @param string $component Component name
     * @return array
     */
    public function runComponentBackup($component)
    {
        if (!isset($this->components[$component])) {
            return array('success' => false, 'error' => 'Unknown component: ' . $component);
        }

        $componentDef = $this->components[$component];
        $timestamp = date('Ymd_His');
        $shopName = Tools::str2url(Configuration::get('PS_SHOP_NAME'));
        $backupName = "prestashop-{$shopName}-{$component}-{$timestamp}";

        $this->log('Starting ' . $component . ' component backup: ' . $backupName);

        try {
            // Create temp directory
            $tempDir = $this->backupDir . 'temp_' . $component . '_' . $timestamp . '/';
            mkdir($tempDir, 0755, true);

            // Backup tables for this component
            if (!empty($componentDef['tables'])) {
                $this->backupComponentTables($tempDir, $componentDef['tables']);
            }

            // Backup paths for this component
            if (!empty($componentDef['paths'])) {
                $this->backupComponentPaths($tempDir, $componentDef['paths']);
            }

            // Backup specific files
            if (!empty($componentDef['files'])) {
                $this->backupComponentFiles($tempDir, $componentDef['files']);
            }

            // Export component-specific JSON data
            $this->exportComponentData($tempDir, $component);

            // Create archive
            $archivePath = $this->backupDir . $backupName . '.tar.gz';
            $this->createArchive($tempDir, $archivePath);
            $this->log('Archive created: ' . $this->formatBytes(filesize($archivePath)));

            // Upload to LOX
            $this->log('Uploading ' . $component . ' backup to LOX...');
            $immutableDays = Configuration::get('LOX_BACKUP_IMMUTABLE_DAYS');
            // Get frequency from component schedules
            $schedules = Configuration::get('LOX_BACKUP_COMPONENT_SCHEDULES');
            $schedules = $schedules ? json_decode($schedules, true) : array();
            $frequency = isset($schedules[$component]) ? $schedules[$component] : 'daily';
            $result = $this->api->uploadBackup($archivePath, array(
                'name' => $backupName,
                'tags' => Configuration::get('LOX_BACKUP_TAGS') ?: 'prestashop,automated,' . $component,
                'retention_days' => null,  // Use frequency-based default
                'immutable_days' => $immutableDays === '' ? null : (int) $immutableDays,
                'frequency' => $frequency !== 'disabled' ? $frequency : 'daily',
                'description' => sprintf(
                    'PrestaShop %s backup from %s (%s)',
                    $componentDef['name'],
                    Tools::getShopDomain(),
                    Configuration::get('PS_SHOP_NAME')
                ),
                'source' => $this->source,
                'component' => $component,
            ));

            if (!$result['success']) {
                $this->cleanup($tempDir, $archivePath);
                $this->updateComponentStatus($component, 'failed', $result['error']);
                return $result;
            }

            $this->log('Upload complete, waiting for processing...');

            // Wait for completion
            $backup = $this->api->waitForCompletion($result['data']['uuid'], 3600);

            // Cleanup
            $this->cleanup($tempDir, $archivePath);

            if (!$backup['success']) {
                $this->updateComponentStatus($component, 'failed', $backup['error']);
                return $backup;
            }

            $this->log(sprintf('%s backup completed. UUID: %s', $componentDef['name'], $backup['data']['uuid']));
            $this->updateComponentStatus($component, 'completed');

            return array(
                'success' => true,
                'uuid' => $backup['data']['uuid'],
                'size_bytes' => $backup['data']['size_bytes'],
                'component' => $component,
            );

        } catch (Exception $e) {
            $this->log('Component backup failed: ' . $e->getMessage(), 'error');
            $this->updateComponentStatus($component, 'failed', $e->getMessage());
            return array('success' => false, 'error' => $e->getMessage());
        }
    }

    /**
     * Backup specific tables for a component
     *
     * @param string $tempDir
     * @param array $tables
     */
    private function backupComponentTables($tempDir, $tables)
    {
        $sqlFile = $tempDir . 'database.sql';
        $handle = fopen($sqlFile, 'w');

        if (!$handle) {
            return;
        }

        $prefix = _DB_PREFIX_;

        // Write header
        fwrite($handle, "-- LOX PrestaShop Component Backup\n");
        fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
        fwrite($handle, "-- Shop: " . Tools::getShopDomain() . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\n");
        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

        foreach ($tables as $table) {
            $tableName = $prefix . $table;

            // Check if table exists
            $exists = Db::getInstance()->executeS("SHOW TABLES LIKE '{$tableName}'");
            if (empty($exists)) {
                continue;
            }

            // Table structure
            $create = Db::getInstance()->executeS("SHOW CREATE TABLE `{$tableName}`");
            if (empty($create)) {
                continue;
            }

            fwrite($handle, "DROP TABLE IF EXISTS `{$tableName}`;\n");
            fwrite($handle, $create[0]['Create Table'] . ";\n\n");

            // Table data
            $rows = Db::getInstance()->executeS("SELECT * FROM `{$tableName}`");

            if (!empty($rows)) {
                $columns = array_keys($rows[0]);
                $columnList = '`' . implode('`, `', $columns) . '`';

                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function ($v) {
                            return $v === null ? 'NULL' : "'" . pSQL($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$tableName}` ({$columnList}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
                fwrite($handle, "\n");
            }
        }

        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);

        $this->log('Backed up ' . count($tables) . ' tables');
    }

    /**
     * Backup specific paths for a component
     *
     * @param string $tempDir
     * @param array $paths
     */
    private function backupComponentPaths($tempDir, $paths)
    {
        foreach ($paths as $path) {
            $sourcePath = _PS_ROOT_DIR_ . '/' . $path;
            if (is_dir($sourcePath)) {
                $this->backupDirectory($sourcePath, $tempDir, basename($path));
            }
        }
    }

    /**
     * Backup specific files for a component
     *
     * @param string $tempDir
     * @param array $files
     */
    private function backupComponentFiles($tempDir, $files)
    {
        $filesDir = $tempDir . 'files/';
        if (!is_dir($filesDir)) {
            mkdir($filesDir, 0755, true);
        }

        foreach ($files as $file) {
            $sourcePath = _PS_ROOT_DIR_ . '/' . $file;
            if (file_exists($sourcePath)) {
                $destPath = $filesDir . basename($file);
                copy($sourcePath, $destPath);
            }
        }
    }

    /**
     * Export component-specific data to JSON
     *
     * @param string $tempDir
     * @param string $component
     */
    private function exportComponentData($tempDir, $component)
    {
        $dataDir = $tempDir . 'data/';
        if (!is_dir($dataDir)) {
            mkdir($dataDir, 0755, true);
        }

        $langId = Context::getContext()->language->id;

        switch ($component) {
            case 'catalog':
                // Export product summary
                $products = Product::getProducts($langId, 0, 0, 'id_product', 'ASC');
                file_put_contents($dataDir . 'products_summary.json', json_encode(array(
                    'count' => count($products),
                    'exported_at' => date('Y-m-d H:i:s'),
                ), JSON_PRETTY_PRINT));

                // Export categories
                $categories = Category::getCategories($langId, false, false);
                file_put_contents($dataDir . 'categories.json', json_encode($categories, JSON_PRETTY_PRINT));

                // Export manufacturers
                $manufacturers = Manufacturer::getManufacturers(false, $langId, true, false, false, false, true);
                file_put_contents($dataDir . 'manufacturers.json', json_encode($manufacturers, JSON_PRETTY_PRINT));

                // Export suppliers
                $suppliers = Supplier::getSuppliers(false, $langId, true, false, false, false);
                file_put_contents($dataDir . 'suppliers.json', json_encode($suppliers, JSON_PRETTY_PRINT));
                break;

            case 'transactional':
                // Export order statistics
                $orderStates = OrderState::getOrderStates($langId);
                file_put_contents($dataDir . 'order_states.json', json_encode($orderStates, JSON_PRETTY_PRINT));

                // Export recent orders summary (last 30 days)
                $dateFrom = date('Y-m-d', strtotime('-30 days'));
                $orders = Order::getOrdersWithInformations();
                file_put_contents($dataDir . 'orders_summary.json', json_encode(array(
                    'total_orders' => count($orders),
                    'exported_at' => date('Y-m-d H:i:s'),
                ), JSON_PRETTY_PRINT));

                // Export customer count
                $customerCount = Customer::getCustomers();
                file_put_contents($dataDir . 'customers_summary.json', json_encode(array(
                    'total_customers' => count($customerCount),
                    'exported_at' => date('Y-m-d H:i:s'),
                ), JSON_PRETTY_PRINT));
                break;

            case 'config':
                // Export configuration
                $config = Configuration::getMultiple(array(
                    'PS_SHOP_NAME',
                    'PS_SHOP_EMAIL',
                    'PS_SHOP_ADDR1',
                    'PS_SHOP_ADDR2',
                    'PS_SHOP_CODE',
                    'PS_SHOP_CITY',
                    'PS_SHOP_COUNTRY',
                    'PS_SHOP_STATE',
                    'PS_SHOP_PHONE',
                    'PS_CURRENCY_DEFAULT',
                    'PS_LANG_DEFAULT',
                    'PS_WEIGHT_UNIT',
                    'PS_DIMENSION_UNIT',
                    'PS_STOCK_MANAGEMENT',
                    'PS_ORDER_OUT_OF_STOCK',
                    'PS_DISPLAY_QTIES',
                    'PS_TAX',
                    'PS_TAX_DISPLAY',
                    'PS_SHIPPING_FREE_PRICE',
                    'PS_SHIPPING_FREE_WEIGHT',
                ));
                file_put_contents($dataDir . 'configuration.json', json_encode($config, JSON_PRETTY_PRINT));

                // Export carriers
                $carriers = Carrier::getCarriers($langId, false, false, false, null, Carrier::ALL_CARRIERS);
                file_put_contents($dataDir . 'carriers.json', json_encode($carriers, JSON_PRETTY_PRINT));

                // Export zones
                $zones = Zone::getZones(true);
                file_put_contents($dataDir . 'zones.json', json_encode($zones, JSON_PRETTY_PRINT));

                // Export tax rules
                $taxRules = TaxRulesGroup::getTaxRulesGroups(true);
                file_put_contents($dataDir . 'tax_rules.json', json_encode($taxRules, JSON_PRETTY_PRINT));

                // Export payment modules
                $paymentModules = PaymentModule::getInstalledPaymentModules();
                file_put_contents($dataDir . 'payment_modules.json', json_encode($paymentModules, JSON_PRETTY_PRINT));

                // Export installed modules
                $modules = Module::getModulesInstalled();
                file_put_contents($dataDir . 'installed_modules.json', json_encode($modules, JSON_PRETTY_PRINT));
                break;

            case 'files':
                // Export file statistics
                $stats = array(
                    'product_images' => $this->countFilesInDir(_PS_PROD_IMG_DIR_),
                    'category_images' => $this->countFilesInDir(_PS_CAT_IMG_DIR_),
                    'manufacturer_images' => $this->countFilesInDir(_PS_MANU_IMG_DIR_),
                    'supplier_images' => $this->countFilesInDir(_PS_SUPP_IMG_DIR_),
                    'store_images' => $this->countFilesInDir(_PS_STORE_IMG_DIR_),
                    'exported_at' => date('Y-m-d H:i:s'),
                );
                file_put_contents($dataDir . 'files_summary.json', json_encode($stats, JSON_PRETTY_PRINT));
                break;
        }

        $this->log('Exported ' . $component . ' data');
    }

    /**
     * Count files in directory
     *
     * @param string $dir
     * @return int
     */
    private function countFilesInDir($dir)
    {
        if (!is_dir($dir)) {
            return 0;
        }

        $count = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Update component backup status
     *
     * @param string $component
     * @param string $status
     * @param string|null $message
     */
    private function updateComponentStatus($component, $status, $message = null)
    {
        $statuses = Configuration::get('LOX_BACKUP_COMPONENT_STATUS');
        $statuses = $statuses ? json_decode($statuses, true) : array();

        $statuses[$component] = array(
            'status' => $status,
            'timestamp' => date('Y-m-d H:i:s'),
            'message' => $message,
        );

        Configuration::updateValue('LOX_BACKUP_COMPONENT_STATUS', json_encode($statuses));
    }

    /**
     * Get component backup status
     *
     * @param string|null $component
     * @return array
     */
    public function getComponentStatus($component = null)
    {
        $statuses = Configuration::get('LOX_BACKUP_COMPONENT_STATUS');
        $statuses = $statuses ? json_decode($statuses, true) : array();

        if ($component) {
            return isset($statuses[$component]) ? $statuses[$component] : null;
        }

        return $statuses;
    }

    /**
     * Run a saved backup profile (remote profile with UUID)
     *
     * @param string $profileUuid Remote profile UUID
     * @param string $profileName Profile name for labeling
     * @param array $elements Elements to backup
     * @param string $frequency Backup frequency (daily, weekly, monthly)
     * @return array
     */
    public function runProfileBackup($profileUuid, $profileName, $elements, $frequency = 'daily')
    {
        if (empty($elements)) {
            return array('success' => false, 'error' => 'No backup elements specified');
        }

        $timestamp = date('Ymd_His');
        $shopName = Tools::str2url(Configuration::get('PS_SHOP_NAME'));
        $backupName = "prestashop-{$shopName}-" . Tools::str2url($profileName) . "-{$timestamp}";

        $this->log('Starting profile backup: ' . $backupName . ' [Profile: ' . $profileUuid . ']');

        try {
            // Create temp directory
            $tempDir = $this->backupDir . 'temp_profile_' . $timestamp . '/';
            mkdir($tempDir, 0755, true);

            $backedUpElements = array();

            // Backup database if selected
            if (in_array('database', $elements)) {
                $dbFile = $this->backupDatabase($tempDir);
                if ($dbFile) {
                    $backedUpElements[] = 'database';
                    $this->log('Database backup created');
                }
            }

            // Backup files if selected
            if (in_array('files', $elements)) {
                $this->backupDirectory(_PS_UPLOAD_DIR_, $tempDir, 'upload');
                $this->backupDirectory(_PS_DOWNLOAD_DIR_, $tempDir, 'download');
                $backedUpElements[] = 'files';
                $this->log('Files backup created');
            }

            // Backup images if selected
            if (in_array('images', $elements)) {
                $this->backupDirectory(_PS_IMG_DIR_, $tempDir, 'img');
                $backedUpElements[] = 'images';
                $this->log('Images backup created');
            }

            // Backup modules if selected
            if (in_array('modules', $elements)) {
                $this->backupDirectory(_PS_MODULE_DIR_, $tempDir, 'modules');
                $backedUpElements[] = 'modules';
                $this->log('Modules backup created');
            }

            // Backup themes if selected
            if (in_array('themes', $elements)) {
                $this->backupDirectory(_PS_THEME_DIR_, $tempDir, 'themes');
                $backedUpElements[] = 'themes';
                $this->log('Themes backup created');
            }

            // Backup config if selected
            if (in_array('config', $elements)) {
                $configDir = $tempDir . 'config/';
                mkdir($configDir, 0755, true);
                if (file_exists(_PS_ROOT_DIR_ . '/config/settings.inc.php')) {
                    copy(_PS_ROOT_DIR_ . '/config/settings.inc.php', $configDir . 'settings.inc.php');
                }
                $backedUpElements[] = 'config';
                $this->log('Config backup created');
            }

            if (empty($backedUpElements)) {
                $this->cleanup($tempDir);
                return array('success' => false, 'error' => 'No files were backed up');
            }

            // Create archive
            $archivePath = $this->backupDir . $backupName . '.tar.gz';
            $this->createArchive($tempDir, $archivePath);
            $this->log('Archive created: ' . $this->formatBytes(filesize($archivePath)));

            // Build tags
            $tags = array('prestashop', 'profile');
            $tags = array_merge($tags, $backedUpElements);
            $tags[] = 'profile:' . Tools::str2url($profileName);

            // Upload to LOX with profile_uuid
            $this->log('Uploading profile backup to LOX...');
            $immutableDays = Configuration::get('LOX_BACKUP_IMMUTABLE_DAYS');
            $result = $this->api->uploadBackup($archivePath, array(
                'name' => $backupName,
                'tags' => implode(',', $tags),
                'retention_days' => null,  // Use frequency-based default
                'immutable_days' => $immutableDays === '' ? null : (int) $immutableDays,
                'frequency' => $frequency,
                'description' => sprintf(
                    'PrestaShop profile backup (%s) from %s (%s) - Profile: %s',
                    implode(', ', $backedUpElements),
                    Tools::getShopDomain(),
                    Configuration::get('PS_SHOP_NAME'),
                    $profileName
                ),
                'source' => $this->source,
                'component' => 'profile',
                'profile_uuid' => $profileUuid,
            ));

            if (!$result['success']) {
                $this->cleanup($tempDir, $archivePath);
                $this->updateStatus('failed', $result['error']);
                return $result;
            }

            $this->log('Upload complete, waiting for processing...');

            // Wait for completion
            $backup = $this->api->waitForCompletion($result['data']['uuid'], 3600);

            // Cleanup
            $this->cleanup($tempDir, $archivePath);

            if (!$backup['success']) {
                $this->updateStatus('failed', $backup['error']);
                return $backup;
            }

            $this->log(sprintf('Profile backup completed. UUID: %s', $backup['data']['uuid']));
            $this->updateStatus('completed');

            return array(
                'success' => true,
                'uuid' => $backup['data']['uuid'],
                'size_bytes' => $backup['data']['size_bytes'],
                'profile_uuid' => $profileUuid,
            );

        } catch (Exception $e) {
            $this->log('Profile backup failed: ' . $e->getMessage(), 'error');
            $this->updateStatus('failed', $e->getMessage());
            return array('success' => false, 'error' => $e->getMessage());
        }
    }

    /**
     * Run custom backup with selected elements
     *
     * @param array $elements Array of elements to backup
     * @param string $profileName Optional profile name
     * @param string $frequency Backup frequency (daily, weekly, monthly)
     * @return array
     */
    public function runCustomBackup($elements, $profileName = '', $frequency = 'daily')
    {
        if (empty($elements)) {
            return array('success' => false, 'error' => 'No backup elements selected');
        }

        $timestamp = date('Ymd_His');
        $shopName = Tools::str2url(Configuration::get('PS_SHOP_NAME'));
        $elementsSlug = implode('-', array_map('Tools::str2url', $elements));

        if ($profileName) {
            $backupName = "prestashop-{$shopName}-" . Tools::str2url($profileName) . "-{$timestamp}";
        } else {
            $backupName = "prestashop-{$shopName}-custom-{$elementsSlug}-{$timestamp}";
        }

        $this->log('Starting custom backup: ' . $backupName . ' [' . implode(', ', $elements) . ']');

        try {
            // Create temp directory
            $tempDir = $this->backupDir . 'temp_custom_' . $timestamp . '/';
            mkdir($tempDir, 0755, true);

            $backedUpElements = array();

            // Backup database if selected
            if (in_array('database', $elements)) {
                $dbFile = $this->backupDatabase($tempDir);
                if ($dbFile) {
                    $backedUpElements[] = 'database';
                    $this->log('Database backup created');
                }
            }

            // Backup files if selected
            if (in_array('files', $elements)) {
                $this->backupDirectory(_PS_UPLOAD_DIR_, $tempDir, 'upload');
                $this->backupDirectory(_PS_DOWNLOAD_DIR_, $tempDir, 'download');
                $backedUpElements[] = 'files';
                $this->log('Files backup created');
            }

            // Backup images if selected
            if (in_array('images', $elements)) {
                $this->backupDirectory(_PS_IMG_DIR_, $tempDir, 'img');
                $backedUpElements[] = 'images';
                $this->log('Images backup created');
            }

            // Backup modules if selected
            if (in_array('modules', $elements)) {
                $this->backupDirectory(_PS_MODULE_DIR_, $tempDir, 'modules');
                $backedUpElements[] = 'modules';
                $this->log('Modules backup created');
            }

            // Backup themes if selected
            if (in_array('themes', $elements)) {
                $this->backupDirectory(_PS_THEME_DIR_, $tempDir, 'themes');
                $backedUpElements[] = 'themes';
                $this->log('Themes backup created');
            }

            // Backup config if selected
            if (in_array('config', $elements)) {
                $configDir = $tempDir . 'config/';
                mkdir($configDir, 0755, true);
                if (file_exists(_PS_ROOT_DIR_ . '/config/settings.inc.php')) {
                    copy(_PS_ROOT_DIR_ . '/config/settings.inc.php', $configDir . 'settings.inc.php');
                }
                $backedUpElements[] = 'config';
                $this->log('Config backup created');
            }

            if (empty($backedUpElements)) {
                $this->cleanup($tempDir);
                return array('success' => false, 'error' => 'No files were backed up');
            }

            // Create archive
            $archivePath = $this->backupDir . $backupName . '.tar.gz';
            $this->createArchive($tempDir, $archivePath);
            $this->log('Archive created: ' . $this->formatBytes(filesize($archivePath)));

            // Build tags
            $tags = array('prestashop', 'custom');
            $tags = array_merge($tags, $backedUpElements);
            if ($profileName) {
                $tags[] = 'profile:' . Tools::str2url($profileName);
            }

            // Upload to LOX
            $this->log('Uploading custom backup to LOX...');
            $immutableDays = Configuration::get('LOX_BACKUP_IMMUTABLE_DAYS');
            $result = $this->api->uploadBackup($archivePath, array(
                'name' => $backupName,
                'tags' => implode(',', $tags),
                'retention_days' => null,  // Use frequency-based default
                'immutable_days' => $immutableDays === '' ? null : (int) $immutableDays,
                'frequency' => $frequency,
                'description' => sprintf(
                    'PrestaShop custom backup (%s) from %s (%s)%s',
                    implode(', ', $backedUpElements),
                    Tools::getShopDomain(),
                    Configuration::get('PS_SHOP_NAME'),
                    $profileName ? " - Profile: {$profileName}" : ''
                ),
                'source' => $this->source,
                'component' => 'custom',
            ));

            if (!$result['success']) {
                $this->cleanup($tempDir, $archivePath);
                $this->updateStatus('failed', $result['error']);
                return $result;
            }

            $this->log('Upload complete, waiting for processing...');

            // Wait for completion
            $backup = $this->api->waitForCompletion($result['data']['uuid'], 3600);

            // Cleanup
            $this->cleanup($tempDir, $archivePath);

            if (!$backup['success']) {
                $this->updateStatus('failed', $backup['error']);
                return $backup;
            }

            $this->log(sprintf('Custom backup completed. UUID: %s', $backup['data']['uuid']));
            $this->updateStatus('completed');

            return array(
                'success' => true,
                'uuid' => $backup['data']['uuid'],
                'size_bytes' => $backup['data']['size_bytes'],
            );

        } catch (Exception $e) {
            $this->log('Custom backup failed: ' . $e->getMessage(), 'error');
            $this->updateStatus('failed', $e->getMessage());
            return array('success' => false, 'error' => $e->getMessage());
        }
    }
}
