<?php
/**
 * LOX Backup PULL API Controller
 *
 * Exposes secure endpoints for LOX to initiate backups remotely.
 * This enables the PULL model where LOX controls the backup schedule.
 *
 * @author    LOX Backup <support@backlox.com>
 * @copyright 2024 LOX Backup
 * @license   MIT
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once dirname(__FILE__) . '/../../classes/LoxBackupManager.php';

class LoxBackupPullModuleFrontController extends ModuleFrontController
{
    public $ssl = true;
    public $ajax = true;
    public $auth = false;
    public $guestAllowed = true;

    /**
     * Process request
     */
    public function initContent()
    {
        parent::initContent();

        $action = Tools::getValue('action', 'backup');

        switch ($action) {
            case 'backup':
                $this->processBackup();
                break;
            case 'status':
                $this->processStatus();
                break;
            case 'download':
                $this->processDownload();
                break;
            case 'refresh-token':
                $this->processRefreshToken();
                break;
            default:
                $this->jsonResponse(['error' => 'invalid_action'], 400);
        }
    }

    /**
     * Verify PULL token
     */
    protected function verifyPullToken()
    {
        $token = $this->getHeader('X-LOX-Pull-Token');

        if (empty($token)) {
            return ['error' => 'missing_token', 'message' => 'Missing X-LOX-Pull-Token header'];
        }

        $storedToken = Configuration::get('LOX_BACKUP_PULL_TOKEN');
        $tokenExpiry = Configuration::get('LOX_BACKUP_PULL_TOKEN_EXPIRY');

        if (empty($storedToken)) {
            return ['error' => 'not_configured', 'message' => 'PULL token not configured'];
        }

        if ($tokenExpiry && time() > (int) $tokenExpiry) {
            return ['error' => 'token_expired', 'message' => 'PULL token has expired'];
        }

        if (!hash_equals($storedToken, $token)) {
            return ['error' => 'invalid_token', 'message' => 'Invalid PULL token'];
        }

        return null;
    }

    /**
     * Verify API key
     */
    protected function verifyApiKey()
    {
        $key = $this->getHeader('X-API-Key');

        if (empty($key)) {
            return ['error' => 'missing_api_key', 'message' => 'Missing X-API-Key header'];
        }

        $storedKey = Configuration::get('LOX_BACKUP_API_KEY');

        if (empty($storedKey) || !hash_equals($storedKey, $key)) {
            return ['error' => 'invalid_api_key', 'message' => 'Invalid API key'];
        }

        return null;
    }

    /**
     * Get header value
     */
    protected function getHeader($name)
    {
        $key = 'HTTP_' . str_replace('-', '_', strtoupper($name));
        return isset($_SERVER[$key]) ? $_SERVER[$key] : null;
    }

    /**
     * Process backup request
     */
    protected function processBackup()
    {
        if ($error = $this->verifyPullToken()) {
            $this->jsonResponse($error, 401);
            return;
        }

        $input = json_decode(file_get_contents('php://input'), true) ?: [];
        $type = isset($input['type']) ? $input['type'] : 'full';
        $frequency = isset($input['frequency']) ? $input['frequency'] : 'daily';

        try {
            $result = $this->runBackupForPull($type, $input, $frequency);

            if (isset($result['error'])) {
                $this->jsonResponse($result, 400);
                return;
            }

            $downloadToken = $this->createDownloadToken($result['archive_path']);

            $this->jsonResponse([
                'success' => true,
                'backup_name' => $result['name'],
                'size_bytes' => filesize($result['archive_path']),
                'checksum' => hash_file('sha256', $result['archive_path']),
                'download_url' => $this->context->link->getModuleLink('loxbackup', 'pull', ['action' => 'download', 'token' => $downloadToken]),
                'expires_in' => 3600,
            ]);
        } catch (Exception $e) {
            $this->jsonResponse(['error' => 'backup_failed', 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Run backup for PULL mode
     */
    protected function runBackupForPull($type, $options, $frequency)
    {
        $timestamp = date('Ymd_His');
        $shopName = Tools::str2url(Configuration::get('PS_SHOP_NAME'));
        $backupDir = _PS_CACHE_DIR_ . 'lox_backups/';

        if (!is_dir($backupDir)) {
            mkdir($backupDir, 0755, true);
        }

        $tempDir = $backupDir . 'temp_pull_' . $timestamp . '/';
        mkdir($tempDir, 0755, true);

        $backupName = "prestashop-{$shopName}";

        try {
            switch ($type) {
                case 'full':
                    $backupName .= "-full-{$timestamp}";
                    $this->collectFullBackup($tempDir);
                    break;

                case 'component':
                    $component = isset($options['component']) ? $options['component'] : null;
                    if (!$component) {
                        $this->cleanupTemp($tempDir);
                        return ['error' => 'missing_component', 'message' => 'Component required'];
                    }
                    $backupName .= "-{$component}-{$timestamp}";
                    $this->collectComponentBackup($tempDir, $component);
                    break;

                case 'custom':
                    $elements = isset($options['elements']) ? $options['elements'] : [];
                    if (empty($elements)) {
                        $this->cleanupTemp($tempDir);
                        return ['error' => 'missing_elements', 'message' => 'Elements required'];
                    }
                    $slug = implode('-', array_map(function ($e) {
                        return preg_replace('/[^a-z0-9]/', '', strtolower($e));
                    }, $elements));
                    $backupName .= "-custom-{$slug}-{$timestamp}";
                    $this->collectCustomBackup($tempDir, $elements);
                    break;

                default:
                    $this->cleanupTemp($tempDir);
                    return ['error' => 'invalid_type', 'message' => 'Invalid backup type'];
            }

            // Check if we have files
            $files = scandir($tempDir);
            if (count($files) <= 2) {
                $this->cleanupTemp($tempDir);
                return ['error' => 'backup_empty', 'message' => 'No files to backup'];
            }

            // Create archive
            $archivePath = $backupDir . $backupName . '.tar.gz';
            $this->createArchive($tempDir, $archivePath);
            $this->cleanupTemp($tempDir);

            return [
                'name' => $backupName,
                'archive_path' => $archivePath,
                'type' => $type,
            ];
        } catch (Exception $e) {
            $this->cleanupTemp($tempDir);
            throw $e;
        }
    }

    /**
     * Collect full backup
     */
    protected function collectFullBackup($tempDir)
    {
        if (Configuration::get('LOX_BACKUP_DATABASE')) {
            $this->backupDatabase($tempDir);
        }
        if (Configuration::get('LOX_BACKUP_FILES')) {
            $this->recursiveCopy(_PS_UPLOAD_DIR_, $tempDir . 'upload/');
            $this->recursiveCopy(_PS_DOWNLOAD_DIR_, $tempDir . 'download/');
        }
        if (Configuration::get('LOX_BACKUP_IMAGES')) {
            $this->recursiveCopy(_PS_IMG_DIR_, $tempDir . 'img/');
        }
        if (Configuration::get('LOX_BACKUP_MODULES')) {
            $this->recursiveCopy(_PS_MODULE_DIR_, $tempDir . 'modules/');
        }
        if (Configuration::get('LOX_BACKUP_THEMES')) {
            $this->recursiveCopy(_PS_ALL_THEMES_DIR_, $tempDir . 'themes/');
        }
    }

    /**
     * Collect component backup
     */
    protected function collectComponentBackup($tempDir, $component)
    {
        $manager = new LoxBackupManager();
        $components = $manager->getComponents();

        if (!isset($components[$component])) {
            throw new Exception('Invalid component: ' . $component);
        }

        $def = $components[$component];

        if (!empty($def['tables'])) {
            $this->backupComponentTables($tempDir, $def['tables']);
        }

        if (!empty($def['paths'])) {
            foreach ($def['paths'] as $path) {
                $fullPath = _PS_ROOT_DIR_ . '/' . $path;
                if (is_dir($fullPath)) {
                    $this->recursiveCopy($fullPath, $tempDir . basename($path) . '/');
                }
            }
        }
    }

    /**
     * Collect custom backup
     */
    protected function collectCustomBackup($tempDir, $elements)
    {
        if (in_array('database', $elements)) {
            $this->backupDatabase($tempDir);
        }
        if (in_array('files', $elements)) {
            $this->recursiveCopy(_PS_UPLOAD_DIR_, $tempDir . 'upload/');
            $this->recursiveCopy(_PS_DOWNLOAD_DIR_, $tempDir . 'download/');
        }
        if (in_array('images', $elements)) {
            $this->recursiveCopy(_PS_IMG_DIR_, $tempDir . 'img/');
        }
        if (in_array('modules', $elements)) {
            $this->recursiveCopy(_PS_MODULE_DIR_, $tempDir . 'modules/');
        }
        if (in_array('themes', $elements)) {
            $this->recursiveCopy(_PS_ALL_THEMES_DIR_, $tempDir . 'themes/');
        }
    }

    /**
     * Backup database
     */
    protected function backupDatabase($tempDir)
    {
        $sqlFile = $tempDir . 'database.sql';
        $handle = fopen($sqlFile, 'w');

        if (!$handle) {
            throw new Exception('Cannot create database file');
        }

        fwrite($handle, "-- LOX PrestaShop PULL Backup\n");
        fwrite($handle, "-- " . date('Y-m-d H:i:s') . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\nSET FOREIGN_KEY_CHECKS = 0;\n\n");

        $tables = Db::getInstance()->executeS('SHOW TABLES');
        $prefix = _DB_PREFIX_;

        foreach ($tables as $table) {
            $tableName = current($table);

            if ($prefix && strpos($tableName, $prefix) !== 0) {
                continue;
            }

            $create = Db::getInstance()->executeS("SHOW CREATE TABLE `{$tableName}`");
            fwrite($handle, "DROP TABLE IF EXISTS `{$tableName}`;\n");
            fwrite($handle, $create[0]['Create Table'] . ";\n\n");

            $rows = Db::getInstance()->executeS("SELECT * FROM `{$tableName}`");
            if (!empty($rows)) {
                $columns = '`' . implode('`, `', array_keys($rows[0])) . '`';
                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = [];
                    foreach ($chunk as $row) {
                        $escaped = array_map(function ($v) {
                            return $v === null ? 'NULL' : "'" . pSQL($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$tableName}` ({$columns}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
            }
        }

        fwrite($handle, "\nSET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);
    }

    /**
     * Backup component tables
     */
    protected function backupComponentTables($tempDir, $patterns)
    {
        $sqlFile = $tempDir . 'database.sql';
        $handle = fopen($sqlFile, 'w');

        if (!$handle) {
            throw new Exception('Cannot create database file');
        }

        fwrite($handle, "-- LOX Component Backup\n\nSET NAMES utf8mb4;\nSET FOREIGN_KEY_CHECKS = 0;\n\n");

        $allTables = Db::getInstance()->executeS('SHOW TABLES');
        $prefix = _DB_PREFIX_;

        $tables = [];
        foreach ($allTables as $table) {
            $tableName = current($table);
            $unprefixed = $prefix ? preg_replace('/^' . preg_quote($prefix, '/') . '/', '', $tableName) : $tableName;

            foreach ($patterns as $pattern) {
                $regex = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/';
                if (preg_match($regex, $unprefixed)) {
                    $tables[] = $tableName;
                    break;
                }
            }
        }

        foreach ($tables as $tableName) {
            $create = Db::getInstance()->executeS("SHOW CREATE TABLE `{$tableName}`");
            if (empty($create)) {
                continue;
            }

            fwrite($handle, "DROP TABLE IF EXISTS `{$tableName}`;\n");
            fwrite($handle, $create[0]['Create Table'] . ";\n\n");

            $rows = Db::getInstance()->executeS("SELECT * FROM `{$tableName}`");
            if (!empty($rows)) {
                $columns = '`' . implode('`, `', array_keys($rows[0])) . '`';
                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = [];
                    foreach ($chunk as $row) {
                        $escaped = array_map(function ($v) {
                            return $v === null ? 'NULL' : "'" . pSQL($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$tableName}` ({$columns}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
            }
        }

        fwrite($handle, "\nSET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);
    }

    /**
     * Recursive copy
     */
    protected function recursiveCopy($src, $dst)
    {
        if (!is_dir($src)) {
            return;
        }
        if (!is_dir($dst)) {
            mkdir($dst, 0755, true);
        }

        $dir = opendir($src);
        while (($file = readdir($dir)) !== false) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            $srcPath = $src . '/' . $file;
            $dstPath = $dst . '/' . $file;
            if (is_dir($srcPath)) {
                $this->recursiveCopy($srcPath, $dstPath);
            } else {
                copy($srcPath, $dstPath);
            }
        }
        closedir($dir);
    }

    /**
     * Create archive
     */
    protected function createArchive($srcDir, $archivePath)
    {
        $tarPath = str_replace('.tar.gz', '.tar', $archivePath);

        foreach ([$archivePath, $tarPath] as $path) {
            if (file_exists($path)) {
                @unlink($path);
            }
        }

        $phar = new PharData($tarPath);
        $phar->buildFromDirectory($srcDir);
        $phar->compress(Phar::GZ);
        unset($phar);

        if (file_exists($tarPath)) {
            @unlink($tarPath);
        }
    }

    /**
     * Cleanup temp directory
     */
    protected function cleanupTemp($dir)
    {
        if (!is_dir($dir)) {
            return;
        }
        $files = scandir($dir);
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            $path = $dir . $file;
            if (is_dir($path)) {
                $this->cleanupTemp($path . '/');
            } else {
                unlink($path);
            }
        }
        rmdir($dir);
    }

    /**
     * Create download token
     */
    protected function createDownloadToken($archivePath)
    {
        $token = bin2hex(random_bytes(16));
        $downloads = json_decode(Configuration::get('LOX_BACKUP_PULL_DOWNLOADS') ?: '{}', true);
        $downloads[$token] = [
            'path' => $archivePath,
            'expiry' => time() + 3600,
        ];
        Configuration::updateValue('LOX_BACKUP_PULL_DOWNLOADS', json_encode($downloads));
        return $token;
    }

    /**
     * Process download
     */
    protected function processDownload()
    {
        $token = Tools::getValue('token');
        $downloads = json_decode(Configuration::get('LOX_BACKUP_PULL_DOWNLOADS') ?: '{}', true);

        if (!isset($downloads[$token])) {
            $this->jsonResponse(['error' => 'invalid_token'], 404);
            return;
        }

        $download = $downloads[$token];

        if (time() > $download['expiry'] || !file_exists($download['path'])) {
            unset($downloads[$token]);
            Configuration::updateValue('LOX_BACKUP_PULL_DOWNLOADS', json_encode($downloads));
            $this->jsonResponse(['error' => 'expired'], 410);
            return;
        }

        // Remove token
        unset($downloads[$token]);
        Configuration::updateValue('LOX_BACKUP_PULL_DOWNLOADS', json_encode($downloads));

        // Stream file
        header('Content-Type: application/gzip');
        header('Content-Disposition: attachment; filename="' . basename($download['path']) . '"');
        header('Content-Length: ' . filesize($download['path']));
        header('X-LOX-Checksum: ' . hash_file('sha256', $download['path']));

        readfile($download['path']);
        @unlink($download['path']);
        exit;
    }

    /**
     * Process status
     */
    protected function processStatus()
    {
        if ($error = $this->verifyPullToken()) {
            $this->jsonResponse($error, 401);
            return;
        }

        $dbSize = 0;
        try {
            $result = Db::getInstance()->executeS("SELECT SUM(data_length + index_length) as size FROM information_schema.tables WHERE table_schema = '" . _DB_NAME_ . "'");
            $dbSize = (int) $result[0]['size'];
        } catch (Exception $e) {
        }

        $this->jsonResponse([
            'site_url' => Tools::getShopDomainSsl(true),
            'site_name' => Configuration::get('PS_SHOP_NAME'),
            'prestashop_version' => _PS_VERSION_,
            'module_version' => '1.3.0',
            'php_version' => PHP_VERSION,
            'pull_enabled' => true,
            'components' => [
                'database' => ['enabled' => (bool) Configuration::get('LOX_BACKUP_DATABASE'), 'size_bytes' => $dbSize],
                'files' => ['enabled' => (bool) Configuration::get('LOX_BACKUP_FILES')],
                'images' => ['enabled' => (bool) Configuration::get('LOX_BACKUP_IMAGES')],
                'modules' => ['enabled' => (bool) Configuration::get('LOX_BACKUP_MODULES')],
                'themes' => ['enabled' => (bool) Configuration::get('LOX_BACKUP_THEMES')],
            ],
            'last_backup' => Configuration::get('LOX_BACKUP_LAST_RUN'),
            'last_status' => Configuration::get('LOX_BACKUP_LAST_STATUS'),
        ]);
    }

    /**
     * Process refresh token
     */
    protected function processRefreshToken()
    {
        if ($error = $this->verifyApiKey()) {
            $this->jsonResponse($error, 401);
            return;
        }

        $token = bin2hex(random_bytes(32));
        $expiry = time() + (365 * 86400);

        Configuration::updateValue('LOX_BACKUP_PULL_TOKEN', $token);
        Configuration::updateValue('LOX_BACKUP_PULL_TOKEN_EXPIRY', $expiry);

        $this->jsonResponse([
            'success' => true,
            'token' => $token,
            'expires_at' => date('c', $expiry),
        ]);
    }

    /**
     * JSON response helper
     */
    protected function jsonResponse($data, $statusCode = 200)
    {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }

    /**
     * Generate new PULL token
     */
    public static function generatePullToken()
    {
        $token = bin2hex(random_bytes(32));
        $expiry = time() + (365 * 86400);

        Configuration::updateValue('LOX_BACKUP_PULL_TOKEN', $token);
        Configuration::updateValue('LOX_BACKUP_PULL_TOKEN_EXPIRY', $expiry);

        return [
            'token' => $token,
            'expires_at' => date('c', $expiry),
        ];
    }
}
