/**
 * LOX WooCommerce Admin JavaScript
 */

(function($) {
    'use strict';

    var LoxWcAdmin = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            // Create profile button
            $('#lox-wc-create-profile').on('click', this.showCreateModal);

            // Refresh profiles
            $('#lox-wc-refresh-profiles').on('click', this.refreshProfiles);

            // Modal close buttons
            $(document).on('click', '.lox-wc-modal-close, .lox-wc-modal-cancel', this.closeModal);
            $(document).on('click', '.lox-wc-modal', function(e) {
                if ($(e.target).hasClass('lox-wc-modal')) {
                    LoxWcAdmin.closeModal();
                }
            });

            // Create profile form
            $('#lox-wc-create-form').on('submit', this.createProfile);

            // Run backup buttons
            $(document).on('click', '.lox-wc-run-backup', this.showBackupModal);

            // Backup form
            $('#lox-wc-backup-form').on('submit', this.runBackup);

            // View versions buttons
            $(document).on('click', '.lox-wc-view-versions', this.showVersionsModal);

            // ESC key to close modal
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    LoxWcAdmin.closeModal();
                }
            });
        },

        showCreateModal: function() {
            $('#lox-wc-create-modal').show();
            $('#profile-name').focus();
        },

        showBackupModal: function() {
            var $btn = $(this);
            var uuid = $btn.data('uuid');
            var name = $btn.data('name');

            $('#backup-profile-uuid').val(uuid);
            $('#backup-profile-name').val(name);
            $('#backup-profile-display').text(name);
            $('#lox-wc-backup-progress').hide();
            $('#lox-wc-run-backup-btn').prop('disabled', false);
            $('#lox-wc-backup-modal').show();
        },

        showVersionsModal: function() {
            var uuid = $(this).data('uuid');

            $('#lox-wc-versions-container').html('<div class="lox-wc-loading">' + loxWcAdmin.strings.loading + '</div>');
            $('#lox-wc-versions-modal').show();

            LoxWcAdmin.loadVersions(uuid);
        },

        closeModal: function() {
            $('.lox-wc-modal').hide();
        },

        refreshProfiles: function() {
            var $btn = $(this);
            $btn.prop('disabled', true);

            $.ajax({
                url: loxWcAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'lox_wc_get_profiles',
                    nonce: loxWcAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert(response.data.message || loxWcAdmin.strings.error);
                    }
                },
                error: function() {
                    alert(loxWcAdmin.strings.error);
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        },

        createProfile: function(e) {
            e.preventDefault();

            var $form = $(this);
            var $btn = $form.find('button[type="submit"]');
            var name = $('#profile-name').val();
            var components = [];

            $form.find('input[name="components[]"]:checked').each(function() {
                components.push($(this).val());
            });

            if (!name) {
                alert('Please enter a profile name');
                return;
            }

            $btn.prop('disabled', true).text(loxWcAdmin.strings.loading);

            $.ajax({
                url: loxWcAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'lox_wc_create_profile',
                    nonce: loxWcAdmin.nonce,
                    name: name,
                    components: components
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert(response.data.message || loxWcAdmin.strings.error);
                        $btn.prop('disabled', false).text('Create Profile');
                    }
                },
                error: function() {
                    alert(loxWcAdmin.strings.error);
                    $btn.prop('disabled', false).text('Create Profile');
                }
            });
        },

        runBackup: function(e) {
            e.preventDefault();

            var $form = $(this);
            var $btn = $('#lox-wc-run-backup-btn');
            var $progress = $('#lox-wc-backup-progress');
            var uuid = $('#backup-profile-uuid').val();
            var name = $('#backup-profile-name').val();
            var components = [];

            $form.find('input[name="backup_components[]"]:checked').each(function() {
                components.push($(this).val());
            });

            if (components.length === 0) {
                alert('Please select at least one component');
                return;
            }

            $btn.prop('disabled', true);
            $progress.show();
            $('.lox-wc-progress-fill').css('width', '30%');

            $.ajax({
                url: loxWcAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'lox_wc_run_profile_backup',
                    nonce: loxWcAdmin.nonce,
                    profile_uuid: uuid,
                    profile_name: name,
                    components: components
                },
                timeout: 3600000, // 1 hour timeout
                success: function(response) {
                    if (response.success) {
                        $('.lox-wc-progress-fill').css('width', '100%');
                        $('.lox-wc-progress-text').text(loxWcAdmin.strings.success + ' UUID: ' + response.data.uuid);
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        alert(response.data.message || loxWcAdmin.strings.error);
                        $progress.hide();
                        $btn.prop('disabled', false);
                    }
                },
                error: function() {
                    alert(loxWcAdmin.strings.error);
                    $progress.hide();
                    $btn.prop('disabled', false);
                }
            });

            // Animate progress bar
            var width = 30;
            var interval = setInterval(function() {
                if (width < 90) {
                    width += Math.random() * 5;
                    $('.lox-wc-progress-fill').css('width', width + '%');
                }
            }, 2000);
        },

        loadVersions: function(uuid) {
            $.ajax({
                url: loxWcAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'lox_wc_get_profile_versions',
                    nonce: loxWcAdmin.nonce,
                    uuid: uuid
                },
                success: function(response) {
                    if (response.success) {
                        LoxWcAdmin.renderVersions(response.data);
                    } else {
                        $('#lox-wc-versions-container').html('<p>' + (response.data.message || loxWcAdmin.strings.error) + '</p>');
                    }
                },
                error: function() {
                    $('#lox-wc-versions-container').html('<p>' + loxWcAdmin.strings.error + '</p>');
                }
            });
        },

        renderVersions: function(data) {
            var versions = data.versions || data.backups || data;

            if (!versions || versions.length === 0) {
                $('#lox-wc-versions-container').html('<p>No backup versions found.</p>');
                return;
            }

            var html = '<table class="lox-wc-versions-table">';
            html += '<thead><tr>';
            html += '<th>UUID</th>';
            html += '<th>Date</th>';
            html += '<th>Size</th>';
            html += '<th>Status</th>';
            html += '</tr></thead>';
            html += '<tbody>';

            versions.forEach(function(version) {
                var uuid = version.uuid || '';
                var created = version.created_at ? new Date(version.created_at).toLocaleString() : '-';
                var size = LoxWcAdmin.formatBytes(version.size_bytes || 0);
                var status = version.status || 'unknown';

                html += '<tr>';
                html += '<td><span class="lox-wc-version-uuid">' + uuid.substring(0, 8) + '...</span></td>';
                html += '<td>' + created + '</td>';
                html += '<td>' + size + '</td>';
                html += '<td><span class="lox-wc-version-status ' + status + '">' + status + '</span></td>';
                html += '</tr>';
            });

            html += '</tbody></table>';

            $('#lox-wc-versions-container').html(html);
        },

        formatBytes: function(bytes) {
            if (bytes === 0) return '--';

            var units = ['B', 'KB', 'MB', 'GB', 'TB'];
            var i = 0;

            while (bytes >= 1024 && i < units.length - 1) {
                bytes /= 1024;
                i++;
            }

            return bytes.toFixed(2) + ' ' + units[i];
        }
    };

    $(document).ready(function() {
        LoxWcAdmin.init();
    });

})(jQuery);
