<?php
/**
 * LOX WooCommerce Admin - Cloud Profiles
 *
 * @package LOX_Backup_WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Admin class for WooCommerce Cloud Profiles
 */
class LOX_WC_Admin {

    /**
     * Initialize admin hooks
     */
    public static function init() {
        add_action('admin_menu', array(__CLASS__, 'add_menu'), 99);
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
        add_action('wp_ajax_lox_wc_get_profiles', array(__CLASS__, 'ajax_get_profiles'));
        add_action('wp_ajax_lox_wc_get_profile_versions', array(__CLASS__, 'ajax_get_profile_versions'));
        add_action('wp_ajax_lox_wc_run_profile_backup', array(__CLASS__, 'ajax_run_profile_backup'));
        add_action('wp_ajax_lox_wc_create_profile', array(__CLASS__, 'ajax_create_profile'));
    }

    /**
     * Add submenu under WooCommerce
     */
    public static function add_menu() {
        add_submenu_page(
            'woocommerce',
            __('LOX Cloud Profiles', 'lox-backup-woocommerce'),
            __('LOX Profiles', 'lox-backup-woocommerce'),
            'manage_woocommerce',
            'lox-wc-profiles',
            array(__CLASS__, 'render_profiles_page')
        );
    }

    /**
     * Enqueue admin scripts
     */
    public static function enqueue_scripts($hook) {
        if ($hook !== 'woocommerce_page_lox-wc-profiles') {
            return;
        }

        wp_enqueue_style(
            'lox-wc-admin',
            LOX_WC_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            LOX_WC_VERSION
        );

        wp_enqueue_script(
            'lox-wc-admin',
            LOX_WC_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            LOX_WC_VERSION,
            true
        );

        wp_localize_script('lox-wc-admin', 'loxWcAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('lox_wc_admin'),
            'strings' => array(
                'loading' => __('Loading...', 'lox-backup-woocommerce'),
                'error' => __('An error occurred', 'lox-backup-woocommerce'),
                'running' => __('Running backup...', 'lox-backup-woocommerce'),
                'success' => __('Backup completed!', 'lox-backup-woocommerce'),
                'confirm_backup' => __('Run backup for this profile?', 'lox-backup-woocommerce'),
            ),
        ));
    }

    /**
     * Render profiles page
     */
    public static function render_profiles_page() {
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('You do not have permission to access this page.', 'lox-backup-woocommerce'));
        }

        $api = new LOX_API();
        $profiles_result = $api->get_profiles(array('source' => 'woocommerce'));
        $profiles = array();

        if (!is_wp_error($profiles_result)) {
            $profiles = isset($profiles_result['profiles']) ? $profiles_result['profiles'] : $profiles_result;
        }

        ?>
        <div class="wrap lox-wc-profiles-wrap">
            <h1><?php esc_html_e('LOX Cloud Profiles', 'lox-backup-woocommerce'); ?></h1>
            <p class="description">
                <?php esc_html_e('Manage your WooCommerce backup profiles synced with LOX cloud. Each profile groups related backup versions.', 'lox-backup-woocommerce'); ?>
            </p>

            <div class="lox-wc-actions">
                <button type="button" class="button button-primary" id="lox-wc-create-profile">
                    <span class="dashicons dashicons-plus-alt"></span>
                    <?php esc_html_e('Create Profile', 'lox-backup-woocommerce'); ?>
                </button>
                <button type="button" class="button" id="lox-wc-refresh-profiles">
                    <span class="dashicons dashicons-update"></span>
                    <?php esc_html_e('Refresh', 'lox-backup-woocommerce'); ?>
                </button>
            </div>

            <div id="lox-wc-profiles-container">
                <?php if (empty($profiles)): ?>
                    <div class="lox-wc-empty-state">
                        <span class="dashicons dashicons-cloud"></span>
                        <h3><?php esc_html_e('No cloud profiles yet', 'lox-backup-woocommerce'); ?></h3>
                        <p><?php esc_html_e('Create a profile to start grouping your WooCommerce backups.', 'lox-backup-woocommerce'); ?></p>
                    </div>
                <?php else: ?>
                    <div class="lox-wc-profiles-grid">
                        <?php foreach ($profiles as $profile): ?>
                            <?php self::render_profile_card($profile); ?>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Create Profile Modal -->
            <div id="lox-wc-create-modal" class="lox-wc-modal" style="display:none;">
                <div class="lox-wc-modal-content">
                    <span class="lox-wc-modal-close">&times;</span>
                    <h2><?php esc_html_e('Create New Profile', 'lox-backup-woocommerce'); ?></h2>
                    <form id="lox-wc-create-form">
                        <p>
                            <label for="profile-name"><?php esc_html_e('Profile Name', 'lox-backup-woocommerce'); ?></label>
                            <input type="text" id="profile-name" name="name" required placeholder="e.g., Daily Store Backup">
                        </p>
                        <p>
                            <label><?php esc_html_e('Components to Include', 'lox-backup-woocommerce'); ?></label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="components[]" value="catalog" checked>
                                <?php esc_html_e('Catalog (Products, Categories)', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="components[]" value="transactional" checked>
                                <?php esc_html_e('Orders & Customers', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="components[]" value="files">
                                <?php esc_html_e('Media Files', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="components[]" value="config">
                                <?php esc_html_e('Configuration', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <p class="submit">
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e('Create Profile', 'lox-backup-woocommerce'); ?>
                            </button>
                            <button type="button" class="button lox-wc-modal-cancel">
                                <?php esc_html_e('Cancel', 'lox-backup-woocommerce'); ?>
                            </button>
                        </p>
                    </form>
                </div>
            </div>

            <!-- Run Backup Modal -->
            <div id="lox-wc-backup-modal" class="lox-wc-modal" style="display:none;">
                <div class="lox-wc-modal-content">
                    <span class="lox-wc-modal-close">&times;</span>
                    <h2><?php esc_html_e('Run Profile Backup', 'lox-backup-woocommerce'); ?></h2>
                    <form id="lox-wc-backup-form">
                        <input type="hidden" id="backup-profile-uuid" name="profile_uuid">
                        <input type="hidden" id="backup-profile-name" name="profile_name">
                        <p>
                            <strong><?php esc_html_e('Profile:', 'lox-backup-woocommerce'); ?></strong>
                            <span id="backup-profile-display"></span>
                        </p>
                        <p>
                            <label><?php esc_html_e('Components to Backup', 'lox-backup-woocommerce'); ?></label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="backup_components[]" value="catalog" checked>
                                <?php esc_html_e('Catalog (Products, Categories)', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="backup_components[]" value="transactional" checked>
                                <?php esc_html_e('Orders & Customers', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="backup_components[]" value="files">
                                <?php esc_html_e('Media Files', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <p>
                            <label>
                                <input type="checkbox" name="backup_components[]" value="config">
                                <?php esc_html_e('Configuration', 'lox-backup-woocommerce'); ?>
                            </label>
                        </p>
                        <div id="lox-wc-backup-progress" style="display:none;">
                            <div class="lox-wc-progress-bar">
                                <div class="lox-wc-progress-fill"></div>
                            </div>
                            <p class="lox-wc-progress-text"><?php esc_html_e('Running backup...', 'lox-backup-woocommerce'); ?></p>
                        </div>
                        <p class="submit">
                            <button type="submit" class="button button-primary" id="lox-wc-run-backup-btn">
                                <?php esc_html_e('Run Backup Now', 'lox-backup-woocommerce'); ?>
                            </button>
                            <button type="button" class="button lox-wc-modal-cancel">
                                <?php esc_html_e('Cancel', 'lox-backup-woocommerce'); ?>
                            </button>
                        </p>
                    </form>
                </div>
            </div>

            <!-- Versions Modal -->
            <div id="lox-wc-versions-modal" class="lox-wc-modal" style="display:none;">
                <div class="lox-wc-modal-content lox-wc-modal-wide">
                    <span class="lox-wc-modal-close">&times;</span>
                    <h2><?php esc_html_e('Backup Versions', 'lox-backup-woocommerce'); ?></h2>
                    <div id="lox-wc-versions-container">
                        <p><?php esc_html_e('Loading...', 'lox-backup-woocommerce'); ?></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render a single profile card
     *
     * @param array $profile Profile data
     */
    private static function render_profile_card($profile) {
        $uuid = isset($profile['uuid']) ? $profile['uuid'] : '';
        $name = isset($profile['name']) ? $profile['name'] : __('Unnamed', 'lox-backup-woocommerce');
        $backup_count = isset($profile['backup_count']) ? (int) $profile['backup_count'] : 0;
        $total_size = isset($profile['total_size_bytes']) ? $profile['total_size_bytes'] : 0;
        $last_backup = isset($profile['last_backup_at']) ? $profile['last_backup_at'] : null;
        ?>
        <div class="lox-wc-profile-card" data-uuid="<?php echo esc_attr($uuid); ?>">
            <div class="lox-wc-profile-header">
                <h3><?php echo esc_html($name); ?></h3>
                <span class="lox-wc-profile-uuid"><?php echo esc_html(substr($uuid, 0, 8)); ?>...</span>
            </div>
            <div class="lox-wc-profile-stats">
                <div class="lox-wc-stat">
                    <span class="lox-wc-stat-value"><?php echo esc_html($backup_count); ?></span>
                    <span class="lox-wc-stat-label"><?php esc_html_e('Versions', 'lox-backup-woocommerce'); ?></span>
                </div>
                <div class="lox-wc-stat">
                    <span class="lox-wc-stat-value"><?php echo esc_html(self::format_bytes($total_size)); ?></span>
                    <span class="lox-wc-stat-label"><?php esc_html_e('Total Size', 'lox-backup-woocommerce'); ?></span>
                </div>
            </div>
            <div class="lox-wc-profile-meta">
                <span class="dashicons dashicons-clock"></span>
                <?php if ($last_backup): ?>
                    <?php echo esc_html(sprintf(__('Last backup: %s', 'lox-backup-woocommerce'), date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($last_backup)))); ?>
                <?php else: ?>
                    <?php esc_html_e('No backups yet', 'lox-backup-woocommerce'); ?>
                <?php endif; ?>
            </div>
            <div class="lox-wc-profile-actions">
                <button type="button" class="button button-primary lox-wc-run-backup"
                        data-uuid="<?php echo esc_attr($uuid); ?>"
                        data-name="<?php echo esc_attr($name); ?>">
                    <span class="dashicons dashicons-backup"></span>
                    <?php esc_html_e('Run Backup', 'lox-backup-woocommerce'); ?>
                </button>
                <button type="button" class="button lox-wc-view-versions"
                        data-uuid="<?php echo esc_attr($uuid); ?>">
                    <span class="dashicons dashicons-list-view"></span>
                    <?php esc_html_e('Versions', 'lox-backup-woocommerce'); ?>
                </button>
            </div>
        </div>
        <?php
    }

    /**
     * Format bytes to human readable
     *
     * @param int $bytes
     * @return string
     */
    private static function format_bytes($bytes) {
        if ($bytes === 0) {
            return '--';
        }

        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $i = 0;

        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * AJAX: Get profiles
     */
    public static function ajax_get_profiles() {
        check_ajax_referer('lox_wc_admin', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup-woocommerce')));
        }

        $api = new LOX_API();
        $result = $api->get_profiles(array('source' => 'woocommerce'));

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Get profile versions
     */
    public static function ajax_get_profile_versions() {
        check_ajax_referer('lox_wc_admin', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup-woocommerce')));
        }

        $uuid = isset($_POST['uuid']) ? sanitize_text_field($_POST['uuid']) : '';
        $page = isset($_POST['page']) ? (int) $_POST['page'] : 1;

        if (empty($uuid)) {
            wp_send_json_error(array('message' => __('Invalid profile', 'lox-backup-woocommerce')));
        }

        $api = new LOX_API();
        $result = $api->get_profile_versions($uuid, $page, 10);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Run profile backup
     */
    public static function ajax_run_profile_backup() {
        check_ajax_referer('lox_wc_admin', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup-woocommerce')));
        }

        $uuid = isset($_POST['profile_uuid']) ? sanitize_text_field($_POST['profile_uuid']) : '';
        $name = isset($_POST['profile_name']) ? sanitize_text_field($_POST['profile_name']) : 'Profile';
        $components = isset($_POST['components']) ? array_map('sanitize_text_field', $_POST['components']) : array('catalog');

        if (empty($uuid)) {
            wp_send_json_error(array('message' => __('Invalid profile', 'lox-backup-woocommerce')));
        }

        // Increase execution time
        set_time_limit(3600);

        require_once LOX_WC_PLUGIN_DIR . 'includes/class-lox-wc-backup-handler.php';
        $handler = new LOX_WC_Backup_Handler();
        $result = $handler->run_profile_backup($uuid, $name, $components);

        if ($result['success']) {
            wp_send_json_success(array(
                'message' => __('Backup completed successfully!', 'lox-backup-woocommerce'),
                'uuid' => $result['uuid'],
                'size_bytes' => $result['size_bytes'],
            ));
        } else {
            wp_send_json_error(array('message' => $result['error']));
        }
    }

    /**
     * AJAX: Create profile
     */
    public static function ajax_create_profile() {
        check_ajax_referer('lox_wc_admin', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup-woocommerce')));
        }

        $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
        $components = isset($_POST['components']) ? array_map('sanitize_text_field', $_POST['components']) : array();

        if (empty($name)) {
            wp_send_json_error(array('message' => __('Profile name is required', 'lox-backup-woocommerce')));
        }

        $api = new LOX_API();
        $result = $api->create_profile($name, array(
            'source' => 'woocommerce',
            'description' => sprintf(
                'WooCommerce profile for %s - Components: %s',
                get_bloginfo('name'),
                implode(', ', $components)
            ),
        ));

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success(array(
            'message' => __('Profile created successfully!', 'lox-backup-woocommerce'),
            'profile' => $result,
        ));
    }
}
