<?php
/**
 * LOX WooCommerce Backup Integration
 *
 * @package LOX_Backup_WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Integrates WooCommerce data export with LOX Backup
 */
class LOX_WC_Backup {

    /**
     * Initialize
     */
    public static function init() {
        // Add WooCommerce backup components to settings
        add_filter('lox_backup_components', array(__CLASS__, 'add_components'));

        // Hook into backup process
        add_filter('lox_backup_files', array(__CLASS__, 'add_woocommerce_exports'), 10, 2);

        // Add admin settings fields
        add_action('lox_backup_settings_fields', array(__CLASS__, 'add_settings_fields'));

        // Register our settings
        add_action('admin_init', array(__CLASS__, 'register_settings'));

        // Register component backup crons
        add_action('lox_backup_component_catalog', array(__CLASS__, 'run_catalog_backup'));
        add_action('lox_backup_component_orders', array(__CLASS__, 'run_orders_backup'));
    }

    /**
     * Get WooCommerce-specific component definitions
     *
     * @return array
     */
    public static function get_wc_components() {
        return array(
            'catalog' => array(
                'name' => 'Catalog',
                'description' => 'Products, categories, attributes, and variations',
                'tables' => array(
                    'wc_product_meta_lookup',
                    'wc_product_attributes_lookup',
                    'wc_reserved_stock',
                ),
                'exports' => array('products'),
            ),
            'orders' => array(
                'name' => 'Orders & Customers',
                'description' => 'Orders, customers, subscriptions, and payment data',
                'tables' => array(
                    'wc_orders',
                    'wc_order_items',
                    'wc_order_itemmeta',
                    'wc_order_stats',
                    'wc_customer_lookup',
                    'wc_order_addresses',
                    'wc_order_operational_data',
                ),
                'exports' => array('orders', 'customers'),
            ),
        );
    }

    /**
     * Run catalog component backup
     */
    public static function run_catalog_backup() {
        $backup = self::create_wc_backup_instance();
        $backup->run_wc_component_backup('catalog');
    }

    /**
     * Run orders component backup
     */
    public static function run_orders_backup() {
        $backup = self::create_wc_backup_instance();
        $backup->run_wc_component_backup('orders');
    }

    /**
     * Create WooCommerce backup instance
     *
     * @return LOX_WC_Backup_Handler
     */
    private static function create_wc_backup_instance() {
        require_once dirname(__FILE__) . '/class-lox-wc-backup-handler.php';
        return new LOX_WC_Backup_Handler();
    }

    /**
     * Add WooCommerce components to backup options
     *
     * @param array $components
     * @return array
     */
    public static function add_components($components) {
        $components['backup_wc_orders'] = __('WooCommerce Orders', 'lox-backup-woocommerce');
        $components['backup_wc_products'] = __('WooCommerce Products', 'lox-backup-woocommerce');
        $components['backup_wc_customers'] = __('WooCommerce Customers', 'lox-backup-woocommerce');
        $components['backup_wc_coupons'] = __('WooCommerce Coupons', 'lox-backup-woocommerce');
        $components['backup_wc_settings'] = __('WooCommerce Settings', 'lox-backup-woocommerce');

        return $components;
    }

    /**
     * Register additional settings
     */
    public static function register_settings() {
        register_setting('lox_backup_settings_group', 'lox_wc_backup_settings', array(
            'sanitize_callback' => array(__CLASS__, 'sanitize_settings'),
        ));
    }

    /**
     * Sanitize WooCommerce settings
     *
     * @param array $input
     * @return array
     */
    public static function sanitize_settings($input) {
        $sanitized = array();

        $sanitized['backup_wc_orders'] = !empty($input['backup_wc_orders']);
        $sanitized['backup_wc_products'] = !empty($input['backup_wc_products']);
        $sanitized['backup_wc_customers'] = !empty($input['backup_wc_customers']);
        $sanitized['backup_wc_coupons'] = !empty($input['backup_wc_coupons']);
        $sanitized['backup_wc_settings'] = !empty($input['backup_wc_settings']);

        return $sanitized;
    }

    /**
     * Add settings fields to admin
     */
    public static function add_settings_fields() {
        add_settings_section(
            'lox_wc_backup_settings',
            __('WooCommerce Backup', 'lox-backup-woocommerce'),
            array(__CLASS__, 'render_section'),
            'lox-backup'
        );

        add_settings_field(
            'wc_components',
            __('WooCommerce Data', 'lox-backup-woocommerce'),
            array(__CLASS__, 'render_components_field'),
            'lox-backup',
            'lox_wc_backup_settings'
        );
    }

    /**
     * Render section description
     */
    public static function render_section() {
        echo '<p>' . __('Select WooCommerce data to include in backups. Data is exported as JSON for easy restoration.', 'lox-backup-woocommerce') . '</p>';
    }

    /**
     * Render WooCommerce components field
     */
    public static function render_components_field() {
        $settings = get_option('lox_wc_backup_settings', array());
        $components = array(
            'backup_wc_orders' => array(
                'label' => __('Orders', 'lox-backup-woocommerce'),
                'description' => __('All orders with line items, shipping, and payment details', 'lox-backup-woocommerce'),
            ),
            'backup_wc_products' => array(
                'label' => __('Products', 'lox-backup-woocommerce'),
                'description' => __('Products with variations, attributes, and pricing', 'lox-backup-woocommerce'),
            ),
            'backup_wc_customers' => array(
                'label' => __('Customers', 'lox-backup-woocommerce'),
                'description' => __('Customer profiles and addresses', 'lox-backup-woocommerce'),
            ),
            'backup_wc_coupons' => array(
                'label' => __('Coupons', 'lox-backup-woocommerce'),
                'description' => __('All coupon codes and settings', 'lox-backup-woocommerce'),
            ),
            'backup_wc_settings' => array(
                'label' => __('Settings', 'lox-backup-woocommerce'),
                'description' => __('Store configuration and options', 'lox-backup-woocommerce'),
            ),
        );

        echo '<fieldset>';
        foreach ($components as $key => $component) {
            $checked = !empty($settings[$key]);
            ?>
            <label style="display: block; margin-bottom: 8px;">
                <input type="checkbox" name="lox_wc_backup_settings[<?php echo esc_attr($key); ?>]"
                       value="1" <?php checked($checked); ?>>
                <strong><?php echo esc_html($component['label']); ?></strong>
                <span style="color: #666; display: block; margin-left: 24px; font-size: 13px;">
                    <?php echo esc_html($component['description']); ?>
                </span>
            </label>
            <?php
        }
        echo '</fieldset>';
    }

    /**
     * Add WooCommerce exports to backup
     *
     * @param array $files Files to backup
     * @param string $temp_dir Temp directory
     * @return array
     */
    public static function add_woocommerce_exports($files, $temp_dir) {
        $settings = get_option('lox_wc_backup_settings', array());

        // Create WooCommerce export directory
        $wc_dir = $temp_dir . '/woocommerce';
        if (!file_exists($wc_dir)) {
            wp_mkdir_p($wc_dir);
        }

        // Export orders
        if (!empty($settings['backup_wc_orders'])) {
            $orders_file = $wc_dir . '/orders.json';
            $result = LOX_WC_Export::export_orders($orders_file);
            if (!is_wp_error($result) && file_exists($orders_file)) {
                $files[] = $orders_file;
                self::log('Exported WooCommerce orders');
            }
        }

        // Export products
        if (!empty($settings['backup_wc_products'])) {
            $products_file = $wc_dir . '/products.json';
            $result = LOX_WC_Export::export_products($products_file);
            if (!is_wp_error($result) && file_exists($products_file)) {
                $files[] = $products_file;
                self::log('Exported WooCommerce products');
            }
        }

        // Export customers
        if (!empty($settings['backup_wc_customers'])) {
            $customers_file = $wc_dir . '/customers.json';
            $result = LOX_WC_Export::export_customers($customers_file);
            if (!is_wp_error($result) && file_exists($customers_file)) {
                $files[] = $customers_file;
                self::log('Exported WooCommerce customers');
            }
        }

        // Export coupons
        if (!empty($settings['backup_wc_coupons'])) {
            $coupons_file = $wc_dir . '/coupons.json';
            $result = LOX_WC_Export::export_coupons($coupons_file);
            if (!is_wp_error($result) && file_exists($coupons_file)) {
                $files[] = $coupons_file;
                self::log('Exported WooCommerce coupons');
            }
        }

        // Export settings
        if (!empty($settings['backup_wc_settings'])) {
            $settings_file = $wc_dir . '/settings.json';
            $result = LOX_WC_Export::export_settings($settings_file);
            if (!is_wp_error($result) && file_exists($settings_file)) {
                $files[] = $settings_file;
                self::log('Exported WooCommerce settings');
            }
        }

        // Export tax rates
        $tax_file = self::export_tax_rates($wc_dir . '/tax-rates.json');
        if ($tax_file) {
            $files[] = $tax_file;
        }

        // Export shipping zones
        $shipping_file = self::export_shipping_zones($wc_dir . '/shipping-zones.json');
        if ($shipping_file) {
            $files[] = $shipping_file;
        }

        return $files;
    }

    /**
     * Export tax rates
     *
     * @param string $file_path
     * @return string|false
     */
    private static function export_tax_rates($file_path) {
        global $wpdb;

        $rates = $wpdb->get_results("
            SELECT tr.*, GROUP_CONCAT(trl.location_code) as postcodes
            FROM {$wpdb->prefix}woocommerce_tax_rates tr
            LEFT JOIN {$wpdb->prefix}woocommerce_tax_rate_locations trl
                ON tr.tax_rate_id = trl.tax_rate_id AND trl.location_type = 'postcode'
            GROUP BY tr.tax_rate_id
        ", ARRAY_A);

        if (empty($rates)) {
            return false;
        }

        $result = file_put_contents($file_path, json_encode($rates, JSON_PRETTY_PRINT));
        if ($result === false) {
            return false;
        }

        self::log('Exported tax rates');
        return $file_path;
    }

    /**
     * Export shipping zones
     *
     * @param string $file_path
     * @return string|false
     */
    private static function export_shipping_zones($file_path) {
        $zones_data = array();
        $zones = WC_Shipping_Zones::get_zones();

        foreach ($zones as $zone) {
            $zone_obj = new WC_Shipping_Zone($zone['zone_id']);
            $zone_data = array(
                'id' => $zone['zone_id'],
                'name' => $zone['zone_name'],
                'order' => $zone['zone_order'],
                'locations' => array(),
                'methods' => array(),
            );

            // Zone locations
            foreach ($zone_obj->get_zone_locations() as $location) {
                $zone_data['locations'][] = array(
                    'code' => $location->code,
                    'type' => $location->type,
                );
            }

            // Shipping methods
            foreach ($zone_obj->get_shipping_methods() as $method) {
                $zone_data['methods'][] = array(
                    'instance_id' => $method->instance_id,
                    'method_id' => $method->id,
                    'title' => $method->title,
                    'enabled' => $method->enabled,
                    'settings' => $method->instance_settings,
                );
            }

            $zones_data[] = $zone_data;
        }

        if (empty($zones_data)) {
            return false;
        }

        $result = file_put_contents($file_path, json_encode($zones_data, JSON_PRETTY_PRINT));
        if ($result === false) {
            return false;
        }

        self::log('Exported shipping zones');
        return $file_path;
    }

    /**
     * Log message
     *
     * @param string $message
     */
    private static function log($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[LOX WooCommerce] ' . $message);
        }
    }
}
