<?php
/**
 * LOX WooCommerce Data Export
 *
 * @package LOX_Backup_WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles WooCommerce-specific data exports
 */
class LOX_WC_Export {

    /**
     * Export orders to JSON
     *
     * @param string $file_path Output file path
     * @param array $args Export arguments
     * @return bool|WP_Error
     */
    public static function export_orders($file_path, $args = array()) {
        $defaults = array(
            'status' => array('wc-completed', 'wc-processing', 'wc-on-hold', 'wc-pending', 'wc-refunded'),
            'limit' => -1,
            'orderby' => 'ID',
            'order' => 'ASC',
        );

        $args = wp_parse_args($args, $defaults);

        $orders = wc_get_orders($args);

        if (empty($orders)) {
            return true;
        }

        $handle = fopen($file_path, 'w');
        if (!$handle) {
            return new WP_Error('export_failed', __('Failed to create export file', 'lox-backup-woocommerce'));
        }

        fwrite($handle, "[\n");
        $first = true;

        foreach ($orders as $order) {
            if (!$first) {
                fwrite($handle, ",\n");
            }
            $first = false;

            $order_data = self::format_order($order);
            fwrite($handle, json_encode($order_data, JSON_PRETTY_PRINT));
        }

        fwrite($handle, "\n]");
        fclose($handle);

        return true;
    }

    /**
     * Format order data for export
     *
     * @param WC_Order $order
     * @return array
     */
    private static function format_order($order) {
        $data = array(
            'id' => $order->get_id(),
            'number' => $order->get_order_number(),
            'status' => $order->get_status(),
            'currency' => $order->get_currency(),
            'date_created' => $order->get_date_created() ? $order->get_date_created()->format('Y-m-d H:i:s') : null,
            'date_modified' => $order->get_date_modified() ? $order->get_date_modified()->format('Y-m-d H:i:s') : null,
            'date_completed' => $order->get_date_completed() ? $order->get_date_completed()->format('Y-m-d H:i:s') : null,
            'date_paid' => $order->get_date_paid() ? $order->get_date_paid()->format('Y-m-d H:i:s') : null,
            'total' => $order->get_total(),
            'subtotal' => $order->get_subtotal(),
            'total_tax' => $order->get_total_tax(),
            'shipping_total' => $order->get_shipping_total(),
            'discount_total' => $order->get_discount_total(),
            'payment_method' => $order->get_payment_method(),
            'payment_method_title' => $order->get_payment_method_title(),
            'transaction_id' => $order->get_transaction_id(),
            'customer_id' => $order->get_customer_id(),
            'customer_note' => $order->get_customer_note(),
            'billing' => array(
                'first_name' => $order->get_billing_first_name(),
                'last_name' => $order->get_billing_last_name(),
                'company' => $order->get_billing_company(),
                'address_1' => $order->get_billing_address_1(),
                'address_2' => $order->get_billing_address_2(),
                'city' => $order->get_billing_city(),
                'state' => $order->get_billing_state(),
                'postcode' => $order->get_billing_postcode(),
                'country' => $order->get_billing_country(),
                'email' => $order->get_billing_email(),
                'phone' => $order->get_billing_phone(),
            ),
            'shipping' => array(
                'first_name' => $order->get_shipping_first_name(),
                'last_name' => $order->get_shipping_last_name(),
                'company' => $order->get_shipping_company(),
                'address_1' => $order->get_shipping_address_1(),
                'address_2' => $order->get_shipping_address_2(),
                'city' => $order->get_shipping_city(),
                'state' => $order->get_shipping_state(),
                'postcode' => $order->get_shipping_postcode(),
                'country' => $order->get_shipping_country(),
            ),
            'line_items' => array(),
            'shipping_lines' => array(),
            'fee_lines' => array(),
            'coupon_lines' => array(),
            'refunds' => array(),
            'meta_data' => array(),
        );

        // Line items
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $data['line_items'][] = array(
                'id' => $item->get_id(),
                'name' => $item->get_name(),
                'product_id' => $item->get_product_id(),
                'variation_id' => $item->get_variation_id(),
                'quantity' => $item->get_quantity(),
                'subtotal' => $item->get_subtotal(),
                'total' => $item->get_total(),
                'total_tax' => $item->get_total_tax(),
                'sku' => $product ? $product->get_sku() : '',
            );
        }

        // Shipping lines
        foreach ($order->get_shipping_methods() as $shipping) {
            $data['shipping_lines'][] = array(
                'id' => $shipping->get_id(),
                'method_title' => $shipping->get_method_title(),
                'method_id' => $shipping->get_method_id(),
                'total' => $shipping->get_total(),
                'total_tax' => $shipping->get_total_tax(),
            );
        }

        // Fee lines
        foreach ($order->get_fees() as $fee) {
            $data['fee_lines'][] = array(
                'id' => $fee->get_id(),
                'name' => $fee->get_name(),
                'total' => $fee->get_total(),
                'total_tax' => $fee->get_total_tax(),
            );
        }

        // Coupon lines
        foreach ($order->get_coupons() as $coupon) {
            $data['coupon_lines'][] = array(
                'id' => $coupon->get_id(),
                'code' => $coupon->get_code(),
                'discount' => $coupon->get_discount(),
                'discount_tax' => $coupon->get_discount_tax(),
            );
        }

        // Refunds
        foreach ($order->get_refunds() as $refund) {
            $data['refunds'][] = array(
                'id' => $refund->get_id(),
                'reason' => $refund->get_reason(),
                'total' => $refund->get_total(),
                'date_created' => $refund->get_date_created() ? $refund->get_date_created()->format('Y-m-d H:i:s') : null,
            );
        }

        // Meta data (filtered)
        $skip_meta = array('_billing_', '_shipping_', '_order_', '_customer_', '_payment_', '_transaction_');
        foreach ($order->get_meta_data() as $meta) {
            $key = $meta->key;
            $skip = false;
            foreach ($skip_meta as $prefix) {
                if (strpos($key, $prefix) === 0) {
                    $skip = true;
                    break;
                }
            }
            if (!$skip && substr($key, 0, 1) !== '_') {
                $data['meta_data'][] = array(
                    'key' => $key,
                    'value' => $meta->value,
                );
            }
        }

        return $data;
    }

    /**
     * Export products to JSON
     *
     * @param string $file_path Output file path
     * @param array $args Export arguments
     * @return bool|WP_Error
     */
    public static function export_products($file_path, $args = array()) {
        $defaults = array(
            'status' => 'publish',
            'limit' => -1,
            'orderby' => 'ID',
            'order' => 'ASC',
        );

        $args = wp_parse_args($args, $defaults);

        $products = wc_get_products($args);

        if (empty($products)) {
            return true;
        }

        $handle = fopen($file_path, 'w');
        if (!$handle) {
            return new WP_Error('export_failed', __('Failed to create export file', 'lox-backup-woocommerce'));
        }

        fwrite($handle, "[\n");
        $first = true;

        foreach ($products as $product) {
            if (!$first) {
                fwrite($handle, ",\n");
            }
            $first = false;

            $product_data = self::format_product($product);
            fwrite($handle, json_encode($product_data, JSON_PRETTY_PRINT));
        }

        fwrite($handle, "\n]");
        fclose($handle);

        return true;
    }

    /**
     * Format product data for export
     *
     * @param WC_Product $product
     * @return array
     */
    private static function format_product($product) {
        $data = array(
            'id' => $product->get_id(),
            'name' => $product->get_name(),
            'slug' => $product->get_slug(),
            'type' => $product->get_type(),
            'status' => $product->get_status(),
            'featured' => $product->get_featured(),
            'catalog_visibility' => $product->get_catalog_visibility(),
            'description' => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'sku' => $product->get_sku(),
            'price' => $product->get_price(),
            'regular_price' => $product->get_regular_price(),
            'sale_price' => $product->get_sale_price(),
            'date_on_sale_from' => $product->get_date_on_sale_from() ? $product->get_date_on_sale_from()->format('Y-m-d H:i:s') : null,
            'date_on_sale_to' => $product->get_date_on_sale_to() ? $product->get_date_on_sale_to()->format('Y-m-d H:i:s') : null,
            'total_sales' => $product->get_total_sales(),
            'tax_status' => $product->get_tax_status(),
            'tax_class' => $product->get_tax_class(),
            'manage_stock' => $product->get_manage_stock(),
            'stock_quantity' => $product->get_stock_quantity(),
            'stock_status' => $product->get_stock_status(),
            'backorders' => $product->get_backorders(),
            'low_stock_amount' => $product->get_low_stock_amount(),
            'sold_individually' => $product->get_sold_individually(),
            'weight' => $product->get_weight(),
            'length' => $product->get_length(),
            'width' => $product->get_width(),
            'height' => $product->get_height(),
            'shipping_class_id' => $product->get_shipping_class_id(),
            'reviews_allowed' => $product->get_reviews_allowed(),
            'average_rating' => $product->get_average_rating(),
            'rating_count' => $product->get_rating_count(),
            'parent_id' => $product->get_parent_id(),
            'purchase_note' => $product->get_purchase_note(),
            'menu_order' => $product->get_menu_order(),
            'virtual' => $product->get_virtual(),
            'downloadable' => $product->get_downloadable(),
            'category_ids' => $product->get_category_ids(),
            'tag_ids' => $product->get_tag_ids(),
            'image_id' => $product->get_image_id(),
            'gallery_image_ids' => $product->get_gallery_image_ids(),
            'date_created' => $product->get_date_created() ? $product->get_date_created()->format('Y-m-d H:i:s') : null,
            'date_modified' => $product->get_date_modified() ? $product->get_date_modified()->format('Y-m-d H:i:s') : null,
            'attributes' => array(),
            'variations' => array(),
            'meta_data' => array(),
        );

        // Attributes
        foreach ($product->get_attributes() as $attribute) {
            if (is_a($attribute, 'WC_Product_Attribute')) {
                $data['attributes'][] = array(
                    'id' => $attribute->get_id(),
                    'name' => $attribute->get_name(),
                    'options' => $attribute->get_options(),
                    'position' => $attribute->get_position(),
                    'visible' => $attribute->get_visible(),
                    'variation' => $attribute->get_variation(),
                );
            }
        }

        // Variations for variable products
        if ($product->is_type('variable')) {
            $variations = $product->get_children();
            foreach ($variations as $variation_id) {
                $variation = wc_get_product($variation_id);
                if ($variation) {
                    $data['variations'][] = array(
                        'id' => $variation->get_id(),
                        'sku' => $variation->get_sku(),
                        'price' => $variation->get_price(),
                        'regular_price' => $variation->get_regular_price(),
                        'sale_price' => $variation->get_sale_price(),
                        'stock_quantity' => $variation->get_stock_quantity(),
                        'stock_status' => $variation->get_stock_status(),
                        'attributes' => $variation->get_attributes(),
                    );
                }
            }
        }

        return $data;
    }

    /**
     * Export customers to JSON
     *
     * @param string $file_path Output file path
     * @param array $args Export arguments
     * @return bool|WP_Error
     */
    public static function export_customers($file_path, $args = array()) {
        $defaults = array(
            'role' => 'customer',
            'number' => -1,
            'orderby' => 'ID',
            'order' => 'ASC',
        );

        $args = wp_parse_args($args, $defaults);

        $users = get_users($args);

        if (empty($users)) {
            return true;
        }

        $handle = fopen($file_path, 'w');
        if (!$handle) {
            return new WP_Error('export_failed', __('Failed to create export file', 'lox-backup-woocommerce'));
        }

        fwrite($handle, "[\n");
        $first = true;

        foreach ($users as $user) {
            if (!$first) {
                fwrite($handle, ",\n");
            }
            $first = false;

            $customer = new WC_Customer($user->ID);
            $customer_data = self::format_customer($customer);
            fwrite($handle, json_encode($customer_data, JSON_PRETTY_PRINT));
        }

        fwrite($handle, "\n]");
        fclose($handle);

        return true;
    }

    /**
     * Format customer data for export
     *
     * @param WC_Customer $customer
     * @return array
     */
    private static function format_customer($customer) {
        return array(
            'id' => $customer->get_id(),
            'email' => $customer->get_email(),
            'first_name' => $customer->get_first_name(),
            'last_name' => $customer->get_last_name(),
            'username' => $customer->get_username(),
            'date_created' => $customer->get_date_created() ? $customer->get_date_created()->format('Y-m-d H:i:s') : null,
            'date_modified' => $customer->get_date_modified() ? $customer->get_date_modified()->format('Y-m-d H:i:s') : null,
            'is_paying_customer' => $customer->get_is_paying_customer(),
            'order_count' => $customer->get_order_count(),
            'total_spent' => $customer->get_total_spent(),
            'billing' => array(
                'first_name' => $customer->get_billing_first_name(),
                'last_name' => $customer->get_billing_last_name(),
                'company' => $customer->get_billing_company(),
                'address_1' => $customer->get_billing_address_1(),
                'address_2' => $customer->get_billing_address_2(),
                'city' => $customer->get_billing_city(),
                'state' => $customer->get_billing_state(),
                'postcode' => $customer->get_billing_postcode(),
                'country' => $customer->get_billing_country(),
                'email' => $customer->get_billing_email(),
                'phone' => $customer->get_billing_phone(),
            ),
            'shipping' => array(
                'first_name' => $customer->get_shipping_first_name(),
                'last_name' => $customer->get_shipping_last_name(),
                'company' => $customer->get_shipping_company(),
                'address_1' => $customer->get_shipping_address_1(),
                'address_2' => $customer->get_shipping_address_2(),
                'city' => $customer->get_shipping_city(),
                'state' => $customer->get_shipping_state(),
                'postcode' => $customer->get_shipping_postcode(),
                'country' => $customer->get_shipping_country(),
            ),
        );
    }

    /**
     * Export coupons to JSON
     *
     * @param string $file_path Output file path
     * @return bool|WP_Error
     */
    public static function export_coupons($file_path) {
        $coupons = get_posts(array(
            'post_type' => 'shop_coupon',
            'posts_per_page' => -1,
            'post_status' => array('publish', 'pending', 'draft'),
        ));

        if (empty($coupons)) {
            return true;
        }

        $handle = fopen($file_path, 'w');
        if (!$handle) {
            return new WP_Error('export_failed', __('Failed to create export file', 'lox-backup-woocommerce'));
        }

        fwrite($handle, "[\n");
        $first = true;

        foreach ($coupons as $coupon_post) {
            if (!$first) {
                fwrite($handle, ",\n");
            }
            $first = false;

            $coupon = new WC_Coupon($coupon_post->ID);
            $coupon_data = array(
                'id' => $coupon->get_id(),
                'code' => $coupon->get_code(),
                'description' => $coupon->get_description(),
                'discount_type' => $coupon->get_discount_type(),
                'amount' => $coupon->get_amount(),
                'date_expires' => $coupon->get_date_expires() ? $coupon->get_date_expires()->format('Y-m-d H:i:s') : null,
                'date_created' => $coupon->get_date_created() ? $coupon->get_date_created()->format('Y-m-d H:i:s') : null,
                'usage_count' => $coupon->get_usage_count(),
                'individual_use' => $coupon->get_individual_use(),
                'product_ids' => $coupon->get_product_ids(),
                'excluded_product_ids' => $coupon->get_excluded_product_ids(),
                'usage_limit' => $coupon->get_usage_limit(),
                'usage_limit_per_user' => $coupon->get_usage_limit_per_user(),
                'limit_usage_to_x_items' => $coupon->get_limit_usage_to_x_items(),
                'free_shipping' => $coupon->get_free_shipping(),
                'product_categories' => $coupon->get_product_categories(),
                'excluded_product_categories' => $coupon->get_excluded_product_categories(),
                'exclude_sale_items' => $coupon->get_exclude_sale_items(),
                'minimum_amount' => $coupon->get_minimum_amount(),
                'maximum_amount' => $coupon->get_maximum_amount(),
                'email_restrictions' => $coupon->get_email_restrictions(),
            );

            fwrite($handle, json_encode($coupon_data, JSON_PRETTY_PRINT));
        }

        fwrite($handle, "\n]");
        fclose($handle);

        return true;
    }

    /**
     * Export store settings to JSON
     *
     * @param string $file_path Output file path
     * @return bool|WP_Error
     */
    public static function export_settings($file_path) {
        $settings = array(
            'general' => array(
                'woocommerce_store_address' => get_option('woocommerce_store_address'),
                'woocommerce_store_address_2' => get_option('woocommerce_store_address_2'),
                'woocommerce_store_city' => get_option('woocommerce_store_city'),
                'woocommerce_store_postcode' => get_option('woocommerce_store_postcode'),
                'woocommerce_default_country' => get_option('woocommerce_default_country'),
                'woocommerce_currency' => get_option('woocommerce_currency'),
                'woocommerce_currency_pos' => get_option('woocommerce_currency_pos'),
                'woocommerce_price_thousand_sep' => get_option('woocommerce_price_thousand_sep'),
                'woocommerce_price_decimal_sep' => get_option('woocommerce_price_decimal_sep'),
                'woocommerce_price_num_decimals' => get_option('woocommerce_price_num_decimals'),
            ),
            'products' => array(
                'woocommerce_weight_unit' => get_option('woocommerce_weight_unit'),
                'woocommerce_dimension_unit' => get_option('woocommerce_dimension_unit'),
                'woocommerce_manage_stock' => get_option('woocommerce_manage_stock'),
                'woocommerce_stock_format' => get_option('woocommerce_stock_format'),
                'woocommerce_notify_low_stock' => get_option('woocommerce_notify_low_stock'),
                'woocommerce_notify_no_stock' => get_option('woocommerce_notify_no_stock'),
                'woocommerce_low_stock_amount' => get_option('woocommerce_low_stock_amount'),
                'woocommerce_no_stock_amount' => get_option('woocommerce_no_stock_amount'),
            ),
            'tax' => array(
                'woocommerce_calc_taxes' => get_option('woocommerce_calc_taxes'),
                'woocommerce_prices_include_tax' => get_option('woocommerce_prices_include_tax'),
                'woocommerce_tax_based_on' => get_option('woocommerce_tax_based_on'),
                'woocommerce_tax_round_at_subtotal' => get_option('woocommerce_tax_round_at_subtotal'),
                'woocommerce_tax_display_shop' => get_option('woocommerce_tax_display_shop'),
                'woocommerce_tax_display_cart' => get_option('woocommerce_tax_display_cart'),
            ),
            'shipping' => array(
                'woocommerce_ship_to_countries' => get_option('woocommerce_ship_to_countries'),
                'woocommerce_specific_ship_to_countries' => get_option('woocommerce_specific_ship_to_countries'),
                'woocommerce_enable_shipping_calc' => get_option('woocommerce_enable_shipping_calc'),
                'woocommerce_shipping_cost_requires_address' => get_option('woocommerce_shipping_cost_requires_address'),
            ),
            'checkout' => array(
                'woocommerce_enable_guest_checkout' => get_option('woocommerce_enable_guest_checkout'),
                'woocommerce_enable_checkout_login_reminder' => get_option('woocommerce_enable_checkout_login_reminder'),
                'woocommerce_enable_signup_and_login_from_checkout' => get_option('woocommerce_enable_signup_and_login_from_checkout'),
                'woocommerce_enable_myaccount_registration' => get_option('woocommerce_enable_myaccount_registration'),
            ),
            'emails' => array(
                'woocommerce_email_from_name' => get_option('woocommerce_email_from_name'),
                'woocommerce_email_from_address' => get_option('woocommerce_email_from_address'),
                'woocommerce_email_header_image' => get_option('woocommerce_email_header_image'),
                'woocommerce_email_footer_text' => get_option('woocommerce_email_footer_text'),
            ),
        );

        $result = file_put_contents($file_path, json_encode($settings, JSON_PRETTY_PRINT));

        if ($result === false) {
            return new WP_Error('export_failed', __('Failed to export settings', 'lox-backup-woocommerce'));
        }

        return true;
    }
}
