<?php
/**
 * LOX WooCommerce Restore Handler
 *
 * @package LOX_Backup_WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles restoration of WooCommerce data from LOX backups
 */
class LOX_WC_Restore {

    /**
     * Initialize restore hooks
     */
    public static function init() {
        // Hook into LOX restore process
        add_action('lox_backup_restore_complete', array(__CLASS__, 'restore_woocommerce_data'), 10, 2);

        // Add AJAX handlers for WooCommerce restore
        add_action('wp_ajax_lox_wc_restore_preview', array(__CLASS__, 'ajax_restore_preview'));
        add_action('wp_ajax_lox_wc_restore_component', array(__CLASS__, 'ajax_restore_component'));
    }

    /**
     * Restore WooCommerce data after backup restoration
     *
     * @param string $backup_uuid The backup UUID
     * @param string $restore_path Path where backup was extracted
     */
    public static function restore_woocommerce_data($backup_uuid, $restore_path) {
        $wc_path = $restore_path . '/woocommerce';

        if (!is_dir($wc_path)) {
            self::log('No WooCommerce data found in backup');
            return;
        }

        // Check what data is available
        $available_imports = self::scan_available_imports($wc_path);

        if (empty($available_imports)) {
            self::log('No importable WooCommerce files found');
            return;
        }

        self::log('Found WooCommerce data to restore: ' . implode(', ', array_keys($available_imports)));

        // Import in order of dependencies
        $import_order = array('settings', 'customers', 'products', 'coupons', 'orders', 'tax-rates', 'shipping-zones');

        foreach ($import_order as $component) {
            if (isset($available_imports[$component])) {
                $method = 'import_' . str_replace('-', '_', $component);
                if (method_exists(__CLASS__, $method)) {
                    self::log("Importing WooCommerce {$component}...");
                    $result = self::$method($available_imports[$component]);
                    if (is_wp_error($result)) {
                        self::log("Error importing {$component}: " . $result->get_error_message());
                    } else {
                        self::log("Successfully imported {$component}");
                    }
                }
            }
        }
    }

    /**
     * Scan for available import files
     *
     * @param string $path
     * @return array
     */
    private static function scan_available_imports($path) {
        $imports = array();
        $files = array(
            'products' => 'products.json',
            'orders' => 'orders.json',
            'customers' => 'customers.json',
            'coupons' => 'coupons.json',
            'settings' => 'settings.json',
            'tax-rates' => 'tax-rates.json',
            'shipping-zones' => 'shipping-zones.json',
        );

        foreach ($files as $key => $file) {
            $file_path = $path . '/' . $file;
            if (file_exists($file_path)) {
                $imports[$key] = $file_path;
            }
        }

        return $imports;
    }

    /**
     * Import products
     *
     * @param string $file_path
     * @return true|WP_Error
     */
    private static function import_products($file_path) {
        $data = self::read_json_file($file_path);
        if (is_wp_error($data)) {
            return $data;
        }

        $imported = 0;
        $errors = array();

        foreach ($data as $product_data) {
            try {
                // Check if product exists by SKU
                $existing_id = wc_get_product_id_by_sku($product_data['sku'] ?? '');

                if ($existing_id) {
                    $product = wc_get_product($existing_id);
                } else {
                    $type = $product_data['type'] ?? 'simple';
                    $classname = WC_Product_Factory::get_classname_from_product_type($type);
                    $product = new $classname();
                }

                // Set basic product data
                $product->set_name($product_data['name'] ?? '');
                $product->set_slug($product_data['slug'] ?? '');
                $product->set_status($product_data['status'] ?? 'publish');
                $product->set_description($product_data['description'] ?? '');
                $product->set_short_description($product_data['short_description'] ?? '');
                $product->set_sku($product_data['sku'] ?? '');
                $product->set_price($product_data['price'] ?? '');
                $product->set_regular_price($product_data['regular_price'] ?? '');
                $product->set_sale_price($product_data['sale_price'] ?? '');
                $product->set_manage_stock($product_data['manage_stock'] ?? false);
                $product->set_stock_quantity($product_data['stock_quantity'] ?? null);
                $product->set_stock_status($product_data['stock_status'] ?? 'instock');
                $product->set_weight($product_data['weight'] ?? '');
                $product->set_length($product_data['length'] ?? '');
                $product->set_width($product_data['width'] ?? '');
                $product->set_height($product_data['height'] ?? '');

                // Categories
                if (!empty($product_data['category_ids'])) {
                    $product->set_category_ids($product_data['category_ids']);
                }

                // Meta data
                if (!empty($product_data['meta_data'])) {
                    foreach ($product_data['meta_data'] as $meta) {
                        $product->update_meta_data($meta['key'], $meta['value']);
                    }
                }

                $product->save();
                $imported++;

            } catch (Exception $e) {
                $errors[] = sprintf('Product %s: %s', $product_data['sku'] ?? 'unknown', $e->getMessage());
            }
        }

        if (!empty($errors)) {
            self::log('Product import errors: ' . implode('; ', $errors));
        }

        self::log("Imported {$imported} products");
        return true;
    }

    /**
     * Import customers
     *
     * @param string $file_path
     * @return true|WP_Error
     */
    private static function import_customers($file_path) {
        $data = self::read_json_file($file_path);
        if (is_wp_error($data)) {
            return $data;
        }

        $imported = 0;
        $errors = array();

        foreach ($data as $customer_data) {
            try {
                // Check if customer exists by email
                $existing_user = get_user_by('email', $customer_data['email'] ?? '');

                if ($existing_user) {
                    $customer = new WC_Customer($existing_user->ID);
                } else {
                    $customer = new WC_Customer();
                    $customer->set_email($customer_data['email'] ?? '');
                    $customer->set_username($customer_data['username'] ?? $customer_data['email']);
                }

                // Set customer data
                $customer->set_first_name($customer_data['first_name'] ?? '');
                $customer->set_last_name($customer_data['last_name'] ?? '');
                $customer->set_display_name($customer_data['display_name'] ?? '');

                // Billing address
                if (!empty($customer_data['billing'])) {
                    $billing = $customer_data['billing'];
                    $customer->set_billing_first_name($billing['first_name'] ?? '');
                    $customer->set_billing_last_name($billing['last_name'] ?? '');
                    $customer->set_billing_company($billing['company'] ?? '');
                    $customer->set_billing_address_1($billing['address_1'] ?? '');
                    $customer->set_billing_address_2($billing['address_2'] ?? '');
                    $customer->set_billing_city($billing['city'] ?? '');
                    $customer->set_billing_state($billing['state'] ?? '');
                    $customer->set_billing_postcode($billing['postcode'] ?? '');
                    $customer->set_billing_country($billing['country'] ?? '');
                    $customer->set_billing_phone($billing['phone'] ?? '');
                    $customer->set_billing_email($billing['email'] ?? $customer_data['email']);
                }

                // Shipping address
                if (!empty($customer_data['shipping'])) {
                    $shipping = $customer_data['shipping'];
                    $customer->set_shipping_first_name($shipping['first_name'] ?? '');
                    $customer->set_shipping_last_name($shipping['last_name'] ?? '');
                    $customer->set_shipping_company($shipping['company'] ?? '');
                    $customer->set_shipping_address_1($shipping['address_1'] ?? '');
                    $customer->set_shipping_address_2($shipping['address_2'] ?? '');
                    $customer->set_shipping_city($shipping['city'] ?? '');
                    $customer->set_shipping_state($shipping['state'] ?? '');
                    $customer->set_shipping_postcode($shipping['postcode'] ?? '');
                    $customer->set_shipping_country($shipping['country'] ?? '');
                }

                $customer->save();
                $imported++;

            } catch (Exception $e) {
                $errors[] = sprintf('Customer %s: %s', $customer_data['email'] ?? 'unknown', $e->getMessage());
            }
        }

        if (!empty($errors)) {
            self::log('Customer import errors: ' . implode('; ', $errors));
        }

        self::log("Imported {$imported} customers");
        return true;
    }

    /**
     * Import orders
     *
     * @param string $file_path
     * @return true|WP_Error
     */
    private static function import_orders($file_path) {
        $data = self::read_json_file($file_path);
        if (is_wp_error($data)) {
            return $data;
        }

        $imported = 0;
        $errors = array();

        foreach ($data as $order_data) {
            try {
                // Skip if order number already exists
                $existing = wc_get_orders(array(
                    'order_number' => $order_data['number'] ?? '',
                    'limit' => 1,
                ));

                if (!empty($existing)) {
                    continue;
                }

                $order = wc_create_order(array(
                    'status' => $order_data['status'] ?? 'pending',
                    'customer_id' => $order_data['customer_id'] ?? 0,
                ));

                if (is_wp_error($order)) {
                    throw new Exception($order->get_error_message());
                }

                // Set billing address
                if (!empty($order_data['billing'])) {
                    $order->set_address($order_data['billing'], 'billing');
                }

                // Set shipping address
                if (!empty($order_data['shipping'])) {
                    $order->set_address($order_data['shipping'], 'shipping');
                }

                // Add line items
                if (!empty($order_data['line_items'])) {
                    foreach ($order_data['line_items'] as $item_data) {
                        $product = null;
                        if (!empty($item_data['sku'])) {
                            $product_id = wc_get_product_id_by_sku($item_data['sku']);
                            if ($product_id) {
                                $product = wc_get_product($product_id);
                            }
                        }

                        if ($product) {
                            $order->add_product($product, $item_data['quantity'] ?? 1);
                        } else {
                            // Add as a custom line item
                            $item = new WC_Order_Item_Product();
                            $item->set_name($item_data['name'] ?? 'Product');
                            $item->set_quantity($item_data['quantity'] ?? 1);
                            $item->set_subtotal($item_data['subtotal'] ?? 0);
                            $item->set_total($item_data['total'] ?? 0);
                            $order->add_item($item);
                        }
                    }
                }

                // Set order totals
                $order->set_total($order_data['total'] ?? 0);
                $order->set_currency($order_data['currency'] ?? get_woocommerce_currency());
                $order->set_payment_method($order_data['payment_method'] ?? '');
                $order->set_payment_method_title($order_data['payment_method_title'] ?? '');

                // Set dates
                if (!empty($order_data['date_created'])) {
                    $order->set_date_created($order_data['date_created']);
                }
                if (!empty($order_data['date_paid'])) {
                    $order->set_date_paid($order_data['date_paid']);
                }

                // Add order notes
                if (!empty($order_data['customer_note'])) {
                    $order->set_customer_note($order_data['customer_note']);
                }

                // Meta data
                if (!empty($order_data['meta_data'])) {
                    foreach ($order_data['meta_data'] as $meta) {
                        $order->update_meta_data($meta['key'], $meta['value']);
                    }
                }

                $order->save();
                $imported++;

            } catch (Exception $e) {
                $errors[] = sprintf('Order %s: %s', $order_data['number'] ?? 'unknown', $e->getMessage());
            }
        }

        if (!empty($errors)) {
            self::log('Order import errors: ' . implode('; ', $errors));
        }

        self::log("Imported {$imported} orders");
        return true;
    }

    /**
     * Import coupons
     *
     * @param string $file_path
     * @return true|WP_Error
     */
    private static function import_coupons($file_path) {
        $data = self::read_json_file($file_path);
        if (is_wp_error($data)) {
            return $data;
        }

        $imported = 0;

        foreach ($data as $coupon_data) {
            // Check if coupon code exists
            $existing = wc_get_coupon_id_by_code($coupon_data['code'] ?? '');

            if ($existing) {
                $coupon = new WC_Coupon($existing);
            } else {
                $coupon = new WC_Coupon();
            }

            $coupon->set_code($coupon_data['code'] ?? '');
            $coupon->set_description($coupon_data['description'] ?? '');
            $coupon->set_discount_type($coupon_data['discount_type'] ?? 'fixed_cart');
            $coupon->set_amount($coupon_data['amount'] ?? 0);
            $coupon->set_free_shipping($coupon_data['free_shipping'] ?? false);
            $coupon->set_date_expires($coupon_data['date_expires'] ?? null);
            $coupon->set_minimum_amount($coupon_data['minimum_amount'] ?? '');
            $coupon->set_maximum_amount($coupon_data['maximum_amount'] ?? '');
            $coupon->set_usage_limit($coupon_data['usage_limit'] ?? null);
            $coupon->set_usage_limit_per_user($coupon_data['usage_limit_per_user'] ?? null);
            $coupon->set_individual_use($coupon_data['individual_use'] ?? false);
            $coupon->set_exclude_sale_items($coupon_data['exclude_sale_items'] ?? false);

            if (!empty($coupon_data['product_ids'])) {
                $coupon->set_product_ids($coupon_data['product_ids']);
            }
            if (!empty($coupon_data['excluded_product_ids'])) {
                $coupon->set_excluded_product_ids($coupon_data['excluded_product_ids']);
            }

            $coupon->save();
            $imported++;
        }

        self::log("Imported {$imported} coupons");
        return true;
    }

    /**
     * Import settings
     *
     * @param string $file_path
     * @return true|WP_Error
     */
    private static function import_settings($file_path) {
        $data = self::read_json_file($file_path);
        if (is_wp_error($data)) {
            return $data;
        }

        $imported = 0;

        foreach ($data as $key => $value) {
            // Skip sensitive settings
            $skip_settings = array(
                'woocommerce_stripe_settings',
                'woocommerce_paypal_settings',
                '_transient',
            );

            $should_skip = false;
            foreach ($skip_settings as $skip) {
                if (strpos($key, $skip) !== false) {
                    $should_skip = true;
                    break;
                }
            }

            if ($should_skip) {
                continue;
            }

            update_option($key, $value);
            $imported++;
        }

        self::log("Imported {$imported} settings");
        return true;
    }

    /**
     * Import tax rates
     *
     * @param string $file_path
     * @return true|WP_Error
     */
    private static function import_tax_rates($file_path) {
        global $wpdb;

        $data = self::read_json_file($file_path);
        if (is_wp_error($data)) {
            return $data;
        }

        $imported = 0;

        foreach ($data as $rate_data) {
            // Check if rate exists
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT tax_rate_id FROM {$wpdb->prefix}woocommerce_tax_rates
                WHERE tax_rate_country = %s AND tax_rate_state = %s AND tax_rate_class = %s AND tax_rate_name = %s",
                $rate_data['tax_rate_country'] ?? '',
                $rate_data['tax_rate_state'] ?? '',
                $rate_data['tax_rate_class'] ?? '',
                $rate_data['tax_rate_name'] ?? ''
            ));

            if ($existing) {
                continue;
            }

            $wpdb->insert(
                $wpdb->prefix . 'woocommerce_tax_rates',
                array(
                    'tax_rate_country' => $rate_data['tax_rate_country'] ?? '',
                    'tax_rate_state' => $rate_data['tax_rate_state'] ?? '',
                    'tax_rate' => $rate_data['tax_rate'] ?? 0,
                    'tax_rate_name' => $rate_data['tax_rate_name'] ?? '',
                    'tax_rate_priority' => $rate_data['tax_rate_priority'] ?? 1,
                    'tax_rate_compound' => $rate_data['tax_rate_compound'] ?? 0,
                    'tax_rate_shipping' => $rate_data['tax_rate_shipping'] ?? 1,
                    'tax_rate_order' => $rate_data['tax_rate_order'] ?? 0,
                    'tax_rate_class' => $rate_data['tax_rate_class'] ?? '',
                )
            );

            // Add postcodes if any
            if (!empty($rate_data['postcodes'])) {
                $rate_id = $wpdb->insert_id;
                $postcodes = explode(',', $rate_data['postcodes']);
                foreach ($postcodes as $postcode) {
                    $wpdb->insert(
                        $wpdb->prefix . 'woocommerce_tax_rate_locations',
                        array(
                            'tax_rate_id' => $rate_id,
                            'location_code' => trim($postcode),
                            'location_type' => 'postcode',
                        )
                    );
                }
            }

            $imported++;
        }

        self::log("Imported {$imported} tax rates");
        return true;
    }

    /**
     * Import shipping zones
     *
     * @param string $file_path
     * @return true|WP_Error
     */
    private static function import_shipping_zones($file_path) {
        $data = self::read_json_file($file_path);
        if (is_wp_error($data)) {
            return $data;
        }

        $imported = 0;

        foreach ($data as $zone_data) {
            // Check if zone with same name exists
            $existing_zones = WC_Shipping_Zones::get_zones();
            $exists = false;
            foreach ($existing_zones as $existing) {
                if ($existing['zone_name'] === $zone_data['name']) {
                    $exists = true;
                    break;
                }
            }

            if ($exists) {
                continue;
            }

            $zone = new WC_Shipping_Zone();
            $zone->set_zone_name($zone_data['name']);
            $zone->set_zone_order($zone_data['order'] ?? 0);

            // Add locations
            if (!empty($zone_data['locations'])) {
                foreach ($zone_data['locations'] as $location) {
                    $zone->add_location($location['code'], $location['type']);
                }
            }

            $zone->save();

            // Add shipping methods
            if (!empty($zone_data['methods'])) {
                foreach ($zone_data['methods'] as $method_data) {
                    $instance_id = $zone->add_shipping_method($method_data['method_id']);

                    if ($instance_id) {
                        $method = WC_Shipping_Zones::get_shipping_method($instance_id);
                        if ($method && !empty($method_data['settings'])) {
                            foreach ($method_data['settings'] as $key => $value) {
                                $method->instance_settings[$key] = $value;
                            }
                            update_option($method->get_instance_option_key(), $method->instance_settings);
                        }
                    }
                }
            }

            $imported++;
        }

        self::log("Imported {$imported} shipping zones");
        return true;
    }

    /**
     * Read and parse JSON file
     *
     * @param string $file_path
     * @return array|WP_Error
     */
    private static function read_json_file($file_path) {
        if (!file_exists($file_path)) {
            return new WP_Error('file_not_found', "File not found: {$file_path}");
        }

        $content = file_get_contents($file_path);
        if ($content === false) {
            return new WP_Error('read_error', "Cannot read file: {$file_path}");
        }

        $data = json_decode($content, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', 'Invalid JSON: ' . json_last_error_msg());
        }

        return $data;
    }

    /**
     * AJAX: Get restore preview
     */
    public static function ajax_restore_preview() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup-woocommerce')));
        }

        $backup_path = sanitize_text_field($_POST['backup_path'] ?? '');
        $wc_path = $backup_path . '/woocommerce';

        if (!is_dir($wc_path)) {
            wp_send_json_error(array('message' => __('No WooCommerce data in this backup', 'lox-backup-woocommerce')));
        }

        $preview = array();
        $files = array(
            'products' => array('file' => 'products.json', 'label' => __('Products', 'lox-backup-woocommerce')),
            'orders' => array('file' => 'orders.json', 'label' => __('Orders', 'lox-backup-woocommerce')),
            'customers' => array('file' => 'customers.json', 'label' => __('Customers', 'lox-backup-woocommerce')),
            'coupons' => array('file' => 'coupons.json', 'label' => __('Coupons', 'lox-backup-woocommerce')),
            'settings' => array('file' => 'settings.json', 'label' => __('Settings', 'lox-backup-woocommerce')),
        );

        foreach ($files as $key => $info) {
            $file_path = $wc_path . '/' . $info['file'];
            if (file_exists($file_path)) {
                $data = self::read_json_file($file_path);
                $preview[$key] = array(
                    'label' => $info['label'],
                    'count' => is_array($data) ? count($data) : 0,
                    'available' => true,
                );
            }
        }

        wp_send_json_success(array('components' => $preview));
    }

    /**
     * Log message
     *
     * @param string $message
     */
    private static function log($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[LOX WooCommerce Restore] ' . $message);
        }
    }
}
