<?php
/**
 * LOX Backup Admin
 *
 * @package LOX_Backup
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Admin interface for LOX Backup
 */
class LOX_Backup_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_ajax_lox_backup_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_lox_backup_run_now', array($this, 'ajax_run_backup'));
        add_action('wp_ajax_lox_backup_list', array($this, 'ajax_list_backups'));
        add_action('wp_ajax_lox_backup_restore', array($this, 'ajax_restore_backup'));
        add_action('wp_ajax_lox_backup_download', array($this, 'ajax_download_backup'));
        add_action('wp_ajax_lox_backup_run_custom', array($this, 'ajax_run_custom_backup'));
        add_action('wp_ajax_lox_backup_save_profile', array($this, 'ajax_save_profile'));
        add_action('wp_ajax_lox_backup_delete_profile', array($this, 'ajax_delete_profile'));
        add_action('wp_ajax_lox_backup_run_profile', array($this, 'ajax_run_profile'));
        add_action('wp_ajax_lox_backup_get_remote_profiles', array($this, 'ajax_get_remote_profiles'));
        add_action('admin_notices', array($this, 'admin_notices'));
        add_action('admin_footer', array($this, 'render_qa_widget'));

        // Profile cron hooks
        add_action('lox_backup_profile_cron', array($this, 'run_profile_cron'), 10, 1);
    }

    /**
     * Add admin menu
     */
    public function add_menu() {
        add_menu_page(
            __('LOX Backup', 'lox-backup'),
            __('LOX Backup', 'lox-backup'),
            'manage_options',
            'lox-backup',
            array($this, 'render_settings_page'),
            'dashicons-shield',
            80
        );

        add_submenu_page(
            'lox-backup',
            __('Settings', 'lox-backup'),
            __('Settings', 'lox-backup'),
            'manage_options',
            'lox-backup',
            array($this, 'render_settings_page')
        );

        add_submenu_page(
            'lox-backup',
            __('Backups', 'lox-backup'),
            __('Backups', 'lox-backup'),
            'manage_options',
            'lox-backup-list',
            array($this, 'render_backups_page')
        );

        add_submenu_page(
            'lox-backup',
            __('Schedules', 'lox-backup'),
            __('Schedules', 'lox-backup'),
            'manage_options',
            'lox-backup-schedules',
            array($this, 'render_schedules_page')
        );

        add_submenu_page(
            'lox-backup',
            __('Custom Profiles', 'lox-backup'),
            __('Custom Profiles', 'lox-backup'),
            'manage_options',
            'lox-backup-profiles',
            array($this, 'render_profiles_page')
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('lox_backup_settings_group', 'lox_backup_settings', array(
            'sanitize_callback' => array($this, 'sanitize_settings'),
        ));

        register_setting('lox_backup_schedules_group', 'lox_backup_schedules', array(
            'sanitize_callback' => array($this, 'sanitize_schedules'),
        ));

        // API Settings Section
        add_settings_section(
            'lox_api_settings',
            __('API Configuration', 'lox-backup'),
            array($this, 'render_api_section'),
            'lox-backup'
        );

        add_settings_field(
            'api_key',
            __('API Key', 'lox-backup'),
            array($this, 'render_api_key_field'),
            'lox-backup',
            'lox_api_settings'
        );

        add_settings_field(
            'base_url',
            __('API URL', 'lox-backup'),
            array($this, 'render_base_url_field'),
            'lox-backup',
            'lox_api_settings'
        );

        // Backup Settings Section
        add_settings_section(
            'lox_backup_settings',
            __('Backup Configuration', 'lox-backup'),
            array($this, 'render_backup_section'),
            'lox-backup'
        );

        add_settings_field(
            'backup_components',
            __('Backup Components', 'lox-backup'),
            array($this, 'render_components_field'),
            'lox-backup',
            'lox_backup_settings'
        );

        add_settings_field(
            'retention_days',
            __('Retention Days', 'lox-backup'),
            array($this, 'render_retention_field'),
            'lox-backup',
            'lox_backup_settings'
        );

        add_settings_field(
            'immutable_days',
            __('Immutable Period (days)', 'lox-backup'),
            array($this, 'render_immutable_days_field'),
            'lox-backup',
            'lox_backup_settings'
        );

        add_settings_field(
            'tags',
            __('Backup Tags', 'lox-backup'),
            array($this, 'render_tags_field'),
            'lox-backup',
            'lox_backup_settings'
        );
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();

        $sanitized['api_key'] = sanitize_text_field($input['api_key'] ?? '');
        $sanitized['base_url'] = esc_url_raw($input['base_url'] ?? 'https://backlox.com/api');
        $sanitized['backup_database'] = !empty($input['backup_database']);
        $sanitized['backup_uploads'] = !empty($input['backup_uploads']);
        $sanitized['backup_plugins'] = !empty($input['backup_plugins']);
        $sanitized['backup_themes'] = !empty($input['backup_themes']);
        $sanitized['retention_days'] = absint($input['retention_days'] ?? 30);
        // Immutable days: empty/blank = use server default, 0 = no immutability, >0 = immutable for N days
        $sanitized['immutable_days'] = isset($input['immutable_days']) && $input['immutable_days'] !== ''
            ? absint($input['immutable_days'])
            : null;
        $sanitized['tags'] = sanitize_text_field($input['tags'] ?? 'wordpress');

        // Preserve status fields
        $current = get_option('lox_backup_settings', array());
        $sanitized['last_backup'] = $current['last_backup'] ?? null;
        $sanitized['last_backup_status'] = $current['last_backup_status'] ?? null;
        $sanitized['last_backup_message'] = $current['last_backup_message'] ?? null;
        $sanitized['last_backup_uuid'] = $current['last_backup_uuid'] ?? null;
        $sanitized['last_backup_size'] = $current['last_backup_size'] ?? null;

        return $sanitized;
    }

    /**
     * Sanitize schedules
     */
    public function sanitize_schedules($input) {
        $sanitized = array();
        $components = array('database', 'uploads', 'plugins', 'themes', 'full');

        foreach ($components as $component) {
            $key = "schedule_{$component}";
            $sanitized[$key] = array(
                'enabled' => !empty($input[$key]['enabled']),
                'frequency' => sanitize_text_field($input[$key]['frequency'] ?? 'disabled'),
                'time' => sanitize_text_field($input[$key]['time'] ?? '03:00'),
                'day' => absint($input[$key]['day'] ?? 0),
            );
        }

        // Update cron jobs
        $this->update_cron_schedules($sanitized);

        return $sanitized;
    }

    /**
     * Update cron schedules
     */
    private function update_cron_schedules($schedules) {
        // Clear all existing LOX cron jobs
        $crons = _get_cron_array();
        foreach ($crons as $timestamp => $cron) {
            foreach ($cron as $hook => $events) {
                if (strpos($hook, 'lox_backup_scheduled_') === 0) {
                    wp_clear_scheduled_hook($hook);
                }
            }
        }

        // Schedule new jobs
        foreach ($schedules as $key => $schedule) {
            if (!$schedule['enabled'] || $schedule['frequency'] === 'disabled') {
                continue;
            }

            $component = str_replace('schedule_', '', $key);
            $hook = "lox_backup_scheduled_{$component}";

            // Calculate next run time
            $time_parts = explode(':', $schedule['time']);
            $hour = intval($time_parts[0]);
            $minute = intval($time_parts[1] ?? 0);

            $next_run = strtotime("today {$hour}:{$minute}");
            if ($next_run < time()) {
                $next_run = strtotime("tomorrow {$hour}:{$minute}");
            }

            // Adjust for weekly schedules
            if ($schedule['frequency'] === 'weekly' && $schedule['day'] > 0) {
                $days = array('', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday');
                $day_name = $days[$schedule['day']] ?? 'monday';
                $next_run = strtotime("next {$day_name} {$hour}:{$minute}");
            }

            wp_schedule_event($next_run, $schedule['frequency'], $hook);
        }
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'lox-backup') === false) {
            return;
        }

        wp_enqueue_style(
            'lox-backup-admin',
            LOX_BACKUP_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            LOX_BACKUP_VERSION
        );

        wp_enqueue_script(
            'lox-backup-admin',
            LOX_BACKUP_PLUGIN_URL . 'admin/js/admin.js',
            array('jquery'),
            LOX_BACKUP_VERSION,
            true
        );

        wp_localize_script('lox-backup-admin', 'loxBackup', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('lox_backup_nonce'),
            'strings' => array(
                'testing' => __('Testing connection...', 'lox-backup'),
                'success' => __('Connection successful!', 'lox-backup'),
                'error' => __('Connection failed', 'lox-backup'),
                'backing_up' => __('Running backup...', 'lox-backup'),
                'backup_complete' => __('Backup completed!', 'lox-backup'),
                'backup_failed' => __('Backup failed', 'lox-backup'),
                'loading' => __('Loading...', 'lox-backup'),
                'restoring' => __('Requesting restore...', 'lox-backup'),
                'restore_requested' => __('Restore requested!', 'lox-backup'),
                'confirm_restore' => __('Are you sure you want to restore this backup?', 'lox-backup'),
                'confirm_backup' => __('Start backup now? This may take several minutes.', 'lox-backup'),
                'restore_confirmation_text' => __('RESTORE', 'lox-backup'),
                'restore_confirmation_error' => __('Please type RESTORE to confirm', 'lox-backup'),
                'restore_warning' => __('This will overwrite your current site data. This action cannot be undone.', 'lox-backup'),
            ),
        ));
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $settings = get_option('lox_backup_settings', array());
        ?>
        <div class="wrap lox-backup-admin">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="lox-backup-header">
                <div class="lox-backup-logo">
                    <span class="dashicons dashicons-shield"></span>
                    <span><?php _e('LOX Cold Backup', 'lox-backup'); ?></span>
                </div>
            </div>

            <?php $this->render_status_card($settings); ?>

            <form method="post" action="options.php">
                <?php
                settings_fields('lox_backup_settings_group');
                do_settings_sections('lox-backup');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Render backups list page
     */
    public function render_backups_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap lox-backup-admin">
            <h1><?php _e('Backups', 'lox-backup'); ?></h1>

            <!-- Toast Container -->
            <div id="lox-toast-container" class="lox-toast-container"></div>

            <div class="lox-backup-header">
                <div class="lox-backup-logo">
                    <span class="dashicons dashicons-shield"></span>
                    <span><?php _e('LOX Cold Backup', 'lox-backup'); ?></span>
                </div>
            </div>

            <div class="lox-backups-toolbar">
                <button type="button" class="button button-primary" id="lox-run-backup">
                    <span class="dashicons dashicons-backup"></span>
                    <?php _e('Create Backup', 'lox-backup'); ?>
                </button>
                <button type="button" class="button" id="lox-refresh-backups">
                    <span class="dashicons dashicons-update"></span>
                    <?php _e('Refresh', 'lox-backup'); ?>
                </button>
            </div>

            <!-- Tabs -->
            <div class="lox-tabs">
                <button type="button" class="lox-tab active" data-tab="profiles">
                    <span class="dashicons dashicons-portfolio"></span>
                    <?php _e('Profiles', 'lox-backup'); ?>
                </button>
                <button type="button" class="lox-tab" data-tab="activity">
                    <span class="dashicons dashicons-backup"></span>
                    <?php _e('Activity', 'lox-backup'); ?>
                </button>
            </div>

            <div id="lox-backup-message" class="notice" style="display:none;"></div>

            <!-- Profiles Tab Content -->
            <div id="lox-profiles-tab" class="lox-tab-content active">
                <div id="lox-profiles-loading" class="lox-loading">
                    <span class="spinner is-active"></span>
                    <?php _e('Loading profiles...', 'lox-backup'); ?>
                </div>

                <div id="lox-profiles-grid" class="lox-profiles-grid lox-hidden"></div>

                <div id="lox-profiles-empty" class="lox-empty-state lox-hidden">
                    <span class="dashicons dashicons-portfolio"></span>
                    <h3><?php _e('No backup profiles yet', 'lox-backup'); ?></h3>
                    <p><?php _e('Profiles are automatically created when you run backups. Each profile groups related backups with versioning.', 'lox-backup'); ?></p>
                </div>
            </div>

            <!-- Activity Tab Content -->
            <div id="lox-activity-tab" class="lox-tab-content lox-hidden">
                <div id="lox-backups-loading" class="lox-loading">
                    <span class="spinner is-active"></span>
                    <?php _e('Loading backups...', 'lox-backup'); ?>
                </div>

                <table class="wp-list-table widefat fixed striped lox-hidden" id="lox-backups-table">
                    <thead>
                        <tr>
                            <th><?php _e('Name', 'lox-backup'); ?></th>
                            <th><?php _e('Version', 'lox-backup'); ?></th>
                            <th><?php _e('Component', 'lox-backup'); ?></th>
                            <th><?php _e('Size', 'lox-backup'); ?></th>
                            <th><?php _e('Status', 'lox-backup'); ?></th>
                            <th><?php _e('Created', 'lox-backup'); ?></th>
                            <th><?php _e('Actions', 'lox-backup'); ?></th>
                        </tr>
                    </thead>
                    <tbody id="lox-backups-list">
                    </tbody>
                </table>

                <div id="lox-backups-empty" class="lox-empty-state lox-hidden">
                    <span class="dashicons dashicons-backup"></span>
                    <h3><?php _e('No backups yet', 'lox-backup'); ?></h3>
                    <p><?php _e('Create your first backup to get started.', 'lox-backup'); ?></p>
                </div>

                <div id="lox-backups-pagination" class="tablenav bottom lox-hidden">
                <div class="tablenav-pages">
                    <span class="displaying-num"></span>
                    <span class="pagination-links">
                        <button class="button prev-page" disabled>&lsaquo;</button>
                        <span class="paging-input">
                            <span class="current-page"></span> / <span class="total-pages"></span>
                        </span>
                        <button class="button next-page">&rsaquo;</button>
                    </span>
                </div>
            </div>
            </div><!-- End Activity Tab -->
        </div>

        <!-- Restore Confirmation Modal -->
        <div id="lox-restore-modal" class="lox-modal" style="display:none;">
            <div class="lox-modal-backdrop"></div>
            <div class="lox-modal-content lox-restore-modal-content">
                <div class="lox-modal-header lox-restore-modal-header">
                    <div class="lox-restore-icon">
                        <span class="dashicons dashicons-warning"></span>
                    </div>
                    <h3><?php _e('Confirm Restore', 'lox-backup'); ?></h3>
                    <button type="button" class="lox-modal-close">&times;</button>
                </div>
                <div class="lox-modal-body">
                    <input type="hidden" id="restore-backup-uuid" value="">

                    <div class="lox-restore-warning">
                        <p><strong><?php _e('Warning: This action is irreversible!', 'lox-backup'); ?></strong></p>
                        <p><?php _e('This will overwrite your current site data. This action cannot be undone.', 'lox-backup'); ?></p>
                    </div>

                    <div class="lox-restore-details">
                        <p><strong><?php _e('Backup to restore:', 'lox-backup'); ?></strong></p>
                        <div id="restore-backup-info" class="lox-backup-info-box">
                            <span class="backup-name"></span>
                            <span class="backup-date"></span>
                        </div>
                    </div>

                    <div class="lox-form-group lox-restore-confirm-group">
                        <label for="restore-confirmation-input">
                            <?php printf(__('To confirm, type %s below:', 'lox-backup'), '<strong>RESTORE</strong>'); ?>
                        </label>
                        <input type="text" id="restore-confirmation-input" class="regular-text" placeholder="<?php _e('Type RESTORE to confirm', 'lox-backup'); ?>" autocomplete="off">
                        <p class="lox-error-message" id="restore-confirmation-error" style="display:none;">
                            <?php _e('Please type RESTORE to confirm', 'lox-backup'); ?>
                        </p>
                    </div>
                </div>
                <div class="lox-modal-footer">
                    <button type="button" class="button lox-modal-cancel"><?php _e('Cancel', 'lox-backup'); ?></button>
                    <button type="button" class="button button-primary button-danger" id="lox-confirm-restore" disabled>
                        <span class="dashicons dashicons-backup"></span>
                        <?php _e('Restore Backup', 'lox-backup'); ?>
                    </button>
                </div>
            </div>
        </div>

        <style>
        /* Base Modal Styles */
        .lox-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            z-index: 100000;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .lox-modal-backdrop {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
        }
        .lox-modal-content {
            position: relative;
            width: 90%;
            max-width: 600px;
            max-height: 90vh;
            background: #fff;
            border-radius: var(--lox-radius, 12px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 8px 10px -6px rgba(0, 0, 0, 0.1);
            display: flex;
            flex-direction: column;
        }
        .lox-modal-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 20px 24px;
            border-bottom: 1px solid #e5e7eb;
        }
        .lox-modal-header h3 {
            margin: 0;
            font-size: 18px;
            font-weight: 600;
        }
        .lox-modal-close {
            background: none;
            border: none;
            font-size: 28px;
            color: #9ca3af;
            cursor: pointer;
            padding: 0;
            line-height: 1;
        }
        .lox-modal-close:hover {
            color: #4b5563;
        }
        .lox-modal-body {
            padding: 24px;
            overflow-y: auto;
        }
        .lox-modal-footer {
            display: flex;
            justify-content: flex-end;
            gap: 12px;
            padding: 16px 24px;
            border-top: 1px solid #e5e7eb;
            background: #fff;
            border-radius: 0 0 var(--lox-radius, 12px) var(--lox-radius, 12px);
        }

        /* Restore Modal Specific Styles */
        .lox-restore-modal-content {
            max-width: 480px;
        }
        .lox-restore-modal-header {
            background: linear-gradient(135deg, #dc2626 0%, #ef4444 100%);
            color: #fff;
            border-radius: var(--lox-radius, 12px) var(--lox-radius, 12px) 0 0;
        }
        .lox-restore-modal-header h3 {
            color: #fff;
        }
        .lox-restore-modal-header .lox-modal-close {
            color: rgba(255,255,255,0.8);
        }
        .lox-restore-modal-header .lox-modal-close:hover {
            color: #fff;
        }
        .lox-restore-icon {
            width: 48px;
            height: 48px;
            background: rgba(255,255,255,0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
        }
        .lox-restore-icon .dashicons {
            font-size: 24px;
            width: 24px;
            height: 24px;
        }
        .lox-restore-warning {
            background: #fef2f2;
            border: 1px solid #fecaca;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 20px;
        }
        .lox-restore-warning p {
            margin: 0 0 8px 0;
            color: #991b1b;
        }
        .lox-restore-warning p:last-child {
            margin-bottom: 0;
        }
        .lox-restore-details {
            margin-bottom: 20px;
        }
        .lox-backup-info-box {
            background: #f3f4f6;
            border-radius: 8px;
            padding: 12px 16px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .lox-backup-info-box .backup-name {
            font-weight: 600;
            color: #374151;
        }
        .lox-backup-info-box .backup-date {
            color: #6b7280;
            font-size: 13px;
        }
        .lox-restore-confirm-group label {
            display: block;
            margin-bottom: 8px;
            color: #374151;
        }
        .lox-restore-confirm-group input {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #d1d5db;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.2s;
        }
        .lox-restore-confirm-group input:focus {
            outline: none;
            border-color: #dc2626;
        }
        .lox-restore-confirm-group input.valid {
            border-color: #10b981;
            background: #f0fdf4;
        }
        .lox-error-message {
            color: #dc2626;
            font-size: 13px;
            margin-top: 8px;
        }
        .button-danger {
            background: #dc2626 !important;
            border-color: #b91c1c !important;
            color: #fff !important;
        }
        .button-danger:hover:not(:disabled) {
            background: #b91c1c !important;
        }
        .button-danger:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        .button-danger .dashicons {
            margin-right: 4px;
        }
        </style>

        <script>
        jQuery(document).ready(function($) {
            var currentPage = 1;
            var totalPages = 1;

            function loadBackups(page) {
                page = page || 1;
                $('#lox-backups-loading').show();
                $('#lox-backups-table, #lox-backups-empty, #lox-backups-pagination').hide();

                $.post(loxBackup.ajax_url, {
                    action: 'lox_backup_list',
                    nonce: loxBackup.nonce,
                    page: page
                }, function(response) {
                    $('#lox-backups-loading').hide();

                    if (response.success && response.data.items.length > 0) {
                        var html = '';
                        response.data.items.forEach(function(backup) {
                            var statusUpper = (backup.status || '').toUpperCase();
                            var statusClass = statusUpper === 'COMPLETED' ? 'success' :
                                             (statusUpper === 'FAILED' ? 'error' : 'warning');
                            var versionText = backup.version ? 'v' + backup.version : '-';
                            html += '<tr data-uuid="' + backup.uuid + '">';
                            html += '<td><strong>' + backup.name + '</strong><br><small class="lox-uuid">' + backup.uuid + '</small></td>';
                            html += '<td><span class="lox-version-badge">' + versionText + '</span></td>';
                            html += '<td><span class="lox-component-badge">' + (backup.component || 'full') + '</span></td>';
                            html += '<td>' + formatBytes(backup.size_bytes) + '</td>';
                            html += '<td><span class="lox-status ' + statusClass + '">' + backup.status + '</span></td>';
                            html += '<td>' + formatDate(backup.created_at) + '</td>';
                            html += '<td class="lox-actions">';
                            if (statusUpper === 'COMPLETED') {
                                html += '<button class="button button-small lox-download-btn" data-uuid="' + backup.uuid + '">';
                                html += '<span class="dashicons dashicons-download"></span> <?php _e('Download', 'lox-backup'); ?>';
                                html += '</button> ';
                                html += '<button class="button button-small lox-restore-btn" data-uuid="' + backup.uuid + '">';
                                html += '<span class="dashicons dashicons-backup"></span> <?php _e('Restore', 'lox-backup'); ?>';
                                html += '</button>';
                            } else {
                                html += '<span class="lox-actions-pending"><?php _e('Pending...', 'lox-backup'); ?></span>';
                            }
                            html += '</td>';
                            html += '</tr>';
                        });
                        $('#lox-backups-list').html(html);
                        $('#lox-backups-table').show();

                        // Pagination
                        currentPage = response.data.page;
                        totalPages = Math.ceil(response.data.total / response.data.page_size);
                        if (totalPages > 1) {
                            $('.displaying-num').text(response.data.total + ' <?php _e('items', 'lox-backup'); ?>');
                            $('.current-page').text(currentPage);
                            $('.total-pages').text(totalPages);
                            $('.prev-page').prop('disabled', currentPage <= 1);
                            $('.next-page').prop('disabled', currentPage >= totalPages);
                            $('#lox-backups-pagination').show();
                        }
                    } else {
                        $('#lox-backups-empty').show();
                    }
                }).fail(function() {
                    $('#lox-backups-loading').hide();
                    showMessage('error', '<?php _e('Failed to load backups', 'lox-backup'); ?>');
                });
            }

            function formatBytes(bytes) {
                if (bytes === 0) return '0 B';
                var k = 1024;
                var sizes = ['B', 'KB', 'MB', 'GB'];
                var i = Math.floor(Math.log(bytes) / Math.log(k));
                return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
            }

            function formatDate(dateStr) {
                var date = new Date(dateStr);
                return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
            }

            function showMessage(type, message, title) {
                // Map notice types to toast types
                var toastType = type === 'success' ? 'success' :
                               (type === 'error' ? 'error' :
                               (type === 'warning' ? 'warning' : 'info'));

                // Default titles based on type
                var defaultTitles = {
                    success: '<?php _e('Success', 'lox-backup'); ?>',
                    error: '<?php _e('Error', 'lox-backup'); ?>',
                    warning: '<?php _e('Warning', 'lox-backup'); ?>',
                    info: '<?php _e('Info', 'lox-backup'); ?>'
                };

                // Icons based on type
                var icons = {
                    success: 'dashicons-yes-alt',
                    error: 'dashicons-dismiss',
                    warning: 'dashicons-warning',
                    info: 'dashicons-info'
                };

                var toastTitle = title || defaultTitles[toastType];
                var icon = icons[toastType];

                // Create toast element
                var $toast = $('<div class="lox-toast ' + toastType + '">' +
                    '<div class="lox-toast-icon"><span class="dashicons ' + icon + '"></span></div>' +
                    '<div class="lox-toast-content">' +
                        '<div class="lox-toast-title">' + toastTitle + '</div>' +
                        '<div class="lox-toast-message">' + message + '</div>' +
                    '</div>' +
                    '<button class="lox-toast-close">&times;</button>' +
                '</div>');

                // Add to container
                $('#lox-toast-container').append($toast);

                // Trigger animation
                setTimeout(function() {
                    $toast.addClass('show');
                }, 10);

                // Close button
                $toast.find('.lox-toast-close').on('click', function() {
                    closeToast($toast);
                });

                // Auto close after 5 seconds
                setTimeout(function() {
                    closeToast($toast);
                }, 5000);

                function closeToast($el) {
                    $el.removeClass('show').addClass('hide');
                    setTimeout(function() {
                        $el.remove();
                    }, 300);
                }

                // Also update the old message area for accessibility
                var $msg = $('#lox-backup-message');
                $msg.removeClass('notice-success notice-error notice-warning')
                    .addClass('notice-' + type)
                    .html('<p>' + message + '</p>')
                    .show();
            }

            // Initial load - only if activity tab is visible (not using new tabbed interface)
            if ($('#lox-activity-tab').length === 0 || $('#lox-activity-tab').is(':visible')) {
                loadBackups();
            }

            // Refresh button
            $('#lox-refresh-backups').on('click', function() {
                loadBackups(currentPage);
            });

            // Pagination
            $('.prev-page').on('click', function() {
                if (currentPage > 1) loadBackups(currentPage - 1);
            });
            $('.next-page').on('click', function() {
                if (currentPage < totalPages) loadBackups(currentPage + 1);
            });

            // Download button
            $(document).on('click', '.lox-download-btn', function() {
                var uuid = $(this).data('uuid');
                var $btn = $(this);
                $btn.prop('disabled', true).find('.dashicons').removeClass('dashicons-download').addClass('dashicons-update spin');

                $.post(loxBackup.ajax_url, {
                    action: 'lox_backup_download',
                    nonce: loxBackup.nonce,
                    uuid: uuid
                }, function(response) {
                    $btn.prop('disabled', false).find('.dashicons').removeClass('dashicons-update spin').addClass('dashicons-download');
                    if (response.success && response.data.download_url) {
                        window.open(response.data.download_url, '_blank');
                    } else {
                        showMessage('warning', response.data.message || '<?php _e('Download will be ready soon. Check back in a few minutes.', 'lox-backup'); ?>');
                    }
                });
            });

            // Restore button - open modal
            $(document).on('click', '.lox-restore-btn', function() {
                var uuid = $(this).data('uuid');
                var $row = $(this).closest('tr');
                var backupName = $row.find('td:first strong').text();
                var backupDate = $row.find('td:eq(4)').text();

                // Set modal data
                $('#restore-backup-uuid').val(uuid);
                $('#restore-backup-info .backup-name').text(backupName);
                $('#restore-backup-info .backup-date').text(backupDate);
                $('#restore-confirmation-input').val('').removeClass('valid');
                $('#restore-confirmation-error').hide();
                $('#lox-confirm-restore').prop('disabled', true);

                // Show modal
                $('#lox-restore-modal').fadeIn(200);
                $('#restore-confirmation-input').focus();
            });

            // Validate confirmation input
            $('#restore-confirmation-input').on('input', function() {
                var value = $(this).val().toUpperCase().trim();
                var confirmText = loxBackup.strings.restore_confirmation_text || 'RESTORE';
                var isValid = value === confirmText;

                $(this).toggleClass('valid', isValid);
                $('#lox-confirm-restore').prop('disabled', !isValid);
                $('#restore-confirmation-error').toggle(value.length > 0 && !isValid);
            });

            // Close restore modal
            $('#lox-restore-modal .lox-modal-close, #lox-restore-modal .lox-modal-cancel, #lox-restore-modal .lox-modal-backdrop').on('click', function() {
                $('#lox-restore-modal').fadeOut(200);
            });

            // Confirm restore
            $('#lox-confirm-restore').on('click', function() {
                var uuid = $('#restore-backup-uuid').val();
                var $btn = $(this);
                var originalHtml = $btn.html();

                $btn.prop('disabled', true).html('<span class="spinner is-active" style="float:none;margin:0 4px 0 0;"></span> <?php _e('Restoring...', 'lox-backup'); ?>');

                $.post(loxBackup.ajax_url, {
                    action: 'lox_backup_restore',
                    nonce: loxBackup.nonce,
                    uuid: uuid
                }, function(response) {
                    $('#lox-restore-modal').fadeOut(200);
                    if (response.success) {
                        showMessage('success', response.data.message);
                    } else {
                        showMessage('error', response.data.message);
                    }
                }).fail(function() {
                    showMessage('error', '<?php _e('Restore request failed', 'lox-backup'); ?>');
                }).always(function() {
                    $btn.prop('disabled', false).html(originalHtml);
                    $('#restore-confirmation-input').val('').removeClass('valid');
                });
            });

            // ESC key to close modal
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && $('#lox-restore-modal').is(':visible')) {
                    $('#lox-restore-modal').fadeOut(200);
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Render schedules page
     */
    public function render_schedules_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $schedules = get_option('lox_backup_schedules', $this->get_default_schedules());
        $components = array(
            'database' => array('label' => __('Database', 'lox-backup'), 'description' => __('WordPress database (posts, users, settings)', 'lox-backup')),
            'uploads' => array('label' => __('Uploads', 'lox-backup'), 'description' => __('Media library files', 'lox-backup')),
            'plugins' => array('label' => __('Plugins', 'lox-backup'), 'description' => __('All installed plugins', 'lox-backup')),
            'themes' => array('label' => __('Themes', 'lox-backup'), 'description' => __('All installed themes', 'lox-backup')),
            'full' => array('label' => __('Full Backup', 'lox-backup'), 'description' => __('Complete site backup (all components)', 'lox-backup')),
        );

        $frequencies = array(
            'disabled' => __('Disabled', 'lox-backup'),
            'hourly' => __('Hourly', 'lox-backup'),
            'daily' => __('Daily', 'lox-backup'),
            'weekly' => __('Weekly', 'lox-backup'),
            'monthly' => __('Monthly', 'lox-backup'),
        );

        $days = array(
            1 => __('Monday', 'lox-backup'),
            2 => __('Tuesday', 'lox-backup'),
            3 => __('Wednesday', 'lox-backup'),
            4 => __('Thursday', 'lox-backup'),
            5 => __('Friday', 'lox-backup'),
            6 => __('Saturday', 'lox-backup'),
            7 => __('Sunday', 'lox-backup'),
        );
        ?>
        <div class="wrap lox-backup-admin">
            <h1><?php _e('Backup Schedules', 'lox-backup'); ?></h1>

            <div class="lox-backup-header">
                <div class="lox-backup-logo">
                    <span class="dashicons dashicons-shield"></span>
                    <span><?php _e('LOX Cold Backup', 'lox-backup'); ?></span>
                </div>
            </div>

            <p class="description">
                <?php _e('Configure automatic backup schedules for each component. You can set different frequencies for different types of data.', 'lox-backup'); ?>
            </p>

            <form method="post" action="options.php">
                <?php settings_fields('lox_backup_schedules_group'); ?>

                <table class="wp-list-table widefat fixed striped lox-schedules-table">
                    <thead>
                        <tr>
                            <th style="width:40px;"><?php _e('Enable', 'lox-backup'); ?></th>
                            <th><?php _e('Component', 'lox-backup'); ?></th>
                            <th><?php _e('Frequency', 'lox-backup'); ?></th>
                            <th><?php _e('Time', 'lox-backup'); ?></th>
                            <th><?php _e('Day (Weekly)', 'lox-backup'); ?></th>
                            <th><?php _e('Next Run', 'lox-backup'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($components as $key => $component):
                            $schedule_key = "schedule_{$key}";
                            $schedule = $schedules[$schedule_key] ?? array('enabled' => false, 'frequency' => 'disabled', 'time' => '03:00', 'day' => 1);
                            $next_run = wp_next_scheduled("lox_backup_scheduled_{$key}");
                        ?>
                        <tr>
                            <td>
                                <input type="checkbox"
                                       name="lox_backup_schedules[<?php echo $schedule_key; ?>][enabled]"
                                       value="1"
                                       <?php checked($schedule['enabled']); ?>>
                            </td>
                            <td>
                                <strong><?php echo esc_html($component['label']); ?></strong>
                                <p class="description"><?php echo esc_html($component['description']); ?></p>
                            </td>
                            <td>
                                <select name="lox_backup_schedules[<?php echo $schedule_key; ?>][frequency]" class="lox-frequency-select">
                                    <?php foreach ($frequencies as $freq_key => $freq_label): ?>
                                        <option value="<?php echo esc_attr($freq_key); ?>" <?php selected($schedule['frequency'], $freq_key); ?>>
                                            <?php echo esc_html($freq_label); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                            <td>
                                <input type="time"
                                       name="lox_backup_schedules[<?php echo $schedule_key; ?>][time]"
                                       value="<?php echo esc_attr($schedule['time']); ?>">
                            </td>
                            <td>
                                <select name="lox_backup_schedules[<?php echo $schedule_key; ?>][day]" class="lox-day-select">
                                    <?php foreach ($days as $day_key => $day_label): ?>
                                        <option value="<?php echo esc_attr($day_key); ?>" <?php selected($schedule['day'], $day_key); ?>>
                                            <?php echo esc_html($day_label); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                            <td>
                                <?php if ($next_run): ?>
                                    <span class="lox-next-run">
                                        <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $next_run)); ?>
                                    </span>
                                <?php else: ?>
                                    <span class="lox-not-scheduled"><?php _e('Not scheduled', 'lox-backup'); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <?php submit_button(__('Save Schedules', 'lox-backup')); ?>
            </form>
        </div>

        <style>
            .lox-schedules-table th, .lox-schedules-table td { vertical-align: middle; }
            .lox-schedules-table .description { margin: 0; font-size: 12px; color: #666; }
            .lox-not-scheduled { color: #999; font-style: italic; }
            .lox-next-run { color: #0073aa; }
        </style>
        <?php
    }

    /**
     * Get default schedules
     */
    private function get_default_schedules() {
        return array(
            'schedule_database' => array('enabled' => true, 'frequency' => 'daily', 'time' => '03:00', 'day' => 1),
            'schedule_uploads' => array('enabled' => true, 'frequency' => 'weekly', 'time' => '04:00', 'day' => 7),
            'schedule_plugins' => array('enabled' => false, 'frequency' => 'weekly', 'time' => '04:30', 'day' => 7),
            'schedule_themes' => array('enabled' => false, 'frequency' => 'weekly', 'time' => '05:00', 'day' => 7),
            'schedule_full' => array('enabled' => false, 'frequency' => 'weekly', 'time' => '02:00', 'day' => 7),
        );
    }

    /**
     * Render status card
     */
    private function render_status_card($settings) {
        $last_backup = $settings['last_backup'] ?? null;
        $status = $settings['last_backup_status'] ?? null;
        $last_size = $settings['last_backup_size'] ?? null;
        $api = new LOX_API();
        $quota = $api->get_quota();

        // Determine status icon and color
        $status_icon_class = 'primary';
        if ($status === 'completed') {
            $status_icon_class = 'success';
        } elseif ($status === 'failed') {
            $status_icon_class = 'danger';
        }
        ?>
        <div class="lox-backup-status-card">
            <div class="status-item">
                <div class="status-item-header">
                    <div class="status-item-icon <?php echo esc_attr($status_icon_class); ?>">
                        <span class="dashicons dashicons-backup"></span>
                    </div>
                    <h3><?php _e('Last Backup', 'lox-backup'); ?></h3>
                </div>
                <?php if ($last_backup): ?>
                    <p class="status-value">
                        <?php echo esc_html(human_time_diff(strtotime($last_backup), current_time('timestamp'))); ?>
                        <?php _e('ago', 'lox-backup'); ?>
                    </p>
                    <p class="status-detail <?php echo $status === 'completed' ? 'success' : 'error'; ?>">
                        <span class="dashicons dashicons-<?php echo $status === 'completed' ? 'yes' : 'warning'; ?>" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle;"></span>
                        <?php echo esc_html(ucfirst($status)); ?>
                        <?php if ($last_size): ?>
                            &bull; <?php echo esc_html(size_format($last_size)); ?>
                        <?php endif; ?>
                    </p>
                <?php else: ?>
                    <p class="status-value"><?php _e('Never', 'lox-backup'); ?></p>
                    <p class="status-detail"><?php _e('No backups created yet', 'lox-backup'); ?></p>
                <?php endif; ?>
            </div>

            <div class="status-item">
                <div class="status-item-header">
                    <div class="status-item-icon <?php echo (!is_wp_error($quota) && isset($quota['storage_percentage']) && $quota['storage_percentage'] > 80) ? 'warning' : 'primary'; ?>">
                        <span class="dashicons dashicons-cloud"></span>
                    </div>
                    <h3><?php _e('Storage Used', 'lox-backup'); ?></h3>
                </div>
                <?php if (!is_wp_error($quota) && isset($quota['formatted'])): ?>
                    <p class="status-value"><?php echo esc_html($quota['formatted']['storage_used'] ?? '--'); ?></p>
                    <p class="status-detail">
                        <?php if (!empty($quota['unlimited_storage'])): ?>
                            <span class="dashicons dashicons-yes" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle; color: #10b981;"></span>
                            <?php _e('Unlimited storage', 'lox-backup'); ?>
                        <?php else: ?>
                            <?php echo esc_html($quota['storage_percentage'] ?? 0); ?>% <?php _e('of', 'lox-backup'); ?>
                            <?php echo esc_html($quota['formatted']['storage_quota'] ?? '--'); ?>
                        <?php endif; ?>
                    </p>
                    <?php if (empty($quota['unlimited_storage']) && !empty($quota['storage_percentage']) && $quota['storage_percentage'] > 0): ?>
                        <div class="lox-progress">
                            <div class="lox-progress-bar" style="width: <?php echo min(100, $quota['storage_percentage']); ?>%;"></div>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <p class="status-value">--</p>
                    <p class="status-detail error">
                        <span class="dashicons dashicons-warning" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle;"></span>
                        <?php _e('Not connected', 'lox-backup'); ?>
                    </p>
                <?php endif; ?>
            </div>

            <div class="status-item">
                <div class="status-item-header">
                    <div class="status-item-icon primary">
                        <span class="dashicons dashicons-admin-generic"></span>
                    </div>
                    <h3><?php _e('Quick Actions', 'lox-backup'); ?></h3>
                </div>
                <div class="button-group">
                    <button type="button" class="button button-primary" id="lox-run-backup">
                        <span class="dashicons dashicons-backup"></span>
                        <?php _e('Backup Now', 'lox-backup'); ?>
                    </button>
                    <button type="button" class="button" id="lox-test-connection">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <?php _e('Test', 'lox-backup'); ?>
                    </button>
                </div>
            </div>
        </div>
        <div id="lox-backup-message" class="notice" style="display:none;"></div>
        <?php
    }

    /**
     * Render section descriptions
     */
    public function render_api_section() {
        echo '<p>' . __('Configure your LOX API credentials. Get your API key from the LOX dashboard.', 'lox-backup') . '</p>';
    }

    public function render_backup_section() {
        echo '<p>' . __('Configure what to backup and when.', 'lox-backup') . '</p>';
    }

    /**
     * Render field callbacks
     */
    public function render_api_key_field() {
        $settings = get_option('lox_backup_settings', array());
        $value = $settings['api_key'] ?? '';
        ?>
        <input type="password" name="lox_backup_settings[api_key]" id="api_key"
               value="<?php echo esc_attr($value); ?>" class="regular-text">
        <p class="description">
            <?php _e('Your LOX API key. Keep this secret!', 'lox-backup'); ?>
        </p>
        <?php
    }

    public function render_base_url_field() {
        $settings = get_option('lox_backup_settings', array());
        $value = $settings['base_url'] ?? 'https://backlox.com/api';
        ?>
        <input type="url" name="lox_backup_settings[base_url]" id="base_url"
               value="<?php echo esc_attr($value); ?>" class="regular-text">
        <p class="description">
            <?php _e('For self-hosted LOX instances. Leave default for cloud service.', 'lox-backup'); ?>
        </p>
        <?php
    }

    public function render_components_field() {
        $settings = get_option('lox_backup_settings', array());
        $components = array(
            'backup_database' => __('Database', 'lox-backup'),
            'backup_uploads' => __('Uploads', 'lox-backup'),
            'backup_plugins' => __('Plugins', 'lox-backup'),
            'backup_themes' => __('Themes', 'lox-backup'),
        );

        foreach ($components as $key => $label) {
            $checked = !empty($settings[$key]);
            ?>
            <label>
                <input type="checkbox" name="lox_backup_settings[<?php echo $key; ?>]"
                       value="1" <?php checked($checked); ?>>
                <?php echo esc_html($label); ?>
            </label><br>
            <?php
        }
    }

    public function render_retention_field() {
        $settings = get_option('lox_backup_settings', array());
        $value = $settings['retention_days'] ?? 30;
        ?>
        <input type="number" name="lox_backup_settings[retention_days]" id="retention_days"
               value="<?php echo esc_attr($value); ?>" min="1" max="3650" class="small-text">
        <span><?php _e('days', 'lox-backup'); ?></span>
        <?php
    }

    public function render_immutable_days_field() {
        $settings = get_option('lox_backup_settings', array());
        $value = $settings['immutable_days'] ?? '';
        ?>
        <input type="number" name="lox_backup_settings[immutable_days]" id="immutable_days"
               value="<?php echo esc_attr($value); ?>" min="0" max="3650" class="small-text"
               placeholder="<?php esc_attr_e('Same as retention', 'lox-backup'); ?>">
        <span><?php _e('days', 'lox-backup'); ?></span>
        <p class="description">
            <?php _e('Days during which backups cannot be deleted (anti-ransomware protection). Leave empty to use retention period, or 0 to allow immediate deletion.', 'lox-backup'); ?>
        </p>
        <?php
    }

    public function render_tags_field() {
        $settings = get_option('lox_backup_settings', array());
        $value = $settings['tags'] ?? 'wordpress';
        ?>
        <input type="text" name="lox_backup_settings[tags]" id="tags"
               value="<?php echo esc_attr($value); ?>" class="regular-text">
        <p class="description"><?php _e('Comma-separated tags for organizing backups.', 'lox-backup'); ?></p>
        <?php
    }

    /**
     * AJAX: Test connection
     */
    public function ajax_test_connection() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $api = new LOX_API();
        $result = $api->test_connection();

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success(array(
            'message' => __('Connection successful!', 'lox-backup'),
            'tenant' => $result['name'],
        ));
    }

    /**
     * AJAX: Run backup
     */
    public function ajax_run_backup() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        // Increase time limit
        set_time_limit(3600);

        $backup = new LOX_Backup();
        $result = $backup->run_full_backup();

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success(array(
            'message' => __('Backup completed successfully!', 'lox-backup'),
            'uuid' => $result['uuid'],
            'size' => size_format($result['size_bytes']),
        ));
    }

    /**
     * AJAX: List backups
     */
    public function ajax_list_backups() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $page = intval($_POST['page'] ?? 1);
        $profile_id = isset($_POST['profile_id']) ? sanitize_text_field($_POST['profile_id']) : null;

        $api = new LOX_API();
        $params = array(
            'page' => $page,
            'page_size' => 20,
            'source' => 'wordpress',
        );

        // Add profile filter if specified
        if ($profile_id) {
            $params['profile_id'] = $profile_id;
        }

        $result = $api->list_backups($params);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Get remote backup profiles from LOX
     */
    public function ajax_get_remote_profiles() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $api = new LOX_API();
        // Get WordPress profiles - use source_identifier to match this site
        $result = $api->get_profiles(array(
            'source' => 'wordpress',
        ));

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        // Remove account-level storage info (not relevant for individual WordPress sites)
        if (isset($result['storage'])) {
            unset($result['storage']);
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Request backup restore
     */
    public function ajax_restore_backup() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $uuid = sanitize_text_field($_POST['uuid'] ?? '');

        if (empty($uuid)) {
            wp_send_json_error(array('message' => __('Invalid backup UUID', 'lox-backup')));
        }

        $api = new LOX_API();
        $result = $api->request_restore($uuid);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success(array(
            'message' => sprintf(
                __('Restore requested. %s', 'lox-backup'),
                $result['estimated_time'] ?? ''
            ),
            'task_id' => $result['task_id'] ?? null,
        ));
    }

    /**
     * AJAX: Download backup
     */
    public function ajax_download_backup() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $uuid = sanitize_text_field($_POST['uuid'] ?? '');

        if (empty($uuid)) {
            wp_send_json_error(array('message' => __('Invalid backup UUID', 'lox-backup')));
        }

        $api = new LOX_API();
        $result = $api->get_download_url($uuid);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        if (!empty($result['download_url'])) {
            $download_url = $result['download_url'];

            // If the URL is relative, make it absolute using the API base URL
            if (strpos($download_url, 'http') !== 0) {
                $settings = get_option('lox_backup_settings', array());
                $base_url = isset($settings['base_url']) ? rtrim($settings['base_url'], '/') : 'https://backlox.com/api';
                $download_url = $base_url . $download_url;
            }

            wp_send_json_success(array('download_url' => $download_url));
        } else {
            wp_send_json_error(array(
                'message' => __('Download is being prepared. Please try again in a few minutes.', 'lox-backup')
            ));
        }
    }

    /**
     * Admin notices
     */
    public function admin_notices() {
        $screen = get_current_screen();
        if (strpos($screen->id, 'lox-backup') === false) {
            return;
        }

        $settings = get_option('lox_backup_settings', array());

        if (empty($settings['api_key'])) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php _e('LOX Backup:', 'lox-backup'); ?></strong>
                    <?php _e('Please configure your API key to enable backups.', 'lox-backup'); ?>
                </p>
            </div>
            <?php
        }
    }

    /**
     * Render QA Widget (FORGE)
     */
    public function render_qa_widget() {
        $screen = get_current_screen();
        if (strpos($screen->id, 'lox-backup') === false) {
            return;
        }
        ?>
        <!-- FORGE QA Widget -->
        <script src="https://forge.argentica.ai/qa-widget.js" data-project="lox-wordpress" async></script>
        <?php
    }

    /**
     * Render profiles page
     */
    public function render_profiles_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $profiles = get_option('lox_backup_profiles', array());
        $elements = LOX_Backup::get_backup_elements();

        $frequencies = array(
            'disabled' => __('Disabled', 'lox-backup'),
            'hourly' => __('Hourly', 'lox-backup'),
            'daily' => __('Daily', 'lox-backup'),
            'weekly' => __('Weekly', 'lox-backup'),
            'monthly' => __('Monthly', 'lox-backup'),
        );
        ?>
        <div class="wrap lox-backup-admin">
            <h1><?php _e('Custom Backup Profiles', 'lox-backup'); ?></h1>

            <div class="lox-backup-header">
                <div class="lox-backup-logo">
                    <span class="dashicons dashicons-shield"></span>
                    <span><?php _e('LOX Cold Backup', 'lox-backup'); ?></span>
                </div>
            </div>

            <div class="lox-info-box">
                <span class="dashicons dashicons-info"></span>
                <p><?php _e('Create custom backup profiles that combine different elements. You can run them manually or schedule automatic backups.', 'lox-backup'); ?></p>
            </div>

            <div id="lox-backup-message" class="notice" style="display:none;"></div>

            <!-- Quick Custom Backup -->
            <div class="lox-card">
                <div class="lox-card-header">
                    <h2><span class="dashicons dashicons-backup" style="margin-right: 8px;"></span><?php _e('Quick Custom Backup', 'lox-backup'); ?></h2>
                </div>
                <div class="lox-card-body">
                    <p class="description" style="margin-bottom: 16px;">
                        <?php _e('Select the elements you want to backup and run a one-time custom backup.', 'lox-backup'); ?>
                    </p>

                    <div class="lox-elements-grid" id="quick-backup-elements">
                        <?php foreach ($elements as $key => $element): ?>
                        <label class="lox-element-card">
                            <input type="checkbox" name="quick_elements[]" value="<?php echo esc_attr($key); ?>" <?php echo $key === 'database' ? 'checked' : ''; ?>>
                            <span class="lox-element-icon">
                                <span class="dashicons <?php echo esc_attr($element['icon']); ?>"></span>
                            </span>
                            <span class="lox-element-info">
                                <strong><?php echo esc_html($element['label']); ?></strong>
                                <small><?php echo esc_html($element['description']); ?></small>
                            </span>
                            <span class="lox-element-check">
                                <span class="dashicons dashicons-yes-alt"></span>
                            </span>
                        </label>
                        <?php endforeach; ?>
                    </div>

                    <div class="lox-card-actions">
                        <button type="button" class="button button-primary" id="lox-run-quick-backup">
                            <span class="dashicons dashicons-backup"></span>
                            <?php _e('Run Custom Backup Now', 'lox-backup'); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Saved Profiles -->
            <div class="lox-card">
                <div class="lox-card-header">
                    <h2><span class="dashicons dashicons-portfolio" style="margin-right: 8px;"></span><?php _e('Saved Profiles', 'lox-backup'); ?></h2>
                    <button type="button" class="button button-primary" id="lox-add-profile">
                        <span class="dashicons dashicons-plus-alt"></span>
                        <?php _e('Add Profile', 'lox-backup'); ?>
                    </button>
                </div>
                <div class="lox-card-body">
                    <?php if (empty($profiles)): ?>
                    <div class="lox-empty-state" style="padding: 40px;">
                        <span class="dashicons dashicons-portfolio"></span>
                        <h3><?php _e('No profiles yet', 'lox-backup'); ?></h3>
                        <p><?php _e('Create your first backup profile to save custom element combinations.', 'lox-backup'); ?></p>
                    </div>
                    <?php else: ?>
                    <table class="wp-list-table widefat fixed striped" id="lox-profiles-table">
                        <thead>
                            <tr>
                                <th><?php _e('Profile Name', 'lox-backup'); ?></th>
                                <th><?php _e('Elements', 'lox-backup'); ?></th>
                                <th><?php _e('Schedule', 'lox-backup'); ?></th>
                                <th><?php _e('Last Run', 'lox-backup'); ?></th>
                                <th><?php _e('Actions', 'lox-backup'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($profiles as $id => $profile): ?>
                            <tr data-profile-id="<?php echo esc_attr($id); ?>">
                                <td>
                                    <strong><?php echo esc_html($profile['name']); ?></strong>
                                </td>
                                <td>
                                    <?php
                                    foreach ($profile['elements'] as $elem) {
                                        if (isset($elements[$elem])) {
                                            echo '<span class="lox-element-badge">';
                                            echo '<span class="dashicons ' . esc_attr($elements[$elem]['icon']) . '"></span>';
                                            echo esc_html($elements[$elem]['label']);
                                            echo '</span> ';
                                        }
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    $schedule = $profile['schedule'] ?? 'disabled';
                                    $schedule_label = $frequencies[$schedule] ?? $schedule;
                                    $next_run = wp_next_scheduled('lox_backup_profile_cron', array($id));
                                    ?>
                                    <span class="lox-schedule-badge <?php echo $schedule === 'disabled' ? 'disabled' : 'active'; ?>">
                                        <?php echo esc_html($schedule_label); ?>
                                    </span>
                                    <?php if ($next_run): ?>
                                    <br><small class="lox-next-run"><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $next_run)); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php
                                    $last_run = $profile['last_run'] ?? null;
                                    if ($last_run) {
                                        echo esc_html(human_time_diff(strtotime($last_run), current_time('timestamp'))) . ' ' . __('ago', 'lox-backup');
                                        if (!empty($profile['last_status'])) {
                                            $status_class = $profile['last_status'] === 'completed' ? 'success' : 'error';
                                            echo '<br><span class="lox-status ' . $status_class . '">' . esc_html(ucfirst($profile['last_status'])) . '</span>';
                                        }
                                    } else {
                                        echo '<span class="lox-not-scheduled">' . __('Never', 'lox-backup') . '</span>';
                                    }
                                    ?>
                                </td>
                                <td class="lox-actions">
                                    <button class="button button-small lox-run-profile-btn" data-profile-id="<?php echo esc_attr($id); ?>">
                                        <span class="dashicons dashicons-controls-play"></span>
                                        <?php _e('Run', 'lox-backup'); ?>
                                    </button>
                                    <button class="button button-small lox-edit-profile-btn" data-profile-id="<?php echo esc_attr($id); ?>">
                                        <span class="dashicons dashicons-edit"></span>
                                    </button>
                                    <button class="button button-small lox-delete-profile-btn" data-profile-id="<?php echo esc_attr($id); ?>">
                                        <span class="dashicons dashicons-trash"></span>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Profile Modal -->
        <div id="lox-profile-modal" class="lox-modal" style="display:none;">
            <div class="lox-modal-backdrop"></div>
            <div class="lox-modal-content">
                <div class="lox-modal-header">
                    <div class="lox-modal-header-icon">
                        <span class="dashicons dashicons-portfolio"></span>
                    </div>
                    <div class="lox-modal-header-text">
                        <h3 id="lox-modal-title"><?php _e('Create Profile', 'lox-backup'); ?></h3>
                        <p class="lox-modal-subtitle"><?php _e('Configure what to backup and when', 'lox-backup'); ?></p>
                    </div>
                    <button type="button" class="lox-modal-close" aria-label="<?php esc_attr_e('Close', 'lox-backup'); ?>">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                <div class="lox-modal-body">
                    <input type="hidden" id="profile-id" value="">

                    <!-- Section: Basic Info -->
                    <div class="lox-modal-section">
                        <div class="lox-section-header">
                            <span class="dashicons dashicons-edit"></span>
                            <span><?php _e('Basic Information', 'lox-backup'); ?></span>
                        </div>
                        <div class="lox-form-group">
                            <label for="profile-name"><?php _e('Profile Name', 'lox-backup'); ?></label>
                            <input type="text" id="profile-name" placeholder="<?php esc_attr_e('e.g., Daily Database Backup', 'lox-backup'); ?>">
                        </div>
                    </div>

                    <!-- Section: Elements -->
                    <div class="lox-modal-section">
                        <div class="lox-section-header">
                            <span class="dashicons dashicons-category"></span>
                            <span><?php _e('What to Backup', 'lox-backup'); ?></span>
                        </div>
                        <div class="lox-elements-grid" id="profile-elements">
                            <?php foreach ($elements as $key => $element): ?>
                            <label class="lox-element-card">
                                <input type="checkbox" name="profile_elements[]" value="<?php echo esc_attr($key); ?>">
                                <span class="lox-element-icon">
                                    <span class="dashicons <?php echo esc_attr($element['icon']); ?>"></span>
                                </span>
                                <span class="lox-element-info">
                                    <strong><?php echo esc_html($element['label']); ?></strong>
                                    <small><?php echo esc_html($element['description']); ?></small>
                                </span>
                                <span class="lox-element-check">
                                    <span class="dashicons dashicons-yes-alt"></span>
                                </span>
                            </label>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Section: Schedule -->
                    <div class="lox-modal-section">
                        <div class="lox-section-header">
                            <span class="dashicons dashicons-clock"></span>
                            <span><?php _e('Schedule', 'lox-backup'); ?></span>
                        </div>
                        <div class="lox-form-group">
                            <label for="profile-schedule"><?php _e('Frequency', 'lox-backup'); ?></label>
                            <select id="profile-schedule">
                                <?php foreach ($frequencies as $key => $label): ?>
                                <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="lox-form-row" id="schedule-options-row">
                            <div class="lox-form-group" id="profile-day-group" style="display: none;">
                                <label for="profile-day"><?php _e('Day of Week', 'lox-backup'); ?></label>
                                <select id="profile-day">
                                    <option value="1"><?php _e('Monday', 'lox-backup'); ?></option>
                                    <option value="2"><?php _e('Tuesday', 'lox-backup'); ?></option>
                                    <option value="3"><?php _e('Wednesday', 'lox-backup'); ?></option>
                                    <option value="4"><?php _e('Thursday', 'lox-backup'); ?></option>
                                    <option value="5"><?php _e('Friday', 'lox-backup'); ?></option>
                                    <option value="6"><?php _e('Saturday', 'lox-backup'); ?></option>
                                    <option value="7"><?php _e('Sunday', 'lox-backup'); ?></option>
                                </select>
                            </div>
                            <div class="lox-form-group" id="profile-monthday-group" style="display: none;">
                                <label for="profile-monthday"><?php _e('Day of Month', 'lox-backup'); ?></label>
                                <select id="profile-monthday">
                                    <?php for ($i = 1; $i <= 31; $i++): ?>
                                    <option value="<?php echo $i; ?>"><?php echo $i; ?></option>
                                    <?php endfor; ?>
                                </select>
                            </div>
                            <div class="lox-form-group" id="profile-time-group" style="display: none;">
                                <label for="profile-time"><?php _e('Time', 'lox-backup'); ?></label>
                                <input type="time" id="profile-time" value="03:00">
                            </div>
                        </div>
                        <p class="lox-schedule-hint" id="schedule-description"></p>
                    </div>

                    <!-- Section: Retention -->
                    <div class="lox-modal-section lox-section-collapsible" id="retention-section">
                        <div class="lox-section-header lox-section-toggle">
                            <span class="dashicons dashicons-shield"></span>
                            <span><?php _e('Retention Policy', 'lox-backup'); ?></span>
                            <span class="lox-optional-badge"><?php _e('Optional', 'lox-backup'); ?></span>
                            <span class="lox-toggle-icon dashicons dashicons-arrow-down-alt2"></span>
                        </div>
                        <div class="lox-section-content">
                            <p class="lox-section-description">
                                <?php _e('Leave empty to use your account defaults based on schedule frequency.', 'lox-backup'); ?>
                            </p>
                            <div class="lox-form-row">
                                <div class="lox-form-group">
                                    <label for="profile-retention-days">
                                        <span class="dashicons dashicons-calendar"></span>
                                        <?php _e('Retention Days', 'lox-backup'); ?>
                                    </label>
                                    <input type="number" id="profile-retention-days" min="1" max="3650" placeholder="<?php esc_attr_e('Auto', 'lox-backup'); ?>">
                                    <span class="lox-field-hint"><?php _e('How long to keep backups', 'lox-backup'); ?></span>
                                </div>
                                <div class="lox-form-group">
                                    <label for="profile-immutable-days">
                                        <span class="dashicons dashicons-lock"></span>
                                        <?php _e('Immutable Days', 'lox-backup'); ?>
                                    </label>
                                    <input type="number" id="profile-immutable-days" min="0" max="3650" placeholder="<?php esc_attr_e('Auto', 'lox-backup'); ?>">
                                    <span class="lox-field-hint"><?php _e('Ransomware protection period', 'lox-backup'); ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="lox-modal-footer">
                    <button type="button" class="button lox-modal-cancel"><?php _e('Cancel', 'lox-backup'); ?></button>
                    <button type="button" class="button button-primary" id="lox-save-profile">
                        <span class="dashicons dashicons-saved"></span>
                        <?php _e('Save Profile', 'lox-backup'); ?>
                    </button>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Move modal to body to avoid stacking context issues
            $('#lox-profile-modal').appendTo('body');

            var profiles = <?php echo json_encode($profiles); ?>;
            var elements = <?php echo json_encode($elements); ?>;

            // Schedule descriptions
            var scheduleDescriptions = {
                'disabled': '<?php _e('Backups will only run manually.', 'lox-backup'); ?>',
                'hourly': '<?php _e('Backup runs every hour automatically.', 'lox-backup'); ?>',
                'daily': '<?php _e('Backup runs once a day at the selected time.', 'lox-backup'); ?>',
                'weekly': '<?php _e('Backup runs once a week on the selected day and time.', 'lox-backup'); ?>',
                'monthly': '<?php _e('Backup runs once a month on the selected day and time.', 'lox-backup'); ?>'
            };

            // Update schedule fields visibility
            function updateScheduleFields(schedule) {
                var showTime = ['daily', 'weekly', 'monthly'].indexOf(schedule) !== -1;
                var showDay = schedule === 'weekly';
                var showMonthDay = schedule === 'monthly';

                $('#profile-time-group').toggle(showTime);
                $('#profile-day-group').toggle(showDay);
                $('#profile-monthday-group').toggle(showMonthDay);

                // Update day label based on context
                if (showDay) {
                    $('#profile-day-group label').text('<?php _e('Day of Week', 'lox-backup'); ?>');
                }

                // Update description
                $('#schedule-description').text(scheduleDescriptions[schedule] || '');
            }

            // Listen for schedule changes
            $('#profile-schedule').on('change', function() {
                updateScheduleFields($(this).val());
            });

            // Quick custom backup
            $('#lox-run-quick-backup').on('click', function() {
                var $btn = $(this);
                var selectedElements = [];
                $('#quick-backup-elements input:checked').each(function() {
                    selectedElements.push($(this).val());
                });

                if (selectedElements.length === 0) {
                    showMessage('error', '<?php _e('Please select at least one element to backup', 'lox-backup'); ?>');
                    return;
                }

                var originalHtml = $btn.html();
                $btn.prop('disabled', true).html('<span class="lox-spinner"></span> <?php _e('Running backup...', 'lox-backup'); ?>');

                showMessage('info', '<span class="lox-spinner"></span> <?php echo esc_js(__('Custom backup in progress. Please do not close this page.', 'lox-backup')); ?>');

                $.post(loxBackup.ajax_url, {
                    action: 'lox_backup_run_custom',
                    nonce: loxBackup.nonce,
                    elements: selectedElements
                }, function(response) {
                    if (response.success) {
                        showMessage('success', '<?php _e('Custom backup completed!', 'lox-backup'); ?> UUID: ' + response.data.uuid);
                    } else {
                        showMessage('error', response.data.message);
                    }
                }).fail(function() {
                    showMessage('error', '<?php _e('Backup failed', 'lox-backup'); ?>');
                }).always(function() {
                    $btn.prop('disabled', false).html(originalHtml);
                });
            });

            // Open modal for new profile
            $('#lox-add-profile').on('click', function() {
                $('#lox-modal-title').text('<?php _e('Create Profile', 'lox-backup'); ?>');
                $('#profile-id').val('');
                $('#profile-name').val('');
                $('#profile-elements input').prop('checked', false);
                $('#profile-schedule').val('disabled');
                $('#profile-time').val('03:00');
                $('#profile-day').val('1');
                $('#profile-monthday').val('1');
                $('#profile-retention-days').val('');
                $('#profile-immutable-days').val('');
                updateScheduleFields('disabled');
                $('#lox-profile-modal').fadeIn(200);
            });

            // Edit profile
            $(document).on('click', '.lox-edit-profile-btn', function() {
                var profileId = $(this).data('profile-id');
                var profile = profiles[profileId];

                if (!profile) return;

                $('#lox-modal-title').text('<?php _e('Edit Profile', 'lox-backup'); ?>');
                $('#profile-id').val(profileId);
                $('#profile-name').val(profile.name);
                $('#profile-elements input').prop('checked', false);
                profile.elements.forEach(function(elem) {
                    $('#profile-elements input[value="' + elem + '"]').prop('checked', true);
                });
                $('#profile-schedule').val(profile.schedule || 'disabled');
                $('#profile-time').val(profile.time || '03:00');
                $('#profile-day').val(profile.day || '1');
                $('#profile-monthday').val(profile.monthday || '1');
                $('#profile-retention-days').val(profile.retention_days || '');
                $('#profile-immutable-days').val(profile.immutable_days || '');
                updateScheduleFields(profile.schedule || 'disabled');
                $('#lox-profile-modal').fadeIn(200);
            });

            // Close modal
            $('.lox-modal-close, .lox-modal-cancel').on('click', function() {
                $('#lox-profile-modal').fadeOut(200);
            });

            // Close modal on backdrop click (click on .lox-modal but not on .lox-modal-content)
            $('#lox-profile-modal').on('click', function(e) {
                if (e.target === this) {
                    $(this).fadeOut(200);
                }
            });

            // Toggle collapsible sections
            $('.lox-section-toggle').on('click', function() {
                var $section = $(this).closest('.lox-section-collapsible');
                $section.toggleClass('collapsed');
            });

            // Save profile
            $('#lox-save-profile').on('click', function() {
                var $btn = $(this);
                var profileId = $('#profile-id').val();
                var name = $('#profile-name').val().trim();
                var selectedElements = [];
                $('#profile-elements input:checked').each(function() {
                    selectedElements.push($(this).val());
                });
                var schedule = $('#profile-schedule').val();
                var time = $('#profile-time').val();
                var day = $('#profile-day').val();
                var monthday = $('#profile-monthday').val();
                var retentionDays = $('#profile-retention-days').val();
                var immutableDays = $('#profile-immutable-days').val();

                if (!name) {
                    alert('<?php _e('Please enter a profile name', 'lox-backup'); ?>');
                    return;
                }

                if (selectedElements.length === 0) {
                    alert('<?php _e('Please select at least one element', 'lox-backup'); ?>');
                    return;
                }

                $btn.prop('disabled', true);

                $.post(loxBackup.ajax_url, {
                    action: 'lox_backup_save_profile',
                    nonce: loxBackup.nonce,
                    profile_id: profileId,
                    name: name,
                    elements: selectedElements,
                    schedule: schedule,
                    time: time,
                    day: day,
                    monthday: monthday,
                    retention_days: retentionDays,
                    immutable_days: immutableDays
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert(response.data.message);
                        $btn.prop('disabled', false);
                    }
                });
            });

            // Run profile
            $(document).on('click', '.lox-run-profile-btn', function() {
                var $btn = $(this);
                var profileId = $btn.data('profile-id');
                var originalHtml = $btn.html();

                $btn.prop('disabled', true).html('<span class="lox-spinner"></span>');
                showMessage('info', '<span class="lox-spinner"></span> <?php _e('Running profile backup...', 'lox-backup'); ?>');

                $.post(loxBackup.ajax_url, {
                    action: 'lox_backup_run_profile',
                    nonce: loxBackup.nonce,
                    profile_id: profileId
                }, function(response) {
                    if (response.success) {
                        showMessage('success', '<?php _e('Profile backup completed!', 'lox-backup'); ?> UUID: ' + response.data.uuid);
                        setTimeout(function() { location.reload(); }, 2000);
                    } else {
                        showMessage('error', response.data.message);
                    }
                }).fail(function() {
                    showMessage('error', '<?php _e('Backup failed', 'lox-backup'); ?>');
                }).always(function() {
                    $btn.prop('disabled', false).html(originalHtml);
                });
            });

            // Delete profile
            $(document).on('click', '.lox-delete-profile-btn', function() {
                if (!confirm('<?php _e('Are you sure you want to delete this profile?', 'lox-backup'); ?>')) {
                    return;
                }

                var $btn = $(this);
                var profileId = $btn.data('profile-id');

                $.post(loxBackup.ajax_url, {
                    action: 'lox_backup_delete_profile',
                    nonce: loxBackup.nonce,
                    profile_id: profileId
                }, function(response) {
                    if (response.success) {
                        $btn.closest('tr').fadeOut(300, function() {
                            $(this).remove();
                        });
                    } else {
                        alert(response.data.message);
                    }
                });
            });

            function showMessage(type, message) {
                var $msg = $('#lox-backup-message');
                $msg.removeClass('notice-success notice-error notice-warning notice-info')
                    .addClass('notice-' + type)
                    .html('<p>' + message + '</p>')
                    .show();
            }
        });
        </script>

        <style>
        .lox-elements-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 12px;
            margin-bottom: 20px;
        }
        .lox-elements-grid-compact {
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
        }
        .lox-element-card {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 14px;
            background: #fff;
            border: 2px solid var(--lox-gray-200);
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        .lox-element-card:hover {
            border-color: var(--lox-primary);
            background: var(--lox-gray-50);
        }
        .lox-element-card input {
            display: none;
        }
        .lox-element-card input:checked + .lox-element-icon {
            background: linear-gradient(135deg, var(--lox-primary) 0%, var(--lox-primary-light) 100%);
            color: #fff;
        }
        .lox-element-card input:checked ~ .lox-element-check {
            opacity: 1;
            transform: scale(1);
        }
        .lox-element-card input:checked ~ .lox-element-info strong {
            color: var(--lox-primary);
        }
        .lox-element-icon {
            width: 44px;
            height: 44px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: var(--lox-gray-100);
            border-radius: 10px;
            color: var(--lox-gray-500);
            transition: all 0.2s ease;
            flex-shrink: 0;
        }
        .lox-element-icon .dashicons {
            font-size: 22px;
            width: 22px;
            height: 22px;
        }
        .lox-element-info {
            flex: 1;
        }
        .lox-element-info strong {
            display: block;
            font-size: 14px;
            color: var(--lox-gray-800);
            margin-bottom: 2px;
        }
        .lox-element-info small {
            font-size: 11px;
            color: var(--lox-gray-500);
            line-height: 1.3;
        }
        .lox-element-check {
            opacity: 0;
            transform: scale(0.5);
            transition: all 0.2s ease;
            color: var(--lox-success);
        }
        .lox-element-check .dashicons {
            font-size: 24px;
            width: 24px;
            height: 24px;
        }
        .lox-element-badge {
            display: inline-flex;
            align-items: center;
            gap: 4px;
            padding: 4px 10px;
            background: var(--lox-gray-100);
            border-radius: 20px;
            font-size: 11px;
            font-weight: 500;
            color: var(--lox-gray-700);
            margin: 2px;
        }
        .lox-element-badge .dashicons {
            font-size: 14px;
            width: 14px;
            height: 14px;
        }
        .lox-schedule-badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 600;
        }
        .lox-schedule-badge.disabled {
            background: var(--lox-gray-100);
            color: var(--lox-gray-500);
        }
        .lox-schedule-badge.active {
            background: #d1fae5;
            color: #065f46;
        }
        .lox-card-actions {
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid var(--lox-gray-200);
        }

        /* Modal - Modern Design */
        @keyframes loxSlideUp {
            from { opacity: 0; transform: translateY(20px) scale(0.98); }
            to { opacity: 1; transform: translateY(0) scale(1); }
        }
        #lox-profile-modal {
            position: fixed !important;
            inset: 0 !important;
            width: 100vw !important;
            height: 100vh !important;
            max-width: none !important;
            max-height: none !important;
            min-width: 100vw !important;
            min-height: 100vh !important;
            z-index: 999999 !important;
            box-sizing: border-box !important;
            background: rgba(0, 0, 0, 0.5);
            margin: 0 !important;
            padding: 0 !important;
            transform: none !important;
            border: none !important;
        }
        #lox-profile-modal[style*="block"] {
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
        }
        #lox-profile-modal .lox-modal-backdrop {
            display: none;
        }
        #lox-profile-modal .lox-modal-content {
            position: relative;
            width: calc(100% - 40px);
            max-width: 720px;
            max-height: calc(100vh - 40px);
            background: #fff;
            border-radius: 16px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
            display: flex;
            flex-direction: column;
            animation: loxSlideUp 0.3s ease-out;
            z-index: 1;
            margin: 10px auto;
        }
        .lox-modal-header {
            display: flex;
            align-items: flex-start;
            gap: 16px;
            padding: 24px 24px 20px;
            border-bottom: 1px solid var(--lox-gray-100);
        }
        .lox-modal-header-icon {
            width: 48px;
            height: 48px;
            background: linear-gradient(135deg, var(--lox-primary), #6366f1);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }
        .lox-modal-header-icon .dashicons {
            color: #fff;
            font-size: 24px;
            width: 24px;
            height: 24px;
        }
        .lox-modal-header-text {
            flex: 1;
            min-width: 0;
        }
        .lox-modal-header h3 {
            margin: 0 0 4px;
            font-size: 20px;
            font-weight: 600;
            color: var(--lox-gray-900);
        }
        .lox-modal-subtitle {
            margin: 0;
            font-size: 14px;
            color: var(--lox-gray-500);
        }
        .lox-modal-close {
            background: none;
            border: none;
            width: 36px;
            height: 36px;
            border-radius: 8px;
            color: var(--lox-gray-400);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.15s ease;
            flex-shrink: 0;
        }
        .lox-modal-close:hover {
            background: var(--lox-gray-100);
            color: var(--lox-gray-600);
        }
        .lox-modal-close .dashicons {
            font-size: 20px;
            width: 20px;
            height: 20px;
        }
        .lox-modal-body {
            padding: 0;
            overflow-y: auto;
            flex: 1;
        }
        .lox-modal-section {
            padding: 20px 24px;
            border-bottom: 1px solid var(--lox-gray-100);
        }
        .lox-modal-section:last-child {
            border-bottom: none;
        }
        .lox-section-header {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 16px;
            font-size: 13px;
            font-weight: 600;
            color: var(--lox-gray-900);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .lox-section-header .dashicons {
            color: var(--lox-primary);
            font-size: 18px;
            width: 18px;
            height: 18px;
        }
        .lox-section-collapsible .lox-section-header {
            cursor: pointer;
            padding: 12px 16px;
            margin: -12px -16px 0;
            border-radius: 8px;
            transition: background 0.15s ease;
        }
        .lox-section-collapsible .lox-section-header:hover {
            background: var(--lox-gray-50);
        }
        .lox-toggle-icon {
            margin-left: auto;
            transition: transform 0.2s ease;
        }
        .lox-section-collapsible.collapsed .lox-toggle-icon {
            transform: rotate(-90deg);
        }
        .lox-section-collapsible.collapsed .lox-section-content {
            display: none;
        }
        .lox-section-content {
            margin-top: 16px;
        }
        .lox-section-description {
            font-size: 13px;
            color: var(--lox-gray-500);
            margin: 0 0 16px 0;
            line-height: 1.5;
        }
        .lox-modal-footer {
            display: flex;
            justify-content: flex-end;
            gap: 12px;
            padding: 16px 24px;
            border-top: 1px solid var(--lox-gray-100);
            background: #fff;
            border-radius: 0 0 16px 16px;
        }
        .lox-modal-footer .button {
            padding: 10px 20px;
            font-size: 14px;
            font-weight: 500;
            border-radius: 8px;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.15s ease;
        }
        .lox-modal-footer .button-primary {
            background: var(--lox-primary);
            border-color: var(--lox-primary);
            box-shadow: 0 1px 3px rgba(79, 70, 229, 0.3);
        }
        .lox-modal-footer .button-primary:hover {
            background: #4338ca;
            border-color: #4338ca;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(79, 70, 229, 0.4);
        }
        .lox-modal-footer .button-primary .dashicons {
            font-size: 16px;
            width: 16px;
            height: 16px;
        }
        .lox-form-group {
            margin-bottom: 16px;
        }
        .lox-form-group:last-child {
            margin-bottom: 0;
        }
        .lox-form-group label {
            display: flex;
            align-items: center;
            gap: 6px;
            font-weight: 500;
            font-size: 13px;
            margin-bottom: 8px;
            color: var(--lox-gray-700);
        }
        .lox-form-group label .dashicons {
            font-size: 16px;
            width: 16px;
            height: 16px;
            color: var(--lox-gray-400);
        }
        .lox-form-group input[type="text"],
        .lox-form-group input[type="number"],
        .lox-form-group input[type="time"],
        .lox-form-group select {
            width: 100%;
            padding: 10px 14px;
            font-size: 14px;
            border: 1px solid var(--lox-gray-300);
            border-radius: 8px;
            background: #fff;
            transition: all 0.15s ease;
        }
        .lox-form-group input:focus,
        .lox-form-group select:focus {
            outline: none;
            border-color: var(--lox-primary);
            box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
        }
        .lox-form-group input::placeholder {
            color: var(--lox-gray-400);
        }
        .lox-form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 16px;
        }
        .lox-schedule-hint {
            font-size: 13px;
            color: var(--lox-gray-500);
            margin: 8px 0 0;
            padding: 10px 12px;
            background: var(--lox-gray-50);
            border-radius: 6px;
            border-left: 3px solid var(--lox-primary);
        }
        .lox-schedule-hint:empty {
            display: none;
        }
        .lox-optional-badge {
            font-size: 10px;
            font-weight: 600;
            color: var(--lox-gray-500);
            background: var(--lox-gray-100);
            padding: 3px 8px;
            border-radius: 4px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .lox-field-hint {
            display: block;
            font-size: 12px;
            color: var(--lox-gray-500);
            margin-top: 6px;
        }
        /* Elements Grid */
        .lox-elements-grid {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        .lox-element-card {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px 14px;
            border: 1px solid var(--lox-gray-200);
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.15s ease;
            background: #fff;
        }
        .lox-element-card:hover {
            border-color: var(--lox-gray-300);
            background: var(--lox-gray-50);
        }
        .lox-element-card input {
            display: none;
        }
        .lox-element-card input:checked + .lox-element-icon {
            background: var(--lox-primary);
        }
        .lox-element-card input:checked + .lox-element-icon .dashicons {
            color: #fff;
        }
        .lox-element-card input:checked ~ .lox-element-check {
            opacity: 1;
            transform: scale(1);
        }
        .lox-element-card input:checked ~ .lox-element-info strong {
            color: var(--lox-primary);
        }
        .lox-element-icon {
            width: 40px;
            height: 40px;
            background: var(--lox-gray-100);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            transition: all 0.15s ease;
        }
        .lox-element-icon .dashicons {
            font-size: 20px;
            width: 20px;
            height: 20px;
            color: var(--lox-gray-500);
            transition: color 0.15s ease;
        }
        .lox-element-info {
            flex: 1;
            min-width: 0;
        }
        .lox-element-info strong {
            display: block;
            font-size: 14px;
            font-weight: 600;
            color: var(--lox-gray-900);
            transition: color 0.15s ease;
        }
        .lox-element-info small {
            display: block;
            font-size: 12px;
            color: var(--lox-gray-500);
            margin-top: 2px;
        }
        .lox-element-check {
            width: 24px;
            height: 24px;
            opacity: 0;
            transform: scale(0.8);
            transition: all 0.15s ease;
        }
        .lox-element-check .dashicons {
            font-size: 24px;
            width: 24px;
            height: 24px;
            color: var(--lox-primary);
        }
        </style>
        <?php
    }

    /**
     * AJAX: Run custom backup
     */
    public function ajax_run_custom_backup() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $elements = isset($_POST['elements']) ? array_map('sanitize_text_field', $_POST['elements']) : array();

        if (empty($elements)) {
            wp_send_json_error(array('message' => __('No elements selected', 'lox-backup')));
        }

        set_time_limit(3600);

        $backup = new LOX_Backup();
        $result = $backup->run_custom_backup($elements);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success(array(
            'message' => __('Custom backup completed!', 'lox-backup'),
            'uuid' => $result['uuid'],
            'size' => size_format($result['size_bytes']),
        ));
    }

    /**
     * AJAX: Save profile
     */
    public function ajax_save_profile() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $profile_id = sanitize_text_field($_POST['profile_id'] ?? '');
        $name = sanitize_text_field($_POST['name'] ?? '');
        $elements = isset($_POST['elements']) ? array_map('sanitize_text_field', $_POST['elements']) : array();
        $schedule = sanitize_text_field($_POST['schedule'] ?? 'disabled');
        $time = sanitize_text_field($_POST['time'] ?? '03:00');
        $day = absint($_POST['day'] ?? 1);
        $monthday = absint($_POST['monthday'] ?? 1);

        // Retention policy (empty string = null = use account default)
        $retention_days = isset($_POST['retention_days']) && $_POST['retention_days'] !== '' ? absint($_POST['retention_days']) : null;
        $immutable_days = isset($_POST['immutable_days']) && $_POST['immutable_days'] !== '' ? absint($_POST['immutable_days']) : null;

        // Validate day values
        $day = max(1, min(7, $day)); // 1-7 for days of week
        $monthday = max(1, min(31, $monthday)); // 1-31 for days of month

        if (empty($name)) {
            wp_send_json_error(array('message' => __('Profile name is required', 'lox-backup')));
        }

        if (empty($elements)) {
            wp_send_json_error(array('message' => __('At least one element is required', 'lox-backup')));
        }

        $profiles = get_option('lox_backup_profiles', array());

        // Generate ID if new
        if (empty($profile_id)) {
            $profile_id = 'profile_' . uniqid();
        }

        // Preserve existing stats
        $existing = $profiles[$profile_id] ?? array();
        $remote_uuid = $existing['remote_uuid'] ?? null;

        // Sync profile with server
        $settings = get_option('lox_backup_settings', array());
        if (!empty($settings['api_key'])) {
            $api = new LOX_Backup_API($settings['api_key'], $settings['base_url'] ?? 'https://backlox.com/api');

            $profile_data = array(
                'name' => $name,
                'component' => implode(',', $elements),
                'is_active' => $schedule !== 'disabled',
            );

            if ($remote_uuid) {
                // Update existing profile on server
                $result = $api->update_profile($remote_uuid, $profile_data);
            } else {
                // Create new profile on server
                $result = $api->create_profile($name, $profile_data);
                if ($result['success'] && isset($result['data']['uuid'])) {
                    $remote_uuid = $result['data']['uuid'];
                }
            }
        }

        $profiles[$profile_id] = array(
            'name' => $name,
            'elements' => $elements,
            'schedule' => $schedule,
            'time' => $time,
            'day' => $day,
            'monthday' => $monthday,
            'retention_days' => $retention_days,
            'immutable_days' => $immutable_days,
            'remote_uuid' => $remote_uuid,
            'last_run' => $existing['last_run'] ?? null,
            'last_status' => $existing['last_status'] ?? null,
            'created' => $existing['created'] ?? current_time('mysql'),
            'updated' => current_time('mysql'),
        );

        update_option('lox_backup_profiles', $profiles);

        // Update cron schedule
        $this->update_profile_cron($profile_id, $schedule, $time, $day, $monthday);

        wp_send_json_success(array(
            'profile_id' => $profile_id,
            'message' => __('Profile saved!', 'lox-backup'),
        ));
    }

    /**
     * AJAX: Delete profile
     */
    public function ajax_delete_profile() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $profile_id = sanitize_text_field($_POST['profile_id'] ?? '');

        if (empty($profile_id)) {
            wp_send_json_error(array('message' => __('Invalid profile', 'lox-backup')));
        }

        $profiles = get_option('lox_backup_profiles', array());

        if (!isset($profiles[$profile_id])) {
            wp_send_json_error(array('message' => __('Profile not found', 'lox-backup')));
        }

        unset($profiles[$profile_id]);
        update_option('lox_backup_profiles', $profiles);

        // Clear cron
        wp_clear_scheduled_hook('lox_backup_profile_cron', array($profile_id));

        wp_send_json_success(array('message' => __('Profile deleted', 'lox-backup')));
    }

    /**
     * AJAX: Run profile backup
     */
    public function ajax_run_profile() {
        check_ajax_referer('lox_backup_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'lox-backup')));
        }

        $profile_id = sanitize_text_field($_POST['profile_id'] ?? '');

        if (empty($profile_id)) {
            wp_send_json_error(array('message' => __('Invalid profile', 'lox-backup')));
        }

        set_time_limit(3600);

        $backup = new LOX_Backup();
        $result = $backup->run_profile_backup($profile_id);

        if (is_wp_error($result)) {
            $this->update_profile_status($profile_id, 'failed', $result->get_error_message());
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        $this->update_profile_status($profile_id, 'completed', null, $result);

        wp_send_json_success(array(
            'message' => __('Profile backup completed!', 'lox-backup'),
            'uuid' => $result['uuid'],
            'size' => size_format($result['size_bytes']),
        ));
    }

    /**
     * Update profile cron schedule
     */
    private function update_profile_cron($profile_id, $schedule, $time, $day = 1, $monthday = 1) {
        // Clear existing schedule
        wp_clear_scheduled_hook('lox_backup_profile_cron', array($profile_id));

        if ($schedule === 'disabled') {
            return;
        }

        // Calculate next run time
        $time_parts = explode(':', $time);
        $hour = intval($time_parts[0]);
        $minute = intval($time_parts[1] ?? 0);

        $next_run = strtotime("today {$hour}:{$minute}");
        if ($next_run < time()) {
            $next_run = strtotime("tomorrow {$hour}:{$minute}");
        }

        // Adjust for weekly schedules
        if ($schedule === 'weekly' && $day > 0) {
            $days = array('', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday');
            $day_name = $days[$day] ?? 'monday';
            $next_run = strtotime("next {$day_name} {$hour}:{$minute}");
            // If today is the scheduled day and we haven't passed the time yet, use today
            $today_day = date('N'); // 1 (Monday) to 7 (Sunday)
            if ($today_day == $day && strtotime("today {$hour}:{$minute}") > time()) {
                $next_run = strtotime("today {$hour}:{$minute}");
            }
        }

        // Adjust for monthly schedules
        if ($schedule === 'monthly' && $monthday > 0) {
            $current_day = date('j');
            $current_month = date('n');
            $current_year = date('Y');

            // Check if the day already passed this month
            if ($current_day > $monthday || ($current_day == $monthday && strtotime("today {$hour}:{$minute}") <= time())) {
                // Schedule for next month
                $current_month++;
                if ($current_month > 12) {
                    $current_month = 1;
                    $current_year++;
                }
            }

            // Handle months with fewer days
            $last_day_of_month = date('t', mktime(0, 0, 0, $current_month, 1, $current_year));
            $actual_day = min($monthday, $last_day_of_month);

            $next_run = mktime($hour, $minute, 0, $current_month, $actual_day, $current_year);
        }

        wp_schedule_event($next_run, $schedule, 'lox_backup_profile_cron', array($profile_id));
    }

    /**
     * Run profile cron job
     */
    public function run_profile_cron($profile_id) {
        $backup = new LOX_Backup();
        $result = $backup->run_profile_backup($profile_id);

        if (is_wp_error($result)) {
            $this->update_profile_status($profile_id, 'failed', $result->get_error_message());
        } else {
            $this->update_profile_status($profile_id, 'completed', null, $result);
        }
    }

    /**
     * Update profile status after backup
     */
    private function update_profile_status($profile_id, $status, $message = null, $backup = null) {
        $profiles = get_option('lox_backup_profiles', array());

        if (!isset($profiles[$profile_id])) {
            return;
        }

        $profiles[$profile_id]['last_run'] = current_time('mysql');
        $profiles[$profile_id]['last_status'] = $status;
        $profiles[$profile_id]['last_message'] = $message;

        if ($backup) {
            $profiles[$profile_id]['last_uuid'] = $backup['uuid'];
            $profiles[$profile_id]['last_size'] = $backup['size_bytes'];
        }

        update_option('lox_backup_profiles', $profiles);
    }
}
