<?php
/**
 * LOX REST API - PULL Backup Endpoint
 *
 * Exposes a secure endpoint for LOX to initiate backups remotely.
 * This enables the PULL model where LOX controls the backup schedule.
 *
 * @package LOX_Backup
 * @since 1.3.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * REST API handler for PULL backups
 */
class LOX_REST_API {

    /**
     * API namespace
     */
    const NAMESPACE = 'lox/v1';

    /**
     * Token option name
     */
    const TOKEN_OPTION = 'lox_pull_token';

    /**
     * Token expiry option name
     */
    const TOKEN_EXPIRY_OPTION = 'lox_pull_token_expiry';

    /**
     * Constructor - register hooks
     */
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    /**
     * Register REST API routes
     */
    public function register_routes() {
        // PULL backup endpoint - LOX calls this to trigger a backup
        register_rest_route(self::NAMESPACE, '/backup', array(
            'methods'             => 'POST',
            'callback'            => array($this, 'handle_backup_request'),
            'permission_callback' => array($this, 'verify_pull_token'),
            'args'                => array(
                'type' => array(
                    'type'              => 'string',
                    'default'           => 'full',
                    'enum'              => array('full', 'component', 'custom', 'profile'),
                    'description'       => 'Backup type',
                ),
                'component' => array(
                    'type'              => 'string',
                    'enum'              => array('content', 'transactional', 'files', 'config'),
                    'description'       => 'Component for component backups',
                ),
                'elements' => array(
                    'type'              => 'array',
                    'items'             => array('type' => 'string'),
                    'description'       => 'Elements for custom backups',
                ),
                'profile_id' => array(
                    'type'              => 'string',
                    'description'       => 'Profile ID for profile backups',
                ),
                'frequency' => array(
                    'type'              => 'string',
                    'default'           => 'daily',
                    'description'       => 'Backup frequency for retention',
                ),
                'retention_days' => array(
                    'type'              => 'integer',
                    'description'       => 'Days to retain backup',
                ),
                'stream' => array(
                    'type'              => 'boolean',
                    'default'           => false,
                    'description'       => 'Stream backup directly (vs provide URL)',
                ),
            ),
        ));

        // Status endpoint - check site availability and info
        register_rest_route(self::NAMESPACE, '/status', array(
            'methods'             => 'GET',
            'callback'            => array($this, 'handle_status_request'),
            'permission_callback' => array($this, 'verify_pull_token'),
        ));

        // Token refresh endpoint - regenerate the PULL token
        register_rest_route(self::NAMESPACE, '/token/refresh', array(
            'methods'             => 'POST',
            'callback'            => array($this, 'handle_token_refresh'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Download endpoint - temporary download URL for generated backups
        register_rest_route(self::NAMESPACE, '/download/(?P<token>[a-zA-Z0-9]+)', array(
            'methods'             => 'GET',
            'callback'            => array($this, 'handle_download'),
            'permission_callback' => '__return_true', // Token in URL provides auth
            'args'                => array(
                'token' => array(
                    'required'          => true,
                    'type'              => 'string',
                    'pattern'           => '[a-zA-Z0-9]+',
                ),
            ),
        ));
    }

    /**
     * Verify PULL token from request header
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error True if valid, WP_Error otherwise
     */
    public function verify_pull_token($request) {
        $auth_header = $request->get_header('X-LOX-Pull-Token');

        if (empty($auth_header)) {
            return new WP_Error(
                'missing_token',
                __('Missing X-LOX-Pull-Token header', 'lox-backup'),
                array('status' => 401)
            );
        }

        $stored_token = get_option(self::TOKEN_OPTION);
        $token_expiry = get_option(self::TOKEN_EXPIRY_OPTION);

        if (empty($stored_token)) {
            return new WP_Error(
                'not_configured',
                __('PULL token not configured. Enable PULL mode in settings.', 'lox-backup'),
                array('status' => 403)
            );
        }

        // Check expiry
        if ($token_expiry && time() > $token_expiry) {
            return new WP_Error(
                'token_expired',
                __('PULL token has expired. Please refresh.', 'lox-backup'),
                array('status' => 401)
            );
        }

        // Constant-time comparison
        if (!hash_equals($stored_token, $auth_header)) {
            return new WP_Error(
                'invalid_token',
                __('Invalid PULL token', 'lox-backup'),
                array('status' => 401)
            );
        }

        return true;
    }

    /**
     * Verify API key for token management
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error True if valid, WP_Error otherwise
     */
    public function verify_api_key($request) {
        $auth_header = $request->get_header('X-API-Key');

        if (empty($auth_header)) {
            return new WP_Error(
                'missing_api_key',
                __('Missing X-API-Key header', 'lox-backup'),
                array('status' => 401)
            );
        }

        $settings = get_option('lox_backup_settings', array());
        $stored_key = isset($settings['api_key']) ? $settings['api_key'] : '';

        if (empty($stored_key)) {
            return new WP_Error(
                'not_configured',
                __('LOX API key not configured', 'lox-backup'),
                array('status' => 403)
            );
        }

        if (!hash_equals($stored_key, $auth_header)) {
            return new WP_Error(
                'invalid_api_key',
                __('Invalid API key', 'lox-backup'),
                array('status' => 401)
            );
        }

        return true;
    }

    /**
     * Handle PULL backup request from LOX
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error Response
     */
    public function handle_backup_request($request) {
        $type = $request->get_param('type');
        $stream = $request->get_param('stream');
        $frequency = $request->get_param('frequency');
        $retention_days = $request->get_param('retention_days');

        $options = array(
            'frequency' => $frequency,
        );
        if ($retention_days) {
            $options['retention_days'] = $retention_days;
        }

        $backup = new LOX_Backup();

        // Disable LOX upload for PULL mode - we'll return the file directly
        add_filter('lox_backup_skip_upload', '__return_true');

        try {
            switch ($type) {
                case 'full':
                    $result = $this->run_backup_for_pull($backup, 'full', $options);
                    break;

                case 'component':
                    $component = $request->get_param('component');
                    if (empty($component)) {
                        return new WP_Error(
                            'missing_component',
                            __('Component parameter required for component backup', 'lox-backup'),
                            array('status' => 400)
                        );
                    }
                    $result = $this->run_backup_for_pull($backup, 'component', $options, $component);
                    break;

                case 'custom':
                    $elements = $request->get_param('elements');
                    if (empty($elements)) {
                        return new WP_Error(
                            'missing_elements',
                            __('Elements parameter required for custom backup', 'lox-backup'),
                            array('status' => 400)
                        );
                    }
                    $result = $this->run_backup_for_pull($backup, 'custom', $options, null, $elements);
                    break;

                case 'profile':
                    $profile_id = $request->get_param('profile_id');
                    if (empty($profile_id)) {
                        return new WP_Error(
                            'missing_profile',
                            __('Profile ID required for profile backup', 'lox-backup'),
                            array('status' => 400)
                        );
                    }
                    $result = $this->run_backup_for_pull($backup, 'profile', $options, null, null, $profile_id);
                    break;

                default:
                    return new WP_Error(
                        'invalid_type',
                        __('Invalid backup type', 'lox-backup'),
                        array('status' => 400)
                    );
            }

            if (is_wp_error($result)) {
                return $result;
            }

            // If streaming, return the file directly
            if ($stream) {
                return $this->stream_backup($result['archive_path']);
            }

            // Otherwise, create a temporary download token
            $download_token = $this->create_download_token($result['archive_path']);

            return rest_ensure_response(array(
                'success'       => true,
                'backup_name'   => $result['name'],
                'size_bytes'    => filesize($result['archive_path']),
                'checksum'      => hash_file('sha256', $result['archive_path']),
                'download_url'  => rest_url(self::NAMESPACE . '/download/' . $download_token),
                'expires_in'    => 3600, // 1 hour
            ));

        } catch (Exception $e) {
            return new WP_Error(
                'backup_failed',
                $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Run backup for PULL mode (without LOX upload)
     *
     * @param LOX_Backup $backup Backup instance
     * @param string $type Backup type
     * @param array $options Options
     * @param string|null $component Component name
     * @param array|null $elements Custom elements
     * @param string|null $profile_id Profile ID
     * @return array|WP_Error Archive info
     */
    private function run_backup_for_pull($backup, $type, $options, $component = null, $elements = null, $profile_id = null) {
        $timestamp = date('Ymd_His');
        $site_name = sanitize_title(get_bloginfo('name'));
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/lox-backups';

        // Ensure backup directory exists
        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
            file_put_contents($backup_dir . '/.htaccess', 'deny from all');
        }

        $temp_dir = $backup_dir . '/temp_pull_' . $timestamp;
        wp_mkdir_p($temp_dir);

        $files_to_backup = array();
        $backup_name = "wordpress-{$site_name}";

        switch ($type) {
            case 'full':
                $backup_name .= "-full-{$timestamp}";
                $files_to_backup = $this->collect_full_backup($temp_dir);
                break;

            case 'component':
                $backup_name .= "-{$component}-{$timestamp}";
                $files_to_backup = $this->collect_component_backup($temp_dir, $component);
                break;

            case 'custom':
                $elements_slug = implode('-', array_map('sanitize_title', $elements));
                $backup_name .= "-custom-{$elements_slug}-{$timestamp}";
                $files_to_backup = $this->collect_custom_backup($temp_dir, $elements);
                break;

            case 'profile':
                $profiles = get_option('lox_backup_profiles', array());
                if (!isset($profiles[$profile_id])) {
                    $this->cleanup_temp($temp_dir);
                    return new WP_Error('profile_not_found', __('Profile not found', 'lox-backup'));
                }
                $profile = $profiles[$profile_id];
                $backup_name .= "-" . sanitize_title($profile['name']) . "-{$timestamp}";
                $files_to_backup = $this->collect_custom_backup($temp_dir, $profile['elements']);
                break;
        }

        if (is_wp_error($files_to_backup)) {
            $this->cleanup_temp($temp_dir);
            return $files_to_backup;
        }

        if (empty($files_to_backup)) {
            $this->cleanup_temp($temp_dir);
            return new WP_Error('backup_empty', __('No files to backup', 'lox-backup'));
        }

        // Create archive
        $archive_path = $backup_dir . '/' . $backup_name . '.tar.gz';
        $this->create_archive($temp_dir, $archive_path);

        // Cleanup temp directory
        $this->cleanup_temp($temp_dir);

        return array(
            'name'         => $backup_name,
            'archive_path' => $archive_path,
            'type'         => $type,
        );
    }

    /**
     * Collect files for full backup
     */
    private function collect_full_backup($temp_dir) {
        $settings = get_option('lox_backup_settings', array());
        $files = array();

        if (!empty($settings['backup_database'])) {
            $db_file = $this->backup_database($temp_dir);
            if (!is_wp_error($db_file)) $files[] = $db_file;
        }

        if (!empty($settings['backup_uploads'])) {
            $f = $this->backup_directory(WP_CONTENT_DIR . '/uploads', $temp_dir, 'uploads');
            if (!is_wp_error($f)) $files[] = $f;
        }

        if (!empty($settings['backup_plugins'])) {
            $f = $this->backup_directory(WP_PLUGIN_DIR, $temp_dir, 'plugins');
            if (!is_wp_error($f)) $files[] = $f;
        }

        if (!empty($settings['backup_themes'])) {
            $f = $this->backup_directory(get_theme_root(), $temp_dir, 'themes');
            if (!is_wp_error($f)) $files[] = $f;
        }

        $config = $this->backup_config($temp_dir);
        if (!is_wp_error($config)) $files[] = $config;

        return $files;
    }

    /**
     * Collect files for component backup
     */
    private function collect_component_backup($temp_dir, $component) {
        $backup = new LOX_Backup();
        $components = $backup->get_components();

        if (!isset($components[$component])) {
            return new WP_Error('invalid_component', __('Invalid component', 'lox-backup'));
        }

        $def = $components[$component];
        $files = array();

        if (!empty($def['tables'])) {
            $f = $this->backup_component_tables($temp_dir, $def['tables']);
            if (!is_wp_error($f)) $files[] = $f;
        }

        if (!empty($def['paths'])) {
            foreach ($def['paths'] as $path) {
                $full_path = WP_CONTENT_DIR . '/' . $path;
                if (is_dir($full_path)) {
                    $f = $this->backup_directory($full_path, $temp_dir, $path);
                    if (!is_wp_error($f)) $files[] = $f;
                }
            }
        }

        if (!empty($def['files'])) {
            foreach ($def['files'] as $file) {
                if ($file === 'wp-config.php') {
                    $f = $this->backup_config($temp_dir);
                    if (!is_wp_error($f)) $files[] = $f;
                }
            }
        }

        return $files;
    }

    /**
     * Collect files for custom backup
     */
    private function collect_custom_backup($temp_dir, $elements) {
        $files = array();

        if (in_array('database', $elements)) {
            $f = $this->backup_database($temp_dir);
            if (!is_wp_error($f)) $files[] = $f;
        }

        if (in_array('uploads', $elements)) {
            $f = $this->backup_directory(WP_CONTENT_DIR . '/uploads', $temp_dir, 'uploads');
            if (!is_wp_error($f)) $files[] = $f;
        }

        if (in_array('plugins', $elements)) {
            $f = $this->backup_directory(WP_PLUGIN_DIR, $temp_dir, 'plugins');
            if (!is_wp_error($f)) $files[] = $f;
        }

        if (in_array('themes', $elements)) {
            $f = $this->backup_directory(get_theme_root(), $temp_dir, 'themes');
            if (!is_wp_error($f)) $files[] = $f;
        }

        if (in_array('config', $elements)) {
            $f = $this->backup_config($temp_dir);
            if (!is_wp_error($f)) $files[] = $f;
        }

        return $files;
    }

    /**
     * Backup database
     */
    private function backup_database($temp_dir) {
        global $wpdb;

        $sql_file = $temp_dir . '/database.sql';
        $handle = fopen($sql_file, 'w');

        if (!$handle) {
            return new WP_Error('db_backup_failed', __('Failed to create database file', 'lox-backup'));
        }

        fwrite($handle, "-- LOX WordPress PULL Backup\n");
        fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
        fwrite($handle, "-- Site: " . home_url() . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\nSET FOREIGN_KEY_CHECKS = 0;\n\n");

        $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);

        foreach ($tables as $table) {
            $table_name = $table[0];
            if ($wpdb->prefix && strpos($table_name, $wpdb->prefix) !== 0) continue;

            $create = $wpdb->get_row("SHOW CREATE TABLE `{$table_name}`", ARRAY_N);
            fwrite($handle, "DROP TABLE IF EXISTS `{$table_name}`;\n");
            fwrite($handle, $create[1] . ";\n\n");

            $rows = $wpdb->get_results("SELECT * FROM `{$table_name}`", ARRAY_A);
            if (!empty($rows)) {
                $columns = '`' . implode('`, `', array_keys($rows[0])) . '`';
                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function($v) use ($wpdb) {
                            return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$table_name}` ({$columns}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
            }
        }

        fwrite($handle, "\nSET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);

        return $sql_file;
    }

    /**
     * Backup specific tables
     */
    private function backup_component_tables($temp_dir, $tables) {
        global $wpdb;

        $sql_file = $temp_dir . '/database.sql';
        $handle = fopen($sql_file, 'w');

        if (!$handle) {
            return new WP_Error('db_backup_failed', __('Failed to create database file', 'lox-backup'));
        }

        fwrite($handle, "-- LOX Component Backup\n");
        fwrite($handle, "-- Tables: " . implode(', ', $tables) . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\nSET FOREIGN_KEY_CHECKS = 0;\n\n");

        foreach ($tables as $suffix) {
            $table_name = $wpdb->prefix . $suffix;
            if (!$wpdb->get_var("SHOW TABLES LIKE '{$table_name}'")) continue;

            $create = $wpdb->get_row("SHOW CREATE TABLE `{$table_name}`", ARRAY_N);
            fwrite($handle, "DROP TABLE IF EXISTS `{$table_name}`;\n");
            fwrite($handle, $create[1] . ";\n\n");

            $rows = $wpdb->get_results("SELECT * FROM `{$table_name}`", ARRAY_A);
            if (!empty($rows)) {
                $columns = '`' . implode('`, `', array_keys($rows[0])) . '`';
                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function($v) use ($wpdb) {
                            return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$table_name}` ({$columns}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
            }
        }

        fwrite($handle, "\nSET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);

        return $sql_file;
    }

    /**
     * Backup directory
     */
    private function backup_directory($source, $temp_dir, $name) {
        if (!is_dir($source)) {
            return new WP_Error('dir_not_found', sprintf(__('Directory not found: %s', 'lox-backup'), $source));
        }

        $archive_path = $temp_dir . '/' . $name . '.tar';

        try {
            $phar = new PharData($archive_path);
            $phar->buildFromDirectory($source);
            unset($phar);
            return $archive_path;
        } catch (Exception $e) {
            return new WP_Error('archive_failed', $e->getMessage());
        }
    }

    /**
     * Backup wp-config.php
     */
    private function backup_config($temp_dir) {
        $config_path = ABSPATH . 'wp-config.php';
        if (!file_exists($config_path)) {
            $config_path = dirname(ABSPATH) . '/wp-config.php';
        }
        if (!file_exists($config_path)) {
            return new WP_Error('config_not_found', __('wp-config.php not found', 'lox-backup'));
        }

        $dest = $temp_dir . '/wp-config.php';
        copy($config_path, $dest);
        return $dest;
    }

    /**
     * Create tar.gz archive
     */
    private function create_archive($source_dir, $archive_path) {
        $tar_path = str_replace('.tar.gz', '.tar', $archive_path);

        foreach (array($archive_path, $tar_path) as $path) {
            if (file_exists($path)) {
                try { Phar::unlinkArchive($path); } catch (Exception $e) { @unlink($path); }
            }
        }

        $phar = new PharData($tar_path);
        $phar->buildFromDirectory($source_dir);
        $phar->compress(Phar::GZ);
        unset($phar);

        if (file_exists($tar_path)) {
            try { Phar::unlinkArchive($tar_path); } catch (Exception $e) { @unlink($tar_path); }
        }
    }

    /**
     * Cleanup temp directory
     */
    private function cleanup_temp($dir) {
        if (!is_dir($dir)) return;

        $items = scandir($dir);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;
            $path = $dir . '/' . $item;
            is_dir($path) ? $this->cleanup_temp($path) : unlink($path);
        }
        rmdir($dir);
    }

    /**
     * Create temporary download token
     *
     * @param string $archive_path Path to archive
     * @return string Download token
     */
    private function create_download_token($archive_path) {
        $token = wp_generate_password(32, false);
        $expiry = time() + 3600; // 1 hour

        $downloads = get_transient('lox_pull_downloads') ?: array();
        $downloads[$token] = array(
            'path'    => $archive_path,
            'expiry'  => $expiry,
            'created' => time(),
        );
        set_transient('lox_pull_downloads', $downloads, 3600);

        return $token;
    }

    /**
     * Handle download request
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error Response
     */
    public function handle_download($request) {
        $token = $request->get_param('token');
        $downloads = get_transient('lox_pull_downloads') ?: array();

        if (!isset($downloads[$token])) {
            return new WP_Error(
                'invalid_token',
                __('Invalid or expired download token', 'lox-backup'),
                array('status' => 404)
            );
        }

        $download = $downloads[$token];

        if (time() > $download['expiry']) {
            unset($downloads[$token]);
            set_transient('lox_pull_downloads', $downloads, 3600);
            return new WP_Error(
                'expired',
                __('Download link has expired', 'lox-backup'),
                array('status' => 410)
            );
        }

        if (!file_exists($download['path'])) {
            return new WP_Error(
                'file_not_found',
                __('Backup file not found', 'lox-backup'),
                array('status' => 404)
            );
        }

        // Remove token after use (one-time download)
        unset($downloads[$token]);
        set_transient('lox_pull_downloads', $downloads, 3600);

        // Stream the file
        return $this->stream_backup($download['path'], true);
    }

    /**
     * Stream backup file to response
     *
     * @param string $archive_path Path to archive
     * @param bool $delete_after Delete file after streaming
     */
    private function stream_backup($archive_path, $delete_after = false) {
        if (!file_exists($archive_path)) {
            return new WP_Error('file_not_found', __('Backup file not found', 'lox-backup'));
        }

        $filename = basename($archive_path);
        $filesize = filesize($archive_path);

        // Set headers for file download
        header('Content-Type: application/gzip');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . $filesize);
        header('Content-Transfer-Encoding: binary');
        header('X-LOX-Checksum: ' . hash_file('sha256', $archive_path));

        // Disable output buffering
        while (ob_get_level()) {
            ob_end_clean();
        }

        // Stream file
        readfile($archive_path);

        // Cleanup if requested
        if ($delete_after) {
            @unlink($archive_path);
        }

        exit;
    }

    /**
     * Handle status request
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response Response
     */
    public function handle_status_request($request) {
        $settings = get_option('lox_backup_settings', array());

        // Get disk usage estimates
        $upload_dir = wp_upload_dir();
        $uploads_size = $this->get_directory_size(WP_CONTENT_DIR . '/uploads');
        $plugins_size = $this->get_directory_size(WP_PLUGIN_DIR);
        $themes_size = $this->get_directory_size(get_theme_root());

        global $wpdb;
        $db_size = $wpdb->get_var("SELECT SUM(data_length + index_length) FROM information_schema.tables WHERE table_schema = DATABASE()");

        return rest_ensure_response(array(
            'site_url'      => home_url(),
            'site_name'     => get_bloginfo('name'),
            'wp_version'    => get_bloginfo('version'),
            'plugin_version' => LOX_BACKUP_VERSION,
            'php_version'   => PHP_VERSION,
            'pull_enabled'  => true,
            'components'    => array(
                'database' => array('enabled' => !empty($settings['backup_database']), 'size_bytes' => (int)$db_size),
                'uploads'  => array('enabled' => !empty($settings['backup_uploads']), 'size_bytes' => $uploads_size),
                'plugins'  => array('enabled' => !empty($settings['backup_plugins']), 'size_bytes' => $plugins_size),
                'themes'   => array('enabled' => !empty($settings['backup_themes']), 'size_bytes' => $themes_size),
            ),
            'estimated_full_size' => $db_size + $uploads_size + $plugins_size + $themes_size,
            'last_backup'   => $settings['last_backup'] ?? null,
            'last_status'   => $settings['last_backup_status'] ?? null,
        ));
    }

    /**
     * Handle token refresh
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response Response
     */
    public function handle_token_refresh($request) {
        $token = wp_generate_password(64, false);
        $expiry = time() + (365 * DAY_IN_SECONDS); // 1 year

        update_option(self::TOKEN_OPTION, $token);
        update_option(self::TOKEN_EXPIRY_OPTION, $expiry);

        return rest_ensure_response(array(
            'success'    => true,
            'token'      => $token,
            'expires_at' => date('c', $expiry),
        ));
    }

    /**
     * Get directory size recursively
     */
    private function get_directory_size($dir) {
        $size = 0;
        if (!is_dir($dir)) return 0;

        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS)) as $file) {
            $size += $file->getSize();
        }
        return $size;
    }

    /**
     * Generate a new PULL token (called during setup)
     *
     * @return array Token info
     */
    public static function generate_pull_token() {
        $token = wp_generate_password(64, false);
        $expiry = time() + (365 * DAY_IN_SECONDS);

        update_option(self::TOKEN_OPTION, $token);
        update_option(self::TOKEN_EXPIRY_OPTION, $expiry);

        return array(
            'token'      => $token,
            'expires_at' => date('c', $expiry),
        );
    }

    /**
     * Get current PULL token
     *
     * @return array|null Token info or null if not set
     */
    public static function get_pull_token() {
        $token = get_option(self::TOKEN_OPTION);
        $expiry = get_option(self::TOKEN_EXPIRY_OPTION);

        if (empty($token)) {
            return null;
        }

        return array(
            'token'      => $token,
            'expires_at' => $expiry ? date('c', $expiry) : null,
            'expired'    => $expiry && time() > $expiry,
        );
    }

    /**
     * Revoke PULL token
     */
    public static function revoke_pull_token() {
        delete_option(self::TOKEN_OPTION);
        delete_option(self::TOKEN_EXPIRY_OPTION);
    }
}
