<?php
/**
 * Plugin Name: LOX Backup
 * Plugin URI: https://backlox.com/wordpress
 * Description: Automated backups to LOX Cold Backup Service - Enterprise-grade ransomware protection for your WordPress site.
 * Version: 1.3.0
 * Author: LOX Team
 * Author URI: https://backlox.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: lox-backup
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('LOX_BACKUP_VERSION', '1.3.0');
define('LOX_BACKUP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('LOX_BACKUP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LOX_BACKUP_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main plugin class
 */
class LOX_Backup_Plugin {

    /**
     * Single instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Load required files
     */
    private function load_dependencies() {
        require_once LOX_BACKUP_PLUGIN_DIR . 'includes/class-lox-api.php';
        require_once LOX_BACKUP_PLUGIN_DIR . 'includes/class-lox-backup.php';
        require_once LOX_BACKUP_PLUGIN_DIR . 'includes/class-lox-admin.php';
        require_once LOX_BACKUP_PLUGIN_DIR . 'includes/class-lox-rest-api.php';
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Register custom cron schedules
        add_filter('cron_schedules', array($this, 'add_cron_schedules'));

        // Initialize admin
        if (is_admin()) {
            new LOX_Backup_Admin();
        }

        // Initialize REST API for PULL backups
        new LOX_REST_API();

        // Schedule backup cron (legacy full backup)
        add_action('lox_backup_scheduled', array($this, 'run_scheduled_backup'));

        // Component backup crons (with frequency from schedule)
        add_action('lox_backup_component_content', function() {
            $this->run_component_backup('content', $this->get_component_frequency('content'));
        });
        add_action('lox_backup_component_transactional', function() {
            $this->run_component_backup('transactional', $this->get_component_frequency('transactional'));
        });
        add_action('lox_backup_component_files', function() {
            $this->run_component_backup('files', $this->get_component_frequency('files'));
        });
        add_action('lox_backup_component_config', function() {
            $this->run_component_backup('config', $this->get_component_frequency('config'));
        });

        // Scheduled component backups from admin UI (lox_backup_scheduled_{component})
        add_action('lox_backup_scheduled_database', function() {
            $this->run_scheduled_component_backup('database');
        });
        add_action('lox_backup_scheduled_uploads', function() {
            $this->run_scheduled_component_backup('uploads');
        });
        add_action('lox_backup_scheduled_plugins', function() {
            $this->run_scheduled_component_backup('plugins');
        });
        add_action('lox_backup_scheduled_themes', function() {
            $this->run_scheduled_component_backup('themes');
        });
        add_action('lox_backup_scheduled_full', function() {
            $this->run_scheduled_full_backup();
        });

        // Add action links
        add_filter('plugin_action_links_' . LOX_BACKUP_PLUGIN_BASENAME, array($this, 'add_action_links'));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        $defaults = array(
            'api_key' => '',
            'base_url' => 'https://backlox.com/api',
            'backup_database' => true,
            'backup_uploads' => true,
            'backup_plugins' => true,
            'backup_themes' => true,
            'schedule' => 'daily',
            'retention_days' => 30,
            'tags' => 'wordpress,automated',
            'last_backup' => null,
            'last_backup_status' => null,
            // Component schedules
            'component_schedules' => array(
                'content' => 'weekly',
                'transactional' => 'daily',
                'files' => 'weekly',
                'config' => 'daily',
            ),
        );

        if (!get_option('lox_backup_settings')) {
            add_option('lox_backup_settings', $defaults);
        }

        // Schedule backup
        $this->schedule_backup();
        $this->schedule_component_backups();

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Remove scheduled events
        wp_clear_scheduled_hook('lox_backup_scheduled');
        wp_clear_scheduled_hook('lox_backup_component_content');
        wp_clear_scheduled_hook('lox_backup_component_transactional');
        wp_clear_scheduled_hook('lox_backup_component_files');
        wp_clear_scheduled_hook('lox_backup_component_config');
    }

    /**
     * Schedule backup cron job (legacy full backup)
     */
    public function schedule_backup() {
        $settings = get_option('lox_backup_settings');
        $schedule = isset($settings['schedule']) ? $settings['schedule'] : 'daily';

        // Clear existing schedule
        wp_clear_scheduled_hook('lox_backup_scheduled');

        if ($schedule !== 'disabled') {
            wp_schedule_event(time(), $schedule, 'lox_backup_scheduled');
        }
    }

    /**
     * Schedule component backups
     */
    public function schedule_component_backups() {
        $settings = get_option('lox_backup_settings');
        $schedules = isset($settings['component_schedules']) ? $settings['component_schedules'] : array();

        $components = array('content', 'transactional', 'files', 'config');
        $default_times = array(
            'content' => strtotime('next sunday 03:00'),
            'transactional' => strtotime('today 04:00'),
            'files' => strtotime('next saturday 02:00'),
            'config' => strtotime('today 05:00'),
        );

        foreach ($components as $component) {
            $hook = 'lox_backup_component_' . $component;
            $schedule = isset($schedules[$component]) ? $schedules[$component] : 'daily';

            // Clear existing schedule
            wp_clear_scheduled_hook($hook);

            if ($schedule !== 'disabled') {
                $start_time = isset($default_times[$component]) ? $default_times[$component] : time();
                wp_schedule_event($start_time, $schedule, $hook);
            }
        }
    }

    /**
     * Run scheduled backup (legacy full backup)
     */
    public function run_scheduled_backup() {
        $settings = get_option('lox_backup_settings');
        $frequency = isset($settings['schedule']) ? $settings['schedule'] : 'daily';

        $backup = new LOX_Backup();
        $backup->run_full_backup(array('frequency' => $frequency));
    }

    /**
     * Run scheduled component backup (legacy component schedule)
     *
     * @param string $component Component name
     * @param string $frequency Backup frequency (daily, weekly, monthly)
     */
    public function run_component_backup($component, $frequency = 'daily') {
        $backup = new LOX_Backup();
        $backup->run_component_backup($component, array('frequency' => $frequency));
    }

    /**
     * Get component frequency from legacy settings
     *
     * @param string $component Component name
     * @return string Frequency (daily, weekly, monthly)
     */
    private function get_component_frequency($component) {
        $settings = get_option('lox_backup_settings');
        $schedules = isset($settings['component_schedules']) ? $settings['component_schedules'] : array();
        return isset($schedules[$component]) ? $schedules[$component] : 'daily';
    }

    /**
     * Run scheduled component backup from admin UI schedules
     *
     * @param string $component Component key (database, uploads, plugins, themes)
     */
    public function run_scheduled_component_backup($component) {
        $schedules = get_option('lox_backup_schedules', array());
        $schedule_key = "schedule_{$component}";
        $frequency = isset($schedules[$schedule_key]['frequency']) ? $schedules[$schedule_key]['frequency'] : 'daily';

        // Map admin schedule components to backup components
        $component_map = array(
            'database' => 'content',        // Database maps to content component
            'uploads' => 'files',           // Uploads maps to files component
            'plugins' => 'config',          // Plugins are part of config
            'themes' => 'config',           // Themes are part of config
        );

        $backup = new LOX_Backup();

        // For database/uploads, use component backup
        if (in_array($component, array('database', 'uploads'))) {
            $mapped_component = $component_map[$component] ?? $component;
            $backup->run_component_backup($mapped_component, array('frequency' => $frequency));
        } else {
            // For plugins/themes, run custom backup with just that element
            $backup->run_custom_backup(array($component), '', array('frequency' => $frequency));
        }
    }

    /**
     * Run scheduled full backup from admin UI
     */
    public function run_scheduled_full_backup() {
        $schedules = get_option('lox_backup_schedules', array());
        $frequency = isset($schedules['schedule_full']['frequency']) ? $schedules['schedule_full']['frequency'] : 'weekly';

        $backup = new LOX_Backup();
        $backup->run_full_backup(array('frequency' => $frequency));
    }

    /**
     * Add plugin action links
     */
    public function add_action_links($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=lox-backup') . '">' . __('Settings', 'lox-backup') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Add custom cron schedules
     *
     * @param array $schedules Existing cron schedules
     * @return array Modified cron schedules
     */
    public function add_cron_schedules($schedules) {
        $schedules['weekly'] = array(
            'interval' => 7 * DAY_IN_SECONDS,
            'display' => __('Once Weekly', 'lox-backup'),
        );
        $schedules['monthly'] = array(
            'interval' => 30 * DAY_IN_SECONDS,
            'display' => __('Once Monthly', 'lox-backup'),
        );
        return $schedules;
    }
}

// Initialize plugin
function lox_backup_init() {
    return LOX_Backup_Plugin::get_instance();
}

// Run
lox_backup_init();
